/*****************************************************************************
 
            Copyright (c)2005 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/
 
/* $Id: motif_client.c 16194 2010-03-04 16:31:34Z foucher $ */
 
#include <GTXClient.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

#include <Xm/Xm.h>
#include <X11/Intrinsic.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/MessageB.h>
#include <Xm/PanedW.h>
#include <Xm/PushB.h>
#include <Xm/ScrolledW.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/DialogS.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>

/* Hummingbird Exceed XDK Xm DLL */
#if defined(WIN32)
#include <X11/XlibXtra.h>
#endif

static String fallback_resources[] =
{
  "*background: #B2B2B2",
  "*foreground: black",
  "*XmTextField.background: #D3B5B5",
  "*XmText.background: #D3B5B5",
  "*XmList.background: #D3B5B5",
  NULL,
};

#define DEFAULT_GTX_SERVERNAME "localhost"

#define MAX_HOSTNAME_LEN 1024
#define MAX_PATH_LEN     1024
#define MAX_ITEM_LEN       50

#define MODE_INIT  0
#define MODE_STUDY 1
#define MODE_DIR   2
#define MODE_FILE  3
#define MODE_VAR   4

#define NONE_STRING "None"

/* Global Variables to store Isatis File System Names*/
char current_study[MAX_ITEM_LEN];
char current_dir[MAX_ITEM_LEN];
char current_file[MAX_ITEM_LEN];
char current_var[MAX_ITEM_LEN];
char current_sel[MAX_ITEM_LEN];
int  current_indice;
char current_alpha_indice[MAX_ITEM_LEN];

/* Global Variables for Connection */
int run_server = 1;
char host[MAX_HOSTNAME_LEN];
unsigned short port;
char data_path[MAX_PATH_LEN];

/* Global Variables for Dump */
char dump_output[MAX_PATH_LEN];
char dump_format[10];
int  dump_nbitem = 1;

/* Global Widgets */
Widget init_formW = NULL;
Widget data_pathW = NULL;
Widget hostFormW  = NULL;
Widget hostW      = NULL;
Widget portW      = NULL;

Widget paneW      = NULL;
Widget listW      = NULL;
Widget ind_listW  = NULL;
Widget deleteW    = NULL;
Widget infoW      = NULL;
Widget labelW     = NULL;
Widget infoZoneW  = NULL;
Widget dumpW      = NULL;
Widget selW       = NULL;
Widget statW      = NULL;
Widget statusW    = NULL;

/* Dump shell widgets*/
Widget dump_dialW   = NULL;
Widget dump_outputW = NULL;
Widget dump_formatW = NULL;
Widget dump_nbitemW = NULL;

/* Current Option */
int mode;

char *GTXfile_types[] =
{
  "Points",
  "Lines",
  "Grid"
};

char *GTXvar_types[] =
{
  "Float",
  "Character",
  "X gravity center",
  "Y gravity center",
  "Z gravity center",
  "Macro"
};

/*****************************************************************************
**
** FUNCTION:    st_usage
**
** DESCRIPTION: gives this program usage
**
** RETURNS:     Nothing
**
** IN_ARGS:     char *exe_name : name of the executable
** IN_ARGS:     int help : 1 to obtain a complete help
**
*****************************************************************************/
static void st_usage(char *exe_name,
                     int help)
{
  (void)printf("Usage : %s [-help] [-host hostname] [-port port_number]\n",
               exe_name);
  if (help)
  {
    (void)printf(" -help            : Print this message.\n");
    (void)printf(" -debug           : Print client/server messages.\n");
    (void)printf(" -host hostname   : Server's hostname.\n");
    (void)printf(" -port port_number: Port used between client and server\n");
    (void)printf(" -path data_path  : Initialize connection with given data_path.\n");
  }
}

/*****************************************************************************
**
** FUNCTION:    st_set_cur_label
**
** DESCRIPTION: Compose a string like "study: dir/file/sel" and create a label
**
** RETURNS:     Nothing
**
*****************************************************************************/
static void st_set_cur_label()
{
#define end_label &label_str[strlen(label_str)]
  char label_str[300];

  if (!strcmp(current_study,NONE_STRING))
  {
    (void)strcpy(label_str, "Study list:");
    goto label_end;
  }
  else
    (void)strcpy(label_str, current_study);

  if (!strcmp(current_dir,NONE_STRING))
  {
    (void)sprintf(end_label, "\nDirectory list:");
    goto label_end;
  }
  else
    (void)sprintf(end_label, ": %s", current_dir);

  if (!strcmp(current_file,NONE_STRING))
  {
    (void)sprintf(end_label, "\nFile list:");
    goto label_end;
  }
  else
    (void)sprintf(end_label, "/%s", current_file);

  if (strcmp(current_sel,NONE_STRING))
    (void)sprintf(end_label, "/%s", current_sel);

  (void)sprintf(end_label, "\nVariable list:");

label_end:
  XtVaSetValues(labelW,
                XtVaTypedArg, "labelString",
                XmRString, label_str, strlen(label_str)+1,
                NULL);
#undef end_label
}

/*****************************************************************************
**
** FUNCTION:    st_status_mess
**
** DESCRIPTION: print a message in the status zone
**
** RETURNS:     Nothing
**
** IN_ARGS:     char *format : formatted string
**
*****************************************************************************/
static void st_status_mess(const char *format, ...)
{
  va_list ap;
  char buffer[1024];
 
  va_start(ap, format);
  (void)vsprintf(buffer, format, ap);
  va_end(ap);

  XtVaSetValues(statusW,"value",buffer,NULL);
  XmUpdateDisplay(statusW);
}

/*****************************************************************************
**
** FUNCTION:    st_gtx_error
**
** DESCRIPTION: Callback for GTXClient errors
**
** RETURNS:     Nothing
**
** IN_ARGS:     char *string: error message
**
*****************************************************************************/
static void st_gtx_error(const char *string)
{
  st_status_mess("%s", string);
}

/*****************************************************************************
**
** FUNCTION:    st_get_selected_item
**
** DESCRIPTION: get the selected item in the list
**
** RETURNS:     char *selected_item
**
** REMARKS:     selected_item must be freed using XtFree
**
*****************************************************************************/
static char *st_get_selected_item()
{
  XmString *sel_lst;
  int nb_sel_items;
  char *sel_str;

  sel_str = NULL;
  XtVaGetValues(listW,
                "selectedItems", &sel_lst,
                "selectedItemCount", &nb_sel_items,
                NULL);

  if (nb_sel_items == 1)
    XmStringGetLtoR(sel_lst[0],XmFONTLIST_DEFAULT_TAG,&sel_str);
  else
    st_status_mess("No item selected");

  return(sel_str);
}

/*****************************************************************************
**
** FUNCTION:    st_set_macro_indice
**
** DESCRIPTION: get the selected indice in the list
**
** RETURNS:     int selected_indice (-1 if error)
**
*****************************************************************************/
static int st_set_macro_indice()
{
  XmString *sel_lst;
  int nb_sel_items,indice;
  char *sel_str,*ptr;
  int nb_indices;
  int *indices = NULL;
  char **alpha_indices = NULL;
  XtPointer alpha_ptr;
  int alpha_indices_avail;
  indice = -1;
 
  sel_str = NULL;
  XtVaGetValues(ind_listW,
                "selectedItems", &sel_lst,
                "selectedItemCount", &nb_sel_items, 
                XmNuserData, &alpha_ptr,
                NULL);

  alpha_indices_avail = (int)(size_t)alpha_ptr;
  if (nb_sel_items == 1)
  {
    if (GTXClientGetMacroIndicesWithAlpha(&nb_indices, &indices, &alpha_indices)) return 0;

    XmStringGetLtoR(sel_lst[0],XmFONTLIST_DEFAULT_TAG,&sel_str);


    if (alpha_indices_avail)
    {
      GTXClientSetAlphaIndice(sel_str);
      strcpy(current_alpha_indice, sel_str);
      current_indice = -1;
      strcpy(current_alpha_indice,"");

    }
    else
    {
      if ((indice = strtol(sel_str, &ptr, 10)) == 0 && ptr == sel_str)
        return 0;
      GTXClientSetIndice(indice);
      strcpy(current_alpha_indice,"");
      current_indice = indice ;

    }
  }
  else
    st_status_mess("No indice selected");

  XtFree(sel_str);
  return 1;
}

/*****************************************************************************
**
** FUNCTION:    st_get_study_list
**
** DESCRIPTION: Get the list of studies
**
** RETURNS:     int error code: 1 if cannot get study list, 0 elsewhere
**
*****************************************************************************/
static int st_get_study_list()
{
  int i,error,nb_study;
  XmString study_str;
  char **studies;

  error = 1;
  studies = NULL;

  (void)strcpy(current_study,NONE_STRING);

  /*
  ** Get study list
  */
  if (GTXClientGetStudyList(&nb_study, &studies)) goto label_end;
  XmListDeleteAllItems(listW);

  for (i=0; i<nb_study; i++)
  {
    study_str = XmStringCreateLtoR(studies[i],XmSTRING_DEFAULT_CHARSET);
    XmListAddItem(listW,study_str,0);
    XmStringFree(study_str);
  } /* end of loop on studies */

  XmListSelectPos(listW,1,0);

  XtUnmanageChild(init_formW);
  XtManageChild(paneW);

  mode = MODE_STUDY;
  st_set_cur_label();

  error = 0;
label_end:
  studies = GTXClientFreeStringArray(nb_study, studies);
  return(error);
}

/*****************************************************************************
**
** FUNCTION:    st_get_dir_list
**
** DESCRIPTION: Get the list of directories
**
** RETURNS:     int error code: 1 if cannot get dir list, 0 elsewhere
**
*****************************************************************************/
static int st_get_dir_list(char *study)
{
  int i,error,nb_dir;
  XmString dir_str;
  char **dirs;

  error = 1;
  dirs = NULL;

  if (mode == MODE_STUDY)
  {
    if (GTXClientSetStudy(study)) goto label_end;
    (void)strcpy(current_study,study);
  }
  else
    (void)strcpy(current_dir, NONE_STRING);

  /*
  ** Get directory list
  */
  if (GTXClientGetDirectoryList(&nb_dir, &dirs)) goto label_end;
  XmListDeleteAllItems(listW);
  dir_str = XmStringCreateLtoR("..",XmSTRING_DEFAULT_CHARSET);
  XmListAddItem(listW,dir_str,0);
  XmStringFree(dir_str);

  for (i=0; i<nb_dir; i++)
  {
    dir_str = XmStringCreateLtoR(dirs[i],XmSTRING_DEFAULT_CHARSET);
    XmListAddItem(listW,dir_str,0);
    XmStringFree(dir_str);
  } /* end of loop on directories */

  XmListSelectPos(listW,1,0);
  XtSetSensitive(infoW,False);

  mode = MODE_DIR;
  st_set_cur_label();

  error = 0;
label_end:
  dirs = GTXClientFreeStringArray(nb_dir, dirs);
  return(error);
}

/*****************************************************************************
**
** FUNCTION:    st_get_file_list
**
** DESCRIPTION: Get the list of files
**
** RETURNS:     int error code: 1 if cannot get file list, 0 elsewhere
**
*****************************************************************************/
static int st_get_file_list(char *dir)
{
  int i,error,nb_file;
  XmString file_str;
  char **files;

  error = 1;
  files = NULL;

  if (mode == MODE_DIR)
  {
    if (GTXClientSetDirectory(dir)) goto label_end;
    (void)strcpy(current_dir,dir);
  }
  else
  {
    (void)strcpy(current_file, NONE_STRING);
    (void)strcpy(current_var,  NONE_STRING);
    (void)strcpy(current_sel,  NONE_STRING);
    strcpy(current_alpha_indice,"");
    current_indice = -1;
  }

  /*
  ** Get file list
  */
  if (GTXClientGetFileList(&nb_file, &files)) goto label_end;
  XmListDeleteAllItems(listW);
  file_str = XmStringCreateLtoR("..",XmSTRING_DEFAULT_CHARSET);
  XmListAddItem(listW,file_str,0);
  XmStringFree(file_str);

  for (i=0; i<nb_file; i++)
  {
    file_str = XmStringCreateLtoR(files[i],XmSTRING_DEFAULT_CHARSET);
    XmListAddItem(listW,file_str,0);
    XmStringFree(file_str);
  } /* end of loop on files */

  XmListSelectPos(listW,1,0);

  XtSetSensitive(infoW,True);
  XtSetSensitive(statW,False);
  XtSetSensitive(selW, False);
  XtSetSensitive(dumpW,False);

  mode = MODE_FILE;
  st_set_cur_label();

  error = 0;
label_end:
  files = GTXClientFreeStringArray(nb_file, files);
  return(error);
}

/*****************************************************************************
**
** FUNCTION:    st_get_var_list
**
** DESCRIPTION: Get the list of variables
**
** RETURNS:     int error code: 1 if cannot get var list, 0 elsewhere
**
*****************************************************************************/
static int st_get_var_list(char *file)
{
  int i,error,nb_var;
  XmString var_str;
  char **vars;

  error = 1;
  vars = NULL;

  if (mode == MODE_FILE)
  {
    if (GTXClientSetFile(file)) goto label_end;
    (void)strcpy(current_file,file);
  }
  else
  {
    (void)strcpy(current_var,  NONE_STRING);
    (void)strcpy(current_sel,  NONE_STRING);
    strcpy(current_alpha_indice,"");
    strcpy(current_alpha_indice,"");
    current_indice = -1;
  }

  /*
  ** Get variable list
  */
  if (GTXClientGetVariableList(&nb_var, &vars)) goto label_end;
  XmListDeleteAllItems(listW);
  var_str = XmStringCreateLtoR("..",XmSTRING_DEFAULT_CHARSET);
  XmListAddItem(listW,var_str,0);
  XmStringFree(var_str);

  for (i=0; i<nb_var; i++)
  {
    var_str = XmStringCreateLtoR(vars[i],XmSTRING_DEFAULT_CHARSET);
    XmListAddItem(listW,var_str,0);
    XmStringFree(var_str);
  } /* end of loop on studies */

  XmListSelectPos(listW,1,0);
  XtSetSensitive(statW,True);
  XtSetSensitive(selW, True);
  XtSetSensitive(dumpW,True);

  mode = MODE_VAR;
  st_set_cur_label();

  error = 0;
label_end:
  vars = GTXClientFreeStringArray(nb_var, vars);
  return(error);
}

/*****************************************************************************
**
** FUNCTION:    st_callb_dump
**
** DESCRIPTION: dump a variable
**
** IN_ARGS:     Widget w:              Unused
** IN_ARGS:     XtPointer client_data: 1 to dump, 0 to cancel
** IN_ARGS:     XtPointer call_data:   Unused
**
*****************************************************************************/
static void st_callb_dump(Widget w,
                          XtPointer client_data,
                          XtPointer call_data)
{
  char *temp_str,*ptr,format[12];
  GTXVariableInfo var_info;
  int ok,compress;
  gtx_long i,j,n_kept;

  double dbl_test_val;
  double *dbl_data = NULL;
  char *char_test_val = NULL;
  char **char_data = NULL;

  FILE *outfd = NULL;

  ok = (int)(gtx_long)client_data;
  compress = 1;

  if (!ok) goto label_end;

  st_status_mess("Retrieving parameters...");
  /* Retrieve output file */
  temp_str = XmTextGetString(dump_outputW);
  (void)strncpy(dump_output,temp_str,MAX_PATH_LEN);
  dump_output[MAX_PATH_LEN-1] = '\0';
  XtFree(temp_str); temp_str = NULL;

  /* Retrieve format */
  temp_str = XmTextGetString(dump_formatW);
  (void)strncpy(dump_format,temp_str,10);
  dump_format[9] = '\0';
  XtFree(temp_str); temp_str = NULL;

  /* Retrieve number of items */
  temp_str = XmTextGetString(dump_nbitemW);
  if ((dump_nbitem = strtol(temp_str,&ptr,10))==0 && ptr==temp_str)
  {
    st_status_mess("Number of items (%s) is not a valid number", temp_str);
    goto label_end;
  }

  st_status_mess("Getting variable information...");
  if (GTXClientGetVariableInfo(&var_info)) goto label_end;

  st_status_mess("Reading variable...");
  if (var_info.type == GTX_VAR_TYPE_CHAR)
  {
    if (GTXClientReadCharVariable(compress, &char_test_val,&n_kept,&char_data))
      goto label_end;
  }
  else
    if (GTXClientReadDoubleVariable(compress, &dbl_test_val,&n_kept,&dbl_data))
      goto label_end;

  st_status_mess("Dumping variable...");
  if ((outfd = fopen(dump_output,"w")) == NULL)
  {
    st_status_mess("Error: Cannot open file for writing: %s",dump_output);
    goto label_end;
  }

  if (fprintf(outfd,"# Directory: %s\n", current_dir) < 0 ||
      fprintf(outfd,"# File: %s\n", current_file) < 0 ||
      fprintf(outfd,"# Selection: %s\n", current_sel) < 0 ||
      fprintf(outfd,"# Variable: %s\n", current_var) < 0) goto label_file;
  if (var_info.type == GTX_VAR_TYPE_MACRO)
  {
    if(current_indice !=-1)
    {
      if (fprintf(outfd,"# Macro Indice: %d\n", current_indice) < 0)
        goto label_file;
    }else{
      if (fprintf(outfd,"# Macro Indice: %s\n", current_alpha_indice) < 0)
        goto label_file;
    }
  }
  else
    if (fprintf(outfd,"# Macro Indice: N/A\n") < 0) goto label_file;

  if (fprintf(outfd, "# Unit: %s\n",var_info.unit) < 0) goto label_file;
  if (fprintf(outfd, "# Number of Samples: %"GTXLONG_FORMAT"\n",
              n_kept) < 0) goto label_file;

  /* alphanumeric dump */
  if (var_info.type == GTX_VAR_TYPE_CHAR)
  {
    for (i=0, j=1; i<n_kept; i++, j++)
    {
      if (i==n_kept-1 || j==dump_nbitem)
      {
        (void)strcpy(format, "%s\n");
        j=0;
      }
      else
        (void)strcpy(format,"%s ");
      if (fprintf(outfd,format,char_data[i]) < 0)
        goto label_file;
    }
  }
  else                          /* float var dump */
  {
    (void)strcpy(&format[1],dump_format);
    for (i=0, j=1; i<n_kept; i++, j++)
    {
      if (i==n_kept-1 || j==dump_nbitem)
      {
        (void)sprintf(format, "%s\n", dump_format);
        j=0;
      }
      else
        (void)sprintf(format, "%s ", dump_format);
      if (dbl_data[i] == dbl_test_val)
        dbl_data[i] = -999.99;
      if (fprintf(outfd,format,dbl_data[i]) < 0)
        goto label_file;
    }
  }

  st_status_mess("Dump Finished.");
  goto label_end;               /* Normal end */

label_file:
  st_status_mess("Error: An error occurred during file writing");
label_end:
  if (outfd != NULL)
    (void)fclose(outfd);
  char_test_val = GTXClientFreePointer(char_test_val);
  char_data = GTXClientFreeStringArray(n_kept, char_data);
  dbl_data = GTXClientFreePointer(dbl_data);
  XtPopdown(dump_dialW);
  return;
}

/*****************************************************************************
**
** FUNCTION:    st_callb_dump_pop
**
** DESCRIPTION: creates and pops up the dump shell
**
** IN_ARGS:     Widget w: box button used to find main form
**
*****************************************************************************/
static void st_callb_dump_pop(Widget w,
                              XtPointer client_data,
                              XtPointer call_data)
{
  Arg arg[3];
  int indice,n;
  char temp_str[15];
  char *sel_str;
  GTXVariableInfo var_info;
  Widget form,label,box;

  sel_str = NULL;
  if (mode != MODE_VAR) goto label_end;

  sel_str = st_get_selected_item();
  if (sel_str == NULL) goto label_end;

  st_status_mess("");
  if (!strcmp(sel_str, "..")) goto label_end;

  if (strcmp(sel_str, current_var))
  {
    if (GTXClientSetVariable(sel_str)) goto label_end;
    (void)strcpy(current_var, sel_str);
    strcpy(current_alpha_indice,"");
    current_indice = -1;
  }
  st_status_mess("Getting variable information...");
  if (GTXClientGetVariableInfo(&var_info)) goto label_end;

  if (var_info.type == GTX_VAR_TYPE_MACRO)
  {
    if ((indice = st_set_macro_indice()) == -1) goto label_end;
  }

  /****** dump shell creation ******/
  if (dump_dialW == NULL)
  {
    st_status_mess("Creating dump shell...");
    n = 0;
    XtSetArg(arg[n],"minWidth",300); n++;
    XtSetArg(arg[n],"minHeight",170); n++;
    XtSetArg(arg[n],"maxHeight",170); n++;
    dump_dialW = XmCreateDialogShell(XtParent(XtParent(w)),"DumpShell",arg,n);
    form = XtVaCreateManagedWidget("dump_f",xmFormWidgetClass,
                                   dump_dialW,NULL);
    
    label =
      XtVaCreateManagedWidget("label_file",xmLabelWidgetClass, form,
                              "x", 5,
                              "y", 10,
                              "height", 30,
                              XtVaTypedArg, "labelString",
                              XmRString, "Output File:", 13,
                              NULL);

    dump_outputW =
      XtVaCreateManagedWidget("output_str",xmTextFieldWidgetClass, form,
                              "y", 10,
                              "height", 30,
                              "marginHeight", 3,
                              "leftAttachment", XmATTACH_WIDGET,
                              "leftWidget", label,
                              "leftOffset", 5,
                              "rightAttachment", XmATTACH_FORM,
                              "rightOffset", 10,
                              "value",dump_output,
                              NULL);

    label =
      XtVaCreateManagedWidget("label_format",xmLabelWidgetClass, form,
                              "x", 5,
                              "y", 45,
                              "height", 30,
                              XtVaTypedArg, "labelString",
                              XmRString, "Format (printf style):", 23,
                              NULL);

    dump_formatW =
      XtVaCreateManagedWidget("format_str",xmTextFieldWidgetClass, form,
                              "y", 45,
                              "height", 30,
                              "marginHeight", 3,
                              "leftAttachment", XmATTACH_WIDGET,
                              "leftWidget", label,
                              "leftOffset", 5,
                              "rightAttachment", XmATTACH_FORM,
                              "rightOffset", 10,
                              "value",dump_format,
                              NULL);

    label =
      XtVaCreateManagedWidget("label_nbitem",xmLabelWidgetClass, form,
                              "x", 5,
                              "y", 80,
                              "height", 30,
                              XtVaTypedArg, "labelString",
                              XmRString, "Number of items per row:", 28,
                              NULL);

    (void)sprintf(temp_str, "%d", dump_nbitem);
    dump_nbitemW =
      XtVaCreateManagedWidget("nbitem_str",xmTextFieldWidgetClass, form,
                              "y", 80,
                              "height", 30,
                              "marginHeight", 3,
                              "leftAttachment", XmATTACH_WIDGET,
                              "leftWidget", label,
                              "leftOffset", 5,
                              "rightAttachment", XmATTACH_FORM,
                              "rightOffset", 10,
                              "value",temp_str,
                              NULL);

    box = XtVaCreateManagedWidget("dump_cancel",xmMessageBoxWidgetClass,form,
                                  "height", 43,
                                  "leftAttachment", XmATTACH_FORM,
                                  "leftOffset", 1,
                                  "bottomAttachment", XmATTACH_FORM,
                                  "bottomOffset", 1,
                                  "rightAttachment", XmATTACH_FORM,
                                  "rightOffset", 1,
                                  "marginHeight", 0,
                                  XtVaTypedArg, "okLabelString",
                                  XmRString, "Dump", 5,
                                  XtVaTypedArg, "cancelLabelString",
                                  XmRString, "Cancel", 7,
                                  NULL);
    XtUnmanageChild(XmMessageBoxGetChild(box,XmDIALOG_HELP_BUTTON));
    XtAddCallback(box,XmNokCallback,
                  (XtCallbackProc)st_callb_dump,(XtPointer)1);
    XtAddCallback(box,XmNcancelCallback,
                  (XtCallbackProc)st_callb_dump,(XtPointer)0);
  }

  XtPopup(dump_dialW,XtGrabExclusive);

  /* must be done after the shell has been realized */
  if (var_info.type == GTX_VAR_TYPE_CHAR)
    XtSetSensitive(dump_formatW,  False);

  st_status_mess("Dump options...");

label_end:
  XtFree(sel_str);
}

/*****************************************************************************
**
** FUNCTION:    st_run_toggle_callb
**
** DESCRIPTION: value Changed Callback attached to run GTXserver toggle
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_run_toggle_callb(Widget w,
                                XtPointer client_data,
                                XtPointer call_data)
{
  XmToggleButtonCallbackStruct *tog;
  tog = (XmToggleButtonCallbackStruct *)call_data;

  if (tog->set)
  {
    XtSetSensitive(hostFormW, False);
    run_server = 1;
  }
  else
  {
    XtSetSensitive(hostFormW, True);
    run_server = 0;
  }
}

/*****************************************************************************
**
** FUNCTION:    st_callb_set_sel
**
** DESCRIPTION: set/unset selection variable
**
** RETURNS:     Nothing
**
*****************************************************************************/
static void st_callb_set_sel(Widget w,
                             XtPointer client_data,
                             XtPointer call_data)
{
  char *sel_str = NULL;

  st_status_mess("");
  XtVaSetValues(infoZoneW,"value","",NULL);
  if (mode != MODE_VAR)
    goto label_end;

  /* Unset selection */
  if (strcmp(current_sel, NONE_STRING))
  {
    st_status_mess("Unsetting selection...");
    if (GTXClientSetSelection(NULL)) goto label_end;
    (void)strcpy(current_sel, NONE_STRING);
  }
  else
  {
    sel_str = st_get_selected_item();
    if (sel_str == NULL) goto label_end;

    if (!strcmp(sel_str, ".."))
    {
      st_status_mess("Cannot set \"..\" as the selection");
      goto label_end;
    }
    st_status_mess("Setting selection %s...", sel_str);
    if (GTXClientSetSelection(sel_str)) goto label_end;
    (void)strcpy(current_sel, sel_str);
    st_status_mess("Selection %s set", sel_str);
  }

  st_set_cur_label();

label_end:
  XtFree(sel_str);
}

/*****************************************************************************
**
** FUNCTION:    st_def_callb_list
**
** DESCRIPTION: Default Action Callback attached to List
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_def_callb_list(Widget w,
                              XtPointer client_data,
                              XtPointer call_data)
{
  char *sel_str;

  sel_str = st_get_selected_item();
  if (sel_str == NULL) goto label_end;

  st_status_mess("");
  XtVaSetValues(infoZoneW,"value","",NULL);

  if (!strcmp(sel_str, ".."))
  {
    switch (mode)
    {
      case MODE_DIR:
        st_status_mess("Retrieving Study List");
        if (st_get_study_list()) goto label_end;
        break;
      case MODE_FILE:
        st_status_mess("Retrieving Directory List");
        if (st_get_dir_list(NULL)) goto label_end;
        break;
      case MODE_VAR:
        st_status_mess("Retrieving Variable List");
        if (st_get_file_list(NULL)) goto label_end;
        mode = MODE_FILE;
        break;
      default:
        st_status_mess("Unknown Mode");
        goto label_end;
    } /* end of switch */
  } /* end of if back */
  else
  {
    switch (mode)
    {
      case MODE_STUDY:
        st_status_mess("Setting Study %s", sel_str);
        if (st_get_dir_list(sel_str)) goto label_end;
        st_status_mess("Study %s set",sel_str);
        break;

      case MODE_DIR:
        st_status_mess("Setting Directory %s", sel_str);
        if (st_get_file_list(sel_str)) goto label_end;
        st_status_mess("Directory %s set",sel_str);
        break;

      case MODE_FILE:
        st_status_mess("Setting File %s", sel_str);
        if (st_get_var_list(sel_str)) goto label_end;
        st_status_mess("File %s set",sel_str);
        break;

      case MODE_VAR:
        break;

      default:
        st_status_mess("Unknown mode");
        goto label_end;
    } /* end of switch */
  }

label_end:
  XtFree(sel_str);
}

/*****************************************************************************
**
** FUNCTION:    st_single_callb_list
**
** DESCRIPTION: Single Selection Callback attached to List
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_single_callb_list(Widget w,
                                 XtPointer client_data,
                                 XtPointer call_data)
{
  char *sel_str,char_ind[15];
  int i,nb_indices,*indices;
  GTXVariableInfo var_info;
  XmString xmstr_ind;
  char **alpha_indices = NULL;

  indices = NULL;
  xmstr_ind = NULL;
  sel_str = st_get_selected_item();
  if (sel_str == NULL) goto label_end;

  st_status_mess("");
  XmListDeleteAllItems(ind_listW);
  XtVaSetValues(infoZoneW,"value","",NULL);

  if (!strcmp(sel_str, "..")) goto label_end;

  if (mode != MODE_VAR) goto label_end;

  if (GTXClientSetVariable(sel_str)) goto label_end;
  if (GTXClientGetVariableInfo(&var_info)) goto label_end;
  if (var_info.type != GTX_VAR_TYPE_MACRO) goto label_end;
  if (GTXClientGetMacroIndicesWithAlpha(&nb_indices, &indices, &alpha_indices)) goto label_end;


  if (alpha_indices){
    
    for (i=0; i<nb_indices; i++){
      
      xmstr_ind = XmStringCreateLtoR(alpha_indices[i],XmSTRING_DEFAULT_CHARSET);
      XmListAddItem(ind_listW,xmstr_ind,0);
      XmStringFree(xmstr_ind);
    }
  }else{
    for (i=0; i<nb_indices; i++){
      (void)sprintf(char_ind,"%d",indices[i]);
      xmstr_ind = XmStringCreateLtoR(char_ind,XmSTRING_DEFAULT_CHARSET);
      XmListAddItem(ind_listW,xmstr_ind,0);
      XmStringFree(xmstr_ind);
    }
  }
  XtVaSetValues(ind_listW, XmNuserData, (XtPointer)(size_t)(alpha_indices != NULL), NULL);

  XmListSelectPos(ind_listW,1,0);

label_end:
  indices = GTXClientFreePointer(indices);
  alpha_indices = GTXClientFreeStringArray(nb_indices, alpha_indices);
  XtFree(sel_str);
}
/*****************************************************************************
**
** FUNCTION:    st_callb_box_conn
**
** DESCRIPTION: Callback attached to Connect in the dialog box
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_callb_box_conn(Widget w,
                              XtPointer client_data,
                              XtPointer call_data)
{
  char *temp_str,*ptr;

  temp_str = NULL;
  XtVaSetValues(infoZoneW,"value","",NULL);
  st_status_mess("");

  if (mode == MODE_INIT)
  {
    /* Retrieve data path name */
    temp_str = XmTextGetString(data_pathW);
    (void)strncpy(data_path,temp_str,MAX_PATH_LEN);
    data_path[MAX_PATH_LEN-1] = '\0';
    XtFree(temp_str); temp_str = NULL;

    /* Retrieve host name */
    temp_str = XmTextGetString(hostW);
    (void)strncpy(host,temp_str,MAX_HOSTNAME_LEN);
    host[MAX_HOSTNAME_LEN-1] = '\0';
    XtFree(temp_str); temp_str = NULL;

    /* Retrieve port */
    temp_str = XmTextGetString(portW);
    if ((port = (unsigned short)strtol(temp_str,&ptr,10))==0 && ptr==temp_str)
    {
      st_status_mess("Port (%s) is not a valid number", temp_str);
      goto label_end;
    }

    if (GTXClientInitialize(GTX_CLIENT_VERSION)) goto label_end;
    if (run_server)
    {
      st_status_mess("Lauching Local Server");
      port = 0;
      (void)strcpy(host, "localhost");
      if (GTXClientRunGTXserver(&port))
      {
        st_status_mess("Failed to run Local Server");
        goto label_end;
      }
    }

    st_status_mess("Trying to connect to %s:%d",host,port);
    /* Connect to the server */
    if (GTXClientConnect(host, port, data_path)) goto label_end;
    if (GTXClientSetUnitMode(1)) goto label_end;

    /* get study list */
    if (st_get_study_list()) goto label_end;
    XtUnmanageChild(XmMessageBoxGetChild(w,XmDIALOG_OK_BUTTON));
    XtManageChild(XmMessageBoxGetChild(w,XmDIALOG_CANCEL_BUTTON));
    st_status_mess("Connected");
  }
  else
  {
    GTXClientDisconnect();
    mode = MODE_INIT;
    XtManageChild(init_formW);
    XtUnmanageChild(paneW);
    XtSetSensitive(infoW, False);
    XtSetSensitive(statW, False);
    XtSetSensitive(selW,  False);
    XtSetSensitive(dumpW, False);
    XtUnmanageChild(XmMessageBoxGetChild(w,XmDIALOG_CANCEL_BUTTON));
    XtManageChild(XmMessageBoxGetChild(w,XmDIALOG_OK_BUTTON));
    st_status_mess("Disconnected");
  }

label_end:
  XtFree(temp_str);
}

/*****************************************************************************
**
** FUNCTION:    st_callb_box_exit
**
** DESCRIPTION: Callback attached to Cancel of the dialog box
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_callb_box_exit(Widget w,
                              XtPointer client_data,
                              XtPointer call_data)
{
  switch (mode)
  {
    case MODE_INIT:
      exit(0);

    case MODE_STUDY:
    case MODE_DIR:
    case MODE_FILE:
    case MODE_VAR:
      GTXClientDisconnect();
      exit(0);

    default:
      st_status_mess("Unknown mode!!");
  } /* end of switch */

  return;
}

/*****************************************************************************
**
** FUNCTION:    st_callb_delete
**
** DESCRIPTION: Callback attached to the Delete button (Warning: Delete button
** DESCRIPTION: is disabled by default.
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_callb_delete(Widget w,
                            XtPointer client_data,
                            XtPointer call_data)
{
  char *sel_str;

  XtVaSetValues(infoZoneW,"value","",NULL);

  sel_str = st_get_selected_item();
  if (sel_str == NULL) goto label_end;

  if (!strcmp(sel_str, ".."))
  {
    st_status_mess("Cannot delete \"..\"");
    goto label_end;
  }

  switch (mode)
  {
    case MODE_STUDY:
      st_status_mess("Deleting Study \"%s\"", sel_str);
      if (GTXClientSetStudy(sel_str)) goto label_end;
      if (GTXClientDeleteStudy()) goto label_end;
      (void)st_get_study_list();
      break;

    case MODE_DIR:
      st_status_mess("Deleting Directory \"%s\"", sel_str);
      if (GTXClientSetDirectory(sel_str)) goto label_end;
      if (GTXClientDeleteDirectory()) goto label_end;
      (void)st_get_dir_list(NULL);
      break;

    case MODE_FILE:
      st_status_mess("Deleting File \"%s\"", sel_str);
      if (GTXClientSetFile(sel_str)) goto label_end;
      if (GTXClientDeleteFile()) goto label_end;
      (void)st_get_file_list(NULL);
      break;

    case MODE_VAR:
      st_status_mess("Deleting Variable \"%s\"", sel_str);
      if (GTXClientSetVariable(sel_str)) goto label_end;
      if (GTXClientDeleteVariable()) goto label_end;
      (void)st_get_var_list(NULL);
      break;

    default:
      (void)fprintf(stderr,"Unknown mode!!");
      exit(1);
  } /* end of switch */

  st_status_mess("");
label_end:
  XtFree(sel_str);
  return;
}

/*****************************************************************************
**
** FUNCTION:    st_callb_info
**
** DESCRIPTION: Callback attached to Information button
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_callb_info(Widget w,
                          XtPointer client_data,
                          XtPointer call_data)
{
#define end_buffer &buffer[strlen(buffer)]

  GTXFileInfo file_info;
  GTXVariableInfo var_info;
  int i,j,len,nb_indices, *indices;
  GTXFaultInfo *fault_info;
  GTXFaultSystem fault_system;
  char buffer[2000];
  char **alpha_indices;
  char *comment,*sel_str;

  indices = NULL;
  nb_indices = 0;
  alpha_indices = NULL;

  st_status_mess("");
  XtVaSetValues(infoZoneW,"value","",NULL);

  sel_str = st_get_selected_item();
  if (sel_str == NULL) goto label_end;

  if (!strcmp(sel_str, ".."))
  {
    st_status_mess("Cannot get information on \"..\"");
    goto label_end;
  }

  switch (mode)
  {
    case MODE_FILE:
      st_status_mess("Getting file information...");
      if (strcmp(sel_str, current_file))
      {
        if (GTXClientSetFile(sel_str)) goto label_end;
        (void)strcpy(current_file, sel_str);
      }
      if (GTXClientGetFileInfo(&file_info)) goto label_end;

      (void)sprintf(buffer,"File: %s\n", sel_str);
      (void)sprintf(end_buffer,"Type: %s\n",
                    GTXfile_types[file_info.type]);
      (void)sprintf(end_buffer,"Dimension: %dD\n", file_info.dimension);
      (void)sprintf(end_buffer,"Number of samples: %"GTXLONG_FORMAT"\n",
                    file_info.s_number);

      if (file_info.type == GTX_FILE_TYPE_GRAVITY_LINES)
        (void)sprintf(end_buffer,"Number of lines: %d\n",
                      file_info.item_number);
      else if (file_info.type == GTX_FILE_TYPE_GRID)
      {
        if (file_info.dimension == 2)
        {
          (void)sprintf(end_buffer, "Number of nodes: (%d,%d)\n",
                        file_info.NX,file_info.NY);
          (void)sprintf(end_buffer, "Origin         : (%g,%g)\n",
                        file_info.X0,file_info.Y0);
          (void)sprintf(end_buffer, "Mesh dimensions: (%g,%g)\n",
                        file_info.DX,file_info.DY);
          if (file_info.rotation)
            (void)sprintf(end_buffer, "Rotation Angle : %g\n",
                          file_info.angle_z);
          else
            (void)sprintf(end_buffer, "No Rotation\n");
        }
        else if (file_info.dimension == 3)
        {
          (void)sprintf(end_buffer, "Number of nodes: (%d,%d,%d)\n",
                        file_info.NX,file_info.NY,file_info.NZ);
          (void)sprintf(end_buffer, "Origin         : (%g,%g,%g)\n",
                        file_info.X0,file_info.Y0,file_info.Z0);
          (void)sprintf(end_buffer, "Mesh dimensions: (%g,%g,%g)\n",
                        file_info.DX,file_info.DY,file_info.DZ);
          if (file_info.rotation)
          {
            (void)sprintf(end_buffer, "Rotation Angle around Z: %g\n",
                          file_info.angle_z);
            (void)sprintf(end_buffer, "Rotation Angle around Y: %g\n",
                          file_info.angle_y);
            (void)sprintf(end_buffer, "Rotation Angle around X: %g\n",
                          file_info.angle_x);
          }
          else
            (void)sprintf(end_buffer, "No Rotation\n");
        } /* end of 3D grid */
      } /* end of  grid type */

      /* special variables information */
      (void)sprintf(end_buffer, "Sample Number variable   : %s\n",
                    file_info.sn_var_name);
      if (file_info.type == GTX_FILE_TYPE_GRAVITY_LINES)
      {
        (void)sprintf(end_buffer, "Line Number variable     : %s\n",
                      file_info.ln_var_name);
        (void)sprintf(end_buffer, "Relative Number variable : %s\n",
                      file_info.rn_var_name);
      }
      (void)sprintf(end_buffer, "X Gravity Center variable: %s\n",
                    file_info.xg_var_name);
      (void)sprintf(end_buffer, "Y Gravity Center variable: %s\n",
                    file_info.yg_var_name);
      if (file_info.dimension == 3)
        (void)sprintf(end_buffer, "Z Gravity Center variable: %s\n",
                      file_info.zg_var_name);

      /* faults information */
      if (file_info.faulted)
      {
        fault_info = &file_info.fault_info;
        (void)sprintf(end_buffer, "\nFaulted File:\n");
        (void)sprintf(end_buffer, "2D Faults              : %c\n",
                      (fault_info->faults_2d)? 'Y': 'N');
        (void)sprintf(end_buffer, "Auxiliary Variable Used: %c\n",
                      (fault_info->aux_var_used)? 'Y': 'N');
        (void)sprintf(end_buffer, "Number of faults       : %d\n",
                      fault_info->faults_number);
        (void)sprintf(end_buffer, "Number of segments     : %d\n",
                      fault_info->segments_number);
        (void)sprintf(end_buffer, "Minimum Used Priority  : %d\n",
                      fault_info->min_fpriority);
        (void)sprintf(end_buffer, "Maximum Used Priority  : %d\n",
                      fault_info->max_fpriority);
        (void)sprintf(end_buffer, "Authorized Priority    : %d\n",
                      fault_info->auth_priority);
        if (GTXClientReadFaults(0,&fault_system))
          goto label_end;
      } /* fault information */
      break;

    case MODE_VAR:
      st_status_mess("Getting variable information...");
      if (strcmp(sel_str, current_var))
      {
        if (GTXClientSetVariable(sel_str)) goto label_end;
        (void)strcpy(current_var, sel_str);
        strcpy(current_alpha_indice,"");
        current_indice = -1;
      }
      if (GTXClientGetVariableInfo(&var_info)) goto label_end;

      (void)sprintf(buffer, "Variable  : %s\n", sel_str);
      (void)sprintf(end_buffer,"Type      : %s%s\n",
                    GTXvar_types[var_info.type],
                    (var_info.length_var)? " (Length)": "");
      if (var_info.type == GTX_VAR_TYPE_CHAR)
        (void)sprintf(end_buffer, "Alpha Length: %d\n", var_info.alpha_length);
      else
      {
        if (var_info.bit_length == 1)
          (void)sprintf(end_buffer, "Precision : 1 bit (selection)\n");
        else
          (void)sprintf(end_buffer, "Precision : %d bits\n",
                        var_info.bit_length);
        if (!strcmp(var_info.unit, var_info.unit_symbol))
          (void)sprintf(end_buffer, "Unit      : %s\n", var_info.unit_symbol);
        else
          (void)sprintf(end_buffer, "Unit      : %s (Factor %s)\n",
                        var_info.unit_symbol, var_info.unit);
      }

      if (GTXClientGetVariableComment(&comment)) goto label_end;
      if (comment[0] != '\0')
        (void)sprintf(end_buffer,"Comment   : %s\n", comment);
      comment = GTXClientFreePointer(comment);
      if (var_info.type == GTX_VAR_TYPE_MACRO)
      {
        if (GTXClientGetMacroIndicesWithAlpha(&nb_indices, &indices,
                                              &alpha_indices)) goto label_end;
        (void)sprintf(end_buffer, "Number of indices : %d\n", nb_indices);
        if (nb_indices != 0)
        {
          if (alpha_indices)
            (void)sprintf(end_buffer, "Index  -> Alpha Index\n");
          else
            (void)sprintf(end_buffer, "Indices:\n");
          len = (int)strlen(buffer);
          for (i=0, j=0; i<nb_indices; i++)
          {
            if (len > 1000)
            {
              (void)strcat(&buffer[len], "...\n");
              break;
            }
            (void)sprintf(&buffer[len], "%06d", indices[i]);
            len += 6;
            if (alpha_indices)
            {
              (void)sprintf(&buffer[len], " -> %s\n", alpha_indices[i]);
              len += (int)strlen(&buffer[len]);
            }
            else
            {
              if (++j == 5)
              {
                buffer[len++] = '\n';
                buffer[len] = '\0';
                j = 0;
              }
            }
          }
        }
      }
      break;

    default:
      (void)fprintf(stderr,"Unknown mode!!");
      exit(1);
  } /* end of switch */

  XtVaSetValues(infoZoneW,"value",buffer,NULL);
  st_status_mess("");
label_end:
  indices = GTXClientFreePointer(indices);
  alpha_indices = GTXClientFreeStringArray(nb_indices,alpha_indices);
  XtFree(sel_str);
  return;
#undef end_buffer
}

/*****************************************************************************
**
** FUNCTION:    st_callb_stat
**
** DESCRIPTION: Callback attached to Statistics button
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_callb_stat(Widget w,
                          XtPointer client_data,
                          XtPointer call_data)
{
#define end_buffer &buffer[strlen(buffer)]
  GTXFileInfo file_info;
  GTXVariableInfo  var_info;
  char buffer[2000];
  int compress,indice;
  char *sel_str;

  gtx_long i,nech,n_kept,nb_def;
  double dbl_test_val;
  double *dbl_data = NULL;
  double min,max,mean;

  char *char_test_val = NULL;
  char **char_data = NULL;

  sel_str = st_get_selected_item();
  if (sel_str == NULL) goto label_end;

  if (!strcmp(sel_str, ".."))
  {
    st_status_mess("Cannot get statistics on \"..\"");
    goto label_end;
  }

  if (GTXClientGetFileInfo(&file_info)) goto label_end;
  if (strcmp(sel_str, current_var))
  {
    if (GTXClientSetVariable(sel_str)) goto label_end;
    (void)strcpy(current_var, sel_str);
    strcpy(current_alpha_indice,"");
    current_indice = -1;
  }
  if (GTXClientGetVariableInfo(&var_info)) goto label_end;

  if (var_info.type == GTX_VAR_TYPE_MACRO)
  {
    if ((indice = st_set_macro_indice()) == -1) goto label_end;
  }

  compress = 1;                 /* always compress */

  nech=file_info.s_number;
  nb_def = 0;

  (void)sprintf(buffer, "Variable: %s\n",sel_str);

  st_status_mess("Reading variable...");

  /* Alphanumeric variable reading */
  if (var_info.type == GTX_VAR_TYPE_CHAR)
  {
    if (GTXClientReadCharVariable(compress, &char_test_val,&n_kept,&char_data))
      goto label_end;
  }
  else                          /* all other variables */
    if (GTXClientReadDoubleVariable(compress, &dbl_test_val,&n_kept,&dbl_data))
      goto label_end;

  /* number of samples (not masked) */
  if (n_kept != nech)
    (void)sprintf(end_buffer, "Total number of samples: %"GTXLONG_FORMAT
                  "/%"GTXLONG_FORMAT"\n",n_kept,nech);
  else
    (void)sprintf(end_buffer, "Total number of samples: %"GTXLONG_FORMAT"\n",nech);

  if (var_info.type == GTX_VAR_TYPE_CHAR)
  {
    for (i=0; i<n_kept; i++)
      if (strcmp(char_data[i],char_test_val))
        nb_def++;
    (void)sprintf(end_buffer, "Number of defined samples: %"GTXLONG_FORMAT"\n",nb_def);
  }
  else if (var_info.type == GTX_VAR_TYPE_FLOAT &&
           var_info.bit_length == 1)
  {
    for (i=0; i<n_kept; i++)
      if (dbl_data[i] == 1)
        nb_def++;
    (void)sprintf(end_buffer, "Number of selected samples: %"GTXLONG_FORMAT"\n",nb_def);
  }
  else
  {
    min  = 1.e300;
    max  = -1.e300;
    mean = 0.;

    for (i=0; i<n_kept; i++)
      if  (dbl_data[i] != dbl_test_val)
      {
        nb_def++;
        mean += dbl_data[i];
        min = (dbl_data[i] < min)? dbl_data[i]: min;
        max = (dbl_data[i] > max)? dbl_data[i]: max;
      }
    mean /= nb_def;
    (void)sprintf(end_buffer, "Number of defined samples: %"GTXLONG_FORMAT"\n",nb_def);
    (void)sprintf(end_buffer, "Minimum Value: %g\n",min);
    (void)sprintf(end_buffer, "Maximum Value: %g\n",max);
    (void)sprintf(end_buffer, "Mean Value:    %g\n",mean);
  }

  XtVaSetValues(infoZoneW,"value",buffer,NULL);
  st_status_mess("");

label_end:
  char_test_val = GTXClientFreePointer(char_test_val);
  char_data = GTXClientFreeStringArray(n_kept, char_data);
  dbl_data = GTXClientFreePointer(dbl_data);
  XtFree(sel_str);
#undef end_buffer
}

/*****************************************************************************
**
** FUNCTION:    st_create_interf
**
** DESCRIPTION: Create the Motif Interface
**
** RETURNS:     None
**
** IN_ARGS:     Widget topLevel: parent Top level shell
**
*****************************************************************************/
static void st_create_interf(Widget topLevel)
{
  Widget form,rowcol,list_form,info_form,box;
  Widget run1, use1;
  char temp_str[10];
  Arg arg[3];
  int n;

  /* Main form */
  form = XtVaCreateManagedWidget("main_form", xmFormWidgetClass, topLevel,
                                 XmNwidth, 400,
                                 XmNheight, 400,
                                 NULL);

  /* form containing all initialization data */
  init_formW =
    XtVaCreateManagedWidget("init_form", xmFormWidgetClass, form,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNleftOffset, 1,
                            XmNrightAttachment, XmATTACH_FORM,
                            XmNrightOffset, 1,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNtopOffset, 1,
                            XmNbottomAttachment, XmATTACH_FORM,
                            XmNbottomOffset,80,
                            NULL);

  /* row_column to run GTXserver / use existing */
  rowcol =
    XtVaCreateManagedWidget("row_col_run",xmRowColumnWidgetClass, init_formW,
                            XmNx, 10,
                            XmNy, 15,
                            XmNheight, 30,
                            XmNradioBehavior, True,
                            NULL);

  /* Run a local GTXserver */
  run1 =
    XtVaCreateManagedWidget("run_GTXserver",xmToggleButtonWidgetClass, rowcol,
                            XtVaTypedArg, XmNlabelString,
                            XmRString, "Run a Local GTXserver", 22,
                            XmNset, True,
                            NULL);
  XtAddCallback(run1, XmNvalueChangedCallback,
                (XtCallbackProc)st_run_toggle_callb, NULL);

  /* Use an existing local GTXserver */
  use1 =
    XtVaCreateManagedWidget("use_GTXserver",xmToggleButtonWidgetClass, rowcol,
                            XtVaTypedArg, XmNlabelString,
                            XmRString, "Use a Running GTXserver", 24,
                            NULL);

  /* data_path label and string */
  (void)
    XtVaCreateManagedWidget("data_path_label",xmLabelWidgetClass, init_formW,
                            XmNx, 10,
                            XmNy, 80,
                            XmNheight, 30,
                            XmNalignment, XmALIGNMENT_BEGINNING,
                            XtVaTypedArg, XmNlabelString,
                            XmRString, "Data path", 10,
                            NULL);
  
  data_pathW =
    XtVaCreateManagedWidget("data_path_str",xmTextFieldWidgetClass, init_formW,
                            XmNy, 80,
                            XmNheight, 30,
                            XmNmarginHeight, 3,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNleftOffset, 80,
                            XmNrightAttachment, XmATTACH_FORM,
                            XmNrightOffset, 10,
                            XmNvalue,data_path,
                            NULL);
  
  hostFormW =
    XtVaCreateManagedWidget("host_form",xmFormWidgetClass, init_formW,
                            XmNx, 10,
                            XmNy, 115,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNleftOffset, 10,
                            XmNrightAttachment, XmATTACH_FORM,
                            XmNrightOffset, 10,
                            NULL);

  /* host label and string */
  (void)
    XtVaCreateManagedWidget("host_label", xmLabelWidgetClass, hostFormW,
                            XmNx, 0,
                            XmNy, 0,
                            XmNheight, 30,
                            XmNalignment, XmALIGNMENT_BEGINNING,
                            XtVaTypedArg, XmNlabelString,
                            XmRString, "Host", 5,
                            NULL);
  
  hostW =
    XtVaCreateManagedWidget("host_str",xmTextFieldWidgetClass, hostFormW,
                            XmNy, 0,
                            XmNheight, 30,
                            XmNmarginHeight, 3,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNleftOffset, 70,
                            XmNrightAttachment, XmATTACH_FORM,
                            XmNrightOffset, 0,
                            XmNvalue,host,
                            NULL);
  
  /* port label and string */
  (void)
    XtVaCreateManagedWidget("port_label", xmLabelWidgetClass, hostFormW,
                            XmNx, 0,
                            XmNy, 35,
                            XmNheight, 30,
                            XmNalignment, XmALIGNMENT_BEGINNING,
                            XtVaTypedArg, XmNlabelString,
                            XmRString, "Port", 5,
                            NULL);
  
  (void)sprintf(temp_str, "%d", port);
  portW =
    XtVaCreateManagedWidget("port_str",xmTextFieldWidgetClass, hostFormW,
                            XmNy, 35,
                            XmNheight, 30,
                            XmNmarginHeight, 3,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNleftOffset, 70,
                            XmNrightAttachment, XmATTACH_FORM,
                            XmNrightOffset, 0,
                            XmNvalue,temp_str,
                            NULL);
  
  /* pane for list and information */
  paneW = XtVaCreateWidget("pane", xmPanedWindowWidgetClass, form,
                           XmNleftAttachment, XmATTACH_FORM,
                           XmNleftOffset, 1,
                           XmNrightAttachment, XmATTACH_FORM,
                           XmNrightOffset, 1,
                           XmNtopAttachment, XmATTACH_FORM,
                           XmNtopOffset, 1,
                           XmNbottomAttachment, XmATTACH_FORM,
                           XmNbottomOffset,80,
                           NULL);

  /* forms for list and information */
  list_form = XtVaCreateManagedWidget("list_form", xmFormWidgetClass, paneW,
                                      XmNpaneMinimum, 80,
                                      XmNheight, 170,
                                      NULL);
  info_form = XtVaCreateManagedWidget("info_form", xmFormWidgetClass, paneW,
                                      XmNpaneMinimum, 130,
                                      NULL);

  /* label giving current study: dir/file/sel */
  labelW = XtVaCreateManagedWidget("label", xmLabelWidgetClass, list_form,
                                   XmNx, 5,
                                   XmNy, 5,
                                   XmNheight, 35,
                                   XmNalignment, XmALIGNMENT_BEGINNING,
                                   NULL);
  st_set_cur_label();           /* set label */

  /* list containing studies, directories,files or variables */
  n=0;
  XtSetArg(arg[n],"listSizePolicy", XmRESIZE_IF_POSSIBLE); n++;
  listW = XmCreateScrolledList(list_form, "list", arg,n);
  XtVaSetValues(XtParent(listW),
                XmNleftAttachment, XmATTACH_FORM,
                XmNleftOffset, 10,
                XmNrightAttachment, XmATTACH_POSITION,
                XmNrightPosition, 80,
                XmNtopAttachment, XmATTACH_FORM,
                XmNtopOffset, 40,
                XmNbottomAttachment, XmATTACH_FORM,
                XmNbottomOffset,5,
                NULL);
  XtManageChild(listW);

  XtAddCallback(listW, XmNdefaultActionCallback,
                (XtCallbackProc)st_def_callb_list, NULL);
  XtAddCallback(listW, XmNbrowseSelectionCallback,
                (XtCallbackProc)st_single_callb_list, NULL);

  /* list containing macro variable indices */
  ind_listW = XmCreateScrolledList(list_form, "ind_list",NULL,0);
  XtVaSetValues(XtParent(ind_listW),
                XmNleftAttachment, XmATTACH_WIDGET,
                XmNleftWidget, listW,
                XmNleftOffset, 5,
                XmNrightAttachment, XmATTACH_FORM,
                XmNrightOffset, 10,
                XmNtopAttachment, XmATTACH_FORM,
                XmNtopOffset, 40,
                XmNbottomAttachment, XmATTACH_FORM,
                XmNbottomOffset,5,
                NULL);
  XtManageChild(ind_listW);

  /* delete button (for study, directory, file and variable)*/
  /* too dangerous for now, uncomment if you want to check it */
  /*deleteW = XtVaCreateManagedWidget("delete",xmPushButtonWidgetClass,
    info_form,
    XmNx, 10,
    XmNy, 0,
    XtVaTypedArg, XmNlabelString,
    XmRString, "Delete", 7,
    NULL,0);
    XtAddCallback(deleteW, XmNactivateCallback,
    (XtCallbackProc)st_callb_delete, NULL);*/

  /* information button (for file and variable)*/
  infoW = XtVaCreateManagedWidget("info",xmPushButtonWidgetClass,info_form,
                                  XmNx, 10,
                                  XmNy, 10,
                                  XtVaTypedArg, XmNlabelString,
                                  XmRString, "Information", 12,
                                  NULL);
  XtAddCallback(infoW, XmNactivateCallback,
                (XtCallbackProc)st_callb_info, NULL);

  /* statistics button (for variables) */
  statW = XtVaCreateManagedWidget("stat",xmPushButtonWidgetClass,info_form,
                                  XmNx, 10,
                                  XmNy, 40,
                                  XtVaTypedArg, XmNlabelString,
                                  XmRString, "Statistics", 11,
                                  NULL);
  XtAddCallback(statW, XmNactivateCallback,
                (XtCallbackProc)st_callb_stat, NULL);

  /* dump button (for variables) */
  dumpW = XtVaCreateManagedWidget("dump",xmPushButtonWidgetClass,info_form,
                                  XmNx, 10,
                                  XmNy, 70,
                                  XtVaTypedArg, XmNlabelString,
                                  XmRString, "Dump", 5,
                                  NULL);
  XtAddCallback(dumpW, XmNactivateCallback,
                (XtCallbackProc)st_callb_dump_pop, NULL);

  /* selection button (set/unset current selection) */
  selW = XtVaCreateManagedWidget("sel",xmPushButtonWidgetClass,info_form,
                                 XmNx, 10,
                                 XmNy, 100,
                                 XtVaTypedArg, XmNlabelString,
                                 XmRString, "(Un)set selection", 18,
                                 NULL);
  XtAddCallback(selW, XmNactivateCallback,
                (XtCallbackProc)st_callb_set_sel, NULL);

  XtSetSensitive(infoW, False);
  XtSetSensitive(statW, False);
  XtSetSensitive(selW,  False);
  XtSetSensitive(dumpW, False);
  XtSetSensitive(hostFormW, False);

  /* information Zone (scrolledText) */
  n=0;
  XtSetArg(arg[n],"scrollHorizontal", False); n++;
  XtSetArg(arg[n],"wordWrap", True); n++;
  infoZoneW = XmCreateScrolledText(info_form,"info_zone",arg,n);
  XtVaSetValues(infoZoneW,
                XmNeditable, 0,
                XmNeditMode, XmMULTI_LINE_EDIT,
                XmNcursorPositionVisible, 0,
                /*XmNscrollHorizontal, False,
                  XmNwordWrap, True,*/
                XmNtraversalOn, 0,
                NULL);
  XtVaSetValues(XtParent(infoZoneW),
                XmNleftAttachment, XmATTACH_WIDGET,
                XmNleftWidget, selW,
                XmNleftOffset, 10,
                XmNrightAttachment, XmATTACH_FORM,
                XmNrightOffset, 10,
                XmNtopAttachment, XmATTACH_FORM,
                XmNtopOffset, 5,
                XmNbottomAttachment, XmATTACH_FORM,
                XmNbottomOffset, 5,
                NULL);
  XtManageChild(infoZoneW);

  /* status textfield */
  statusW =
    XtVaCreateManagedWidget("status",xmTextFieldWidgetClass,form,
                            XmNheight, 30,
                            XmNeditable, 0,
                            XmNtraversalOn, 0,
                            XmNmarginHeight, 3,
                            XmNcursorPositionVisible, 0,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNleftOffset, 10,
                            XmNrightAttachment, XmATTACH_FORM,
                            XmNrightOffset, 10,
                            XmNbottomAttachment, XmATTACH_FORM,
                            XmNbottomOffset, 50,
                            NULL);

  /* main message box */
  box = XtVaCreateManagedWidget("conn_quit",xmMessageBoxWidgetClass,form,
                                XmNheight, 43,
                                XmNleftAttachment, XmATTACH_FORM,
                                XmNleftOffset, 1,
                                XmNbottomAttachment, XmATTACH_FORM,
                                XmNbottomOffset, 1,
                                XmNrightAttachment, XmATTACH_FORM,
                                XmNrightOffset, 1,
                                XmNmarginHeight, 0,
                                XtVaTypedArg, XmNokLabelString,
                                XmRString, "Connect", 8,
                                XtVaTypedArg, XmNcancelLabelString,
                                XmRString, "Disconnect", 11,
                                XtVaTypedArg, XmNhelpLabelString,
                                XmRString, "Exit", 5,
                                NULL);
  XtUnmanageChild(XmMessageBoxGetChild(box,XmDIALOG_CANCEL_BUTTON));
  XtAddCallback(box, XmNokCallback,(XtCallbackProc)st_callb_box_conn,NULL);
  XtAddCallback(box, XmNcancelCallback,(XtCallbackProc)st_callb_box_conn,NULL);
  XtAddCallback(box, XmNhelpCallback,(XtCallbackProc)st_callb_box_exit,NULL);

  XtRealizeWidget(topLevel);
}

/****************************************************************************
**
** FUNCTION:    main
**
** DESCRIPTION: client's main function
**
** RETURNS:     Nothing
**
** IN_ARGS:     int argv, char *argv : program arguments
**
*****************************************************************************/
int main(int argc,
         char *argv[])
{
  char *tmp_str;
  int i,error;

  Widget topLevel;
  XtAppContext context;

  error = 1;
  port = GTXClientGetDefaultPort();

  (void)strcpy(current_study,NONE_STRING);
  (void)strcpy(current_dir,  NONE_STRING);
  (void)strcpy(current_file, NONE_STRING);
  (void)strcpy(current_sel,  NONE_STRING);
  (void)strcpy(current_var,  NONE_STRING);
  strcpy(current_alpha_indice,"");
  current_indice = -1;

  (void)strcpy(host, DEFAULT_GTX_SERVERNAME);
  (void)strcpy(data_path, "");

  for (i=1; i<argc; i++)
  {
    if (!strcmp(argv[i], "-port"))
    {
      i++;
      if (i<argc)
      {
        port = (unsigned short)strtol(argv[i], &tmp_str, 10);
        if (port == 0 && tmp_str == argv[i])
        {
          st_status_mess("Error: Invalid Port Number.");
          exit(1);
        }
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-host"))
    {
      i++;
      if (i<argc)
      {
        if (strlen(argv[i]) > MAX_HOSTNAME_LEN-1)
        {
          st_status_mess("Error: Host Name must contain less than %d characters.",
                         MAX_HOSTNAME_LEN);
          exit(1);
        }
        (void)strcpy(host, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-path"))
    {
      i++;
      if (i<argc)
      {
        if (strlen(argv[i]) >= MAX_PATH_LEN)
        {
          st_status_mess("Error: Path must contain less than %d characters.",
                         MAX_PATH_LEN);
          exit(1);
        }
        (void)strcpy(data_path, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-debug"))
      GTXClientDebug(1);
    else if (!strcmp(argv[i], "-help"))
    {
      st_usage(argv[0], 1);
      exit(0);
    }
  }

  /* dump options */
  (void)strcpy(dump_output, "test.dat");
  (void)strcpy(dump_format,"%15.2f");

  /* Interface */
  
  topLevel=XtAppInitialize(&context,"Client",NULL,0,&argc,argv,
                           fallback_resources,NULL,0);

  st_create_interf(topLevel);

  GTX_ERROR_FUNCTION = st_gtx_error;
  st_status_mess("Disconnected");
  mode = MODE_INIT;

  XtRealizeWidget(topLevel);
  XtAppMainLoop(context);

  error = 0;

  /*
  ** Disconnect from the server
  */
  GTXClientDisconnect();

  return(error);
}
