/*****************************************************************************
 
            Copyright (c)2005 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/
 
/* $Id: text_client.c 12705 2008-07-02 15:14:34Z martin $ */
 
#include <GTXClient.h>

#include <ctype.h>              /* isspace */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

#ifdef winnt
#define strdup _strdup
#endif

#define DEFAULT_GTX_SERVERNAME "localhost"

#define MAX_HOSTNAME_LEN 1024
#define MAX_PATH_LEN     1024
#define MAX_ITEM_LEN       50

#define MODE_INIT  0
#define MODE_STUDY 1
#define MODE_DIR   2
#define MODE_FILE  3
#define MODE_VAR   4

#define NONE_STRING "None"

/* Global Variables to store Isatis File System Names*/
char current_study[MAX_ITEM_LEN];
char current_dir[MAX_ITEM_LEN];
char current_file[MAX_ITEM_LEN];
char current_var[MAX_ITEM_LEN];
char current_sel[MAX_ITEM_LEN];
int  current_indice;

/* Current List of Studies or Directories or ... */
char **current_list  = NULL;
char nb_current_list = 0;

/* Global Variables for Connection */
char host[MAX_HOSTNAME_LEN];
unsigned short port;
char data_path[MAX_PATH_LEN];

/* Functions used as Actions */
static void st_action_list(int index);
static void st_action_set(int index);
static void st_action_indice(int index);
static void st_action_set_sel(int index);
static void st_action_info(int index);
static void st_action_stat(int index);
static void st_action_connect(int index);
static void st_action_run(int index);
static void st_action_exit(int index);

struct act
{
  int visible;                  /* says if item must appear in the list */
  char *action;                 /* user key */
  int need_index;               /* says if a number is needed */
  char *title;                  /* message attached to the key */
  void (*action_func)(int number); /* function to be called */
}

Actions[] = 
{
  {0, "LIST", 0, "Retrieve current list", st_action_list},
  {0, "SET",  1, "Set item as current",   st_action_set},
  {0, "IND",  1, "Set indice as current", st_action_indice},
  {0, "SEL",  1, "Set item as selection", st_action_set_sel},
  {0, "INFO", 1, "Information on item",   st_action_info},
  {0, "STAT", 0, "Statistics on item",    st_action_stat},
  {0, "RUN",  0, "Run & Connect",         st_action_run},
  {0, "CONN", 0, "Connect",               st_action_connect},
  {0, "DISC", 0, "Disconnect",            st_action_connect},
  {0, "EXIT", 0, "Exit",                  st_action_exit},
};
int nb_Action = sizeof (Actions) / sizeof (struct act);

#define ACTION_LIST 0
#define ACTION_SET  1
#define ACTION_IND  2
#define ACTION_SEL  3
#define ACTION_INFO 4
#define ACTION_STAT 5
#define ACTION_RUN  6
#define ACTION_CONN 7
#define ACTION_DISC 8
#define ACTION_EXIT 9

/* Current Action */
int mode;

char *GTXfile_types[] =
{
  "Points",
  "Lines",
  "Grid"
};

char *GTXvar_types[] =
{
  "Float",
  "Character",
  "X gravity center",
  "Y gravity center",
  "Z gravity center",
  "Macro"
};

/*****************************************************************************
**
** FUNCTION:    st_usage
**
** DESCRIPTION: gives this program usage
**
** RETURNS:     Nothing
**
** IN_ARGS:     char *exe_name : name of the executable
** IN_ARGS:     int help : 1 to obtain a complete help
**
*****************************************************************************/
static void st_usage(char *exe_name,
                     int help)
{
  (void)printf("Usage: %s [-help] [-host hostname] [-port port_number]\n",
               exe_name);
  (void)printf("          [-path data_path]\n");       

  if (help)
  {
    (void)printf(" -help            : Print this message.\n");
    (void)printf(" -debug           : Print client/server messages.\n");
    (void)printf(" -host hostname   : Server's hostname.\n");
    (void)printf(" -port port_number: Port used between client and server\n");
    (void)printf(" -path data_path  : Initialize connection with given data_path.\n");
  }
}

/*****************************************************************************
**
** FUNCTION:    st_action list 
**
** DESCRIPTION: Compose a string like "study: dir/file/sel"...
**
** RETURNS:     Nothing
**
** IN_ARGS:     int number : not used
**
*****************************************************************************/
static void st_action_list(int number)
{
  int i,first;
#define end_label &label_str[strlen(label_str)]
  char label_str[300];

  (void)printf("#*#*#*#*#*#*#*#*#*#*#*#\n");
  if (!strcmp(current_study,NONE_STRING))
  {
    (void)strcpy(label_str, "Study list:\n");
    goto label_end;
  }
  else
    (void)strcpy(label_str, current_study);

  if (!strcmp(current_dir,NONE_STRING))
  {
    (void)sprintf(end_label, "\nDirectory list:\n");
    goto label_end;
  }
  else
    (void)sprintf(end_label, ": %s", current_dir);

  if (!strcmp(current_file,NONE_STRING))
  {
    (void)sprintf(end_label, "\nFile list:\n");
    goto label_end;
  }
  else
    (void)sprintf(end_label, "/%s", current_file);

  if (strcmp(current_sel,NONE_STRING))
    (void)sprintf(end_label, "/%s", current_sel);

  (void)sprintf(end_label, "\nVariable list:\n");

label_end:
  (void)printf(label_str);
  first = (nb_current_list > 0 && !strcmp(current_list[0], ".."))? 0 : 1;
  for (i=0; i<nb_current_list; i++)
    (void)printf("%d. %s\n", i+first, current_list[i]);
  (void)printf("#*#*#*#*#*#*#*#*#*#*#*#\n");
#undef end_label
}

/*****************************************************************************
**
** FUNCTION:    st_status_mess
**
** DESCRIPTION: print a message in the status zone
**
** RETURNS:     Nothing
**
** IN_ARGS:     char *format : formatted string
**
*****************************************************************************/
static void st_status_mess(char *format, ...)
{
  va_list ap;
  char buffer[1024];
 
  va_start(ap, format);
  (void)vsprintf(buffer, format, ap);
  va_end(ap);

  (void)printf("Status: %s\n",buffer);
}

/*****************************************************************************
**
** FUNCTION:    st_get_selected_item
**
** DESCRIPTION: get the selected item in the list
**
** RETURNS:     char *selected_item
**
** IN_ARGS:     int index
**
** REMARKS:     sel_str must be freed using free
**
*****************************************************************************/
static char *st_get_selected_item(int index)
{
  int first;
  char *sel_str = NULL;

  first = (nb_current_list > 0 && !strcmp(current_list[0], ".."))? 0 : 1;
  if (index < first || index >= nb_current_list+first)
  {
    (void)fprintf(stderr,"index (%d) must be in [%d,%d]",
                  index,first,nb_current_list+first-1);
    goto label_end;
  }
  sel_str = (char*)strdup(current_list[index-first]);

label_end:
  return(sel_str);
}

/*****************************************************************************
**
** FUNCTION:    st_get_study_list
**
** DESCRIPTION: Get the list of studies
**
** RETURNS:     int error code: 1 if cannot get study list, 0 elsewhere
**
*****************************************************************************/
static int st_get_study_list()
{
  int i,error,nb_study;
  char **studies,**new_list;

  error = 1;
  studies = NULL;

  (void)strcpy(current_study,NONE_STRING);

  /*
  ** Get study list
  */
  if (GTXClientGetStudyList(&nb_study, &studies)) goto label_end;

  /* copy the list as is (do not copy pointers as we need to free memory
     allocated by GTXClient with a special function). */
  new_list = (char **)malloc(sizeof(char*) * (nb_study));
  if (new_list == NULL)
  {
    (void)fprintf(stderr,"Cannot allocate %d bytes",
                  (int)(sizeof(char*)*(nb_study)));
    goto label_end;
  }
  for (i=0; i<nb_study; i++)
    new_list[i] = NULL;
  for (i=0; i<nb_study; i++)
    new_list[i] = (char*)strdup(studies[i]);

  /* free previous list */
  for (i=0; i<nb_current_list; i++)
    if (current_list[i] != NULL) free(current_list[i]);
  if (current_list != NULL) free(current_list);
  studies = GTXClientFreeStringArray(nb_study, studies);

  current_list = new_list;
  nb_current_list = nb_study;

  mode = MODE_STUDY;
  st_action_list(0);

  Actions[ACTION_LIST].visible = 1;
  Actions[ACTION_SET ].visible = 1;

  error = 0;
label_end:
  studies = GTXClientFreeStringArray(nb_study, studies);
  return(error);
}

/*****************************************************************************
**
** FUNCTION:    st_get_dir_list
**
** DESCRIPTION: Get the list of directories
**
** RETURNS:     int error code: 1 if cannot get dir list, 0 elsewhere
**
*****************************************************************************/
static int st_get_dir_list(char *study)
{
  int i,error,nb_dir;
  char **new_list;
  char **dirs;

  error = 1;
  dirs = NULL;
  new_list = NULL;

  if (mode == MODE_STUDY)
  {
    if (GTXClientSetStudy(study)) goto label_end;
    (void)strcpy(current_study,study);
  }
  else
    (void)strcpy(current_dir, NONE_STRING);

  /*
  ** Get directory list
  */
  if (GTXClientGetDirectoryList(&nb_dir, &dirs)) goto label_end;

  /* Allocate new list */
  new_list = (char **)malloc(sizeof(char*) * (nb_dir+1));
  if (new_list == NULL)
  {
    (void)fprintf(stderr,"Cannot allocate %d bytes",
                  (int)(sizeof(char*)*(nb_dir+1)));
    goto label_end;
  }
  for (i=0; i<=nb_dir; i++)
    new_list[i] = NULL;
  new_list[0] = (char*)strdup("..");
  if (new_list[0] == NULL) goto label_end;

  for (i=0; i<nb_dir; i++)
    new_list[i+1] = (char*)strdup(dirs[i]);

  Actions[ACTION_INFO].visible = 0;

  /* free previous list */
  for (i=0; i<nb_current_list; i++)
    if (current_list[i] != NULL) free(current_list[i]);
  if (current_list != NULL) free(current_list);
  dirs = GTXClientFreeStringArray(nb_dir, dirs);

  current_list = new_list;
  nb_current_list = nb_dir+1;

  mode = MODE_DIR;
  st_action_list(0);

  error = 0;
label_end:
  dirs = GTXClientFreeStringArray(nb_dir, dirs);
  return(error);
}

/*****************************************************************************
**
** FUNCTION:    st_get_file_list
**
** DESCRIPTION: get the list of files
**
** RETURNS:     int error code: 1 if cannot get file list, 0 elsewhere
**
*****************************************************************************/
static int st_get_file_list(char *dir)
{
  int i,error,nb_file;
  char **new_list;
  char **files;

  error = 1;
  files = NULL;
  new_list = NULL;

  if (mode == MODE_DIR)
  {
    if (GTXClientSetDirectory(dir)) goto label_end;
    (void)strcpy(current_dir,dir);
  }
  else
  {
    (void)strcpy(current_file, NONE_STRING);
    (void)strcpy(current_var,  NONE_STRING);
    (void)strcpy(current_sel,  NONE_STRING);
    current_indice = -1;
  }

  /*
  ** Get file list
  */
  if (GTXClientGetFileList(&nb_file, &files)) goto label_end;

  /* Allocate new list */
  new_list = (char **)malloc(sizeof(char*) * (nb_file+1));
  if (new_list == NULL)
  {
    (void)fprintf(stderr,"Cannot allocate %d bytes",
                  (int)(sizeof(char*)*(nb_file+1)));
    goto label_end;
  }
  for (i=0; i<=nb_file; i++)
    new_list[i] = NULL;
  new_list[0] = (char*)strdup("..");
  if (new_list[0] == NULL) goto label_end;

  for (i=0; i<nb_file; i++)
    new_list[i+1] = (char*)strdup(files[i]);

  Actions[ACTION_IND ].visible = 0;
  Actions[ACTION_INFO].visible = 1;
  Actions[ACTION_STAT].visible = 0;
  Actions[ACTION_SEL ].visible = 0;

  /* free previous list */
  for (i=0; i<nb_current_list; i++)
    if (current_list[i] != NULL) free(current_list[i]);
  if (current_list != NULL) free(current_list);
  files = GTXClientFreeStringArray(nb_file, files);

  current_list = new_list;
  nb_current_list = nb_file+1;

  mode = MODE_FILE;
  st_action_list(0);

  error = 0;
label_end:
  files = GTXClientFreeStringArray(nb_file, files);
  return(error);
}

/*****************************************************************************
**
** FUNCTION:    st_get_var_list
**
** DESCRIPTION: Get the list of variables
**
** RETURNS:     int error code: 1 if cannot get var list, 0 elsewhere
**
*****************************************************************************/
static int st_get_var_list(char *file)
{
  int i,error,nb_var;
  char **new_list;
  char **vars;

  error = 1;
  vars = NULL;
  new_list = NULL;

  if (mode == MODE_FILE)
  {
    if (GTXClientSetFile(file)) goto label_end;
    (void)strcpy(current_file,file);
  }
  else
  {
    (void)strcpy(current_var,  NONE_STRING);
    (void)strcpy(current_sel,  NONE_STRING);
  }

  /*
  ** Get variable list
  */
  if (GTXClientGetVariableList(&nb_var, &vars)) goto label_end;

  /* Allocate new list */
  new_list = (char **)malloc(sizeof(char*) * (nb_var+1));
  if (new_list == NULL)
  {
    (void)fprintf(stderr,"Cannot allocate %d bytes",
                  (int)(sizeof(char*)*(nb_var+1)));
    goto label_end;
  }
  for (i=0; i<=nb_var; i++)
    new_list[i] = NULL;
  new_list[0] = (char*)strdup("..");
  if (new_list[0] == NULL) goto label_end;

  for (i=0; i<nb_var; i++)
    new_list[i+1] = (char*)strdup(vars[i]);

  Actions[ACTION_SEL ].visible = 1;

  /* free previous list */
  for (i=0; i<nb_current_list; i++)
    if (current_list[i] != NULL) free(current_list[i]);
  if (current_list != NULL) free(current_list);
  vars = GTXClientFreeStringArray(nb_var, vars);

  current_list = new_list;
  nb_current_list = nb_var+1;

  mode = MODE_VAR;
  st_action_list(0);

  error = 0;
label_end:
  vars = GTXClientFreeStringArray(nb_var, vars);
  return(error);
}

/*****************************************************************************
**
** FUNCTION:    st_action_set_sel
**
** DESCRIPTION: set/unset selection
**
** IN_ARGS:     index of the selection in the list
**
** RETURNS:     Nothing
**
*****************************************************************************/
static void st_action_set_sel(int index)
{
  char *sel_str = NULL;

  if (mode != MODE_VAR) goto label_end;

  /* Unset selection */
  if (strcmp(current_sel, NONE_STRING))
  {
    st_status_mess("Unsetting selection...");
    if (GTXClientSetSelection(NULL)) goto label_end;
    (void)strcpy(current_sel, NONE_STRING);
  }
  else
  {
    sel_str = st_get_selected_item(index);
    if (sel_str == NULL) goto label_end;

    if (!strcmp(sel_str, ".."))
    {
      st_status_mess("Cannot set \"..\" as the selection");
      goto label_end;
    }
    st_status_mess("Setting selection %s...", sel_str);
    if (GTXClientSetSelection(sel_str)) goto label_end;
    (void)strcpy(current_sel, sel_str);
    st_status_mess("Selection %s... set", sel_str);
  }

  st_action_list(0);

label_end:
  if (sel_str != NULL) free(sel_str);
  return;
}

/*****************************************************************************
**
** FUNCTION:    st_action_set
**
** DESCRIPTION: Callback attached to Set item
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_action_set(int index)
{
  char *sel_str;
  GTXVariableInfo var_info;

  sel_str = st_get_selected_item(index);
  if (sel_str == NULL) goto label_end;

  if (!strcmp(sel_str, ".."))
  {
    switch (mode)
    {
      case MODE_DIR:
        st_status_mess("Retrieving Study List");
        if (st_get_study_list()) goto label_end;
        break;
      case MODE_FILE:
        st_status_mess("Retrieving Directory List");
        if (st_get_dir_list(NULL)) goto label_end;
        break;
      case MODE_VAR:
        st_status_mess("Retrieving File List");
        if (st_get_file_list(NULL)) goto label_end;
        mode = MODE_FILE;
        break;
      default:
        st_status_mess("Unknown Mode");
        goto label_end;
    } /* end of switch */
  } /* end of if back */
  else
  {
    switch (mode)
    {
      case MODE_STUDY:
        st_status_mess("Setting Study %s", sel_str);
        if (st_get_dir_list(sel_str)) goto label_end;
        st_status_mess("Study %s set",sel_str);
        break;

      case MODE_DIR:
        st_status_mess("Setting Directory %s", sel_str);
        if (st_get_file_list(sel_str)) goto label_end;
        st_status_mess("Directory %s set",sel_str);
        break;

      case MODE_FILE:
        st_status_mess("Setting File %s", sel_str);
        if (st_get_var_list(sel_str)) goto label_end;
        st_status_mess("File %s set",sel_str);
        break;

      case MODE_VAR:
        st_status_mess("Setting Variable %s", sel_str);
        if (GTXClientSetVariable(sel_str)) goto label_end;
        current_indice = -1;
        st_status_mess("Variable %s set",sel_str);
        if (GTXClientGetVariableInfo(&var_info)) goto label_end;
        Actions[ACTION_IND ].visible =
          (var_info.type == GTX_VAR_TYPE_MACRO);
        Actions[ACTION_STAT].visible =
          (var_info.type != GTX_VAR_TYPE_MACRO);
        break;

      default:
        st_status_mess("Unknown Mode");
        goto label_end;
    } /* end of switch */
  }

label_end:
  if (sel_str != NULL) free(sel_str);
  return;
}

/*****************************************************************************
**
** FUNCTION:    st_action_indice
**
** DESCRIPTION: Callback attached to Set indice
**
** RETURNS:     None
**
** IN_ARGS:     Widget w : calling widget
**
*****************************************************************************/
static void st_action_indice(int index)
{
  if (mode != MODE_VAR) goto label_end;

  st_status_mess("Setting Indice %d", index);

  if (GTXClientSetIndice(index)) goto label_end;

  Actions[ACTION_STAT].visible = 1;
  current_indice = index;

label_end:
  return;
}

/*****************************************************************************
**
** FUNCTION:    st_action_connect
**
** DESCRIPTION: Action attached to Connect/Disconnect
**
** RETURNS:     None
**
** IN_ARGS:     int number : not used
**
*****************************************************************************/
static void st_action_connect(int number)
{
  if (mode == MODE_INIT)
  {
    st_status_mess("Trying to connect to %s:%d",host,port);
    /* Connect to the server */
    if (GTXClientInitialize(GTX_CLIENT_VERSION)) goto label_end;
    if (GTXClientConnect(host, port, data_path)) goto label_end;

    /* get study list */
    if (st_get_study_list()) goto label_end;
    Actions[ACTION_RUN ].visible = 0;
    Actions[ACTION_CONN].visible = 0;
    Actions[ACTION_DISC].visible = 1;
    st_status_mess("Connected");
  }
  else
  {
    GTXClientDisconnect();
    mode = MODE_INIT;
    Actions[ACTION_LIST].visible = 0;
    Actions[ACTION_SET ].visible = 0;
    Actions[ACTION_IND ].visible = 0;
    Actions[ACTION_INFO].visible = 0;
    Actions[ACTION_STAT].visible = 0;
    Actions[ACTION_SEL ].visible = 0;
    Actions[ACTION_RUN ].visible = 1;
    Actions[ACTION_CONN].visible = 1;
    Actions[ACTION_DISC].visible = 0;
    st_status_mess("Disconnected");
  }

label_end:
  return;
}

/*****************************************************************************
**
** FUNCTION:    st_action_run
**
** DESCRIPTION: Action attached to Run button
**
** RETURNS:     None
**
** IN_ARGS:     int number : not used
**
*****************************************************************************/
static void st_action_run(int number)
{
  port = 0;

  st_status_mess("Running GTXserver on a random port");
  /* Connect to the server */
  if (GTXClientInitialize(GTX_CLIENT_VERSION)) goto label_end;
  if (GTXClientRunGTXserver(&port)) goto label_end;
  st_action_connect(number);
label_end:
  return;
}

/*****************************************************************************
**
** FUNCTION:    st_action_exit
**
** DESCRIPTION: Action attached to Exit
**
** RETURNS:     None
**
** IN_ARGS:     int number : not used
**
*****************************************************************************/
static void st_action_exit(int number)
{
  switch (mode)
  {
    case MODE_INIT:
      exit(0);

    case MODE_STUDY:
    case MODE_DIR:
    case MODE_FILE:
    case MODE_VAR:
      GTXClientDisconnect();
      exit(0);

    default:
      st_status_mess("Unknown mode!!");
  } /* end of switch */

  return;
}

/*****************************************************************************
**
** FUNCTION:    st_action_info
**
** DESCRIPTION: Action attached to Information
**
** RETURNS:     None
**
** IN_ARGS:     int index : selected item
**
*****************************************************************************/
static void st_action_info(int index)
{
#define end_buffer &buffer[strlen(buffer)]

  GTXFileInfo file_info;
  GTXVariableInfo var_info;
  GTXFaultInfo *fault_info;
  GTXFaultSystem fault_system;
  int i,j,len,nb_indices, *indices;
  char buffer[2000];
  char **alpha_indices;
  char *sel_str,*comment;

  indices = NULL;
  nb_indices = 0;
  alpha_indices = NULL;

  sel_str = st_get_selected_item(index);
  if (sel_str == NULL) goto label_end;

  if (!strcmp(sel_str, ".."))
  {
    st_status_mess("Cannot get Information on \"..\"");
    goto label_end;
  }

  switch (mode)
  {
    case MODE_FILE:
      st_status_mess("Getting File Information...");
      if (strcmp(sel_str, current_file))
      {
        if (GTXClientSetFile(sel_str)) goto label_end;
        (void)strcpy(current_file, sel_str);
      }
      if (GTXClientGetFileInfo(&file_info)) goto label_end;

      (void)sprintf(buffer,"File: %s\n", sel_str);
      (void)sprintf(end_buffer,"Type: %s\n",
                    GTXfile_types[file_info.type]);
      (void)sprintf(end_buffer,"Dimension: %dD\n", file_info.dimension);
      (void)sprintf(end_buffer,"Number of Samples: %"GTXLONG_FORMAT"\n",
                    file_info.s_number);

      if (file_info.type == GTX_FILE_TYPE_GRAVITY_LINES)
        (void)sprintf(end_buffer,"Number of Lines: %d\n",
                      file_info.item_number);
      else if (file_info.type == GTX_FILE_TYPE_GRID)
      {
        if (file_info.dimension == 2)
        {
          (void)sprintf(end_buffer, "Number of Nodes: (%d,%d)\n",
                        file_info.NX,file_info.NY);
          (void)sprintf(end_buffer, "Origin         : (%g,%g)\n",
                        file_info.X0,file_info.Y0);
          (void)sprintf(end_buffer, "Mesh Dimensions: (%g,%g)\n",
                        file_info.DX,file_info.DY);
          if (file_info.rotation)
            (void)sprintf(end_buffer, "Rotation Angle : %g\n",
                          file_info.angle_z);
          else
            (void)sprintf(end_buffer, "No Rotation\n");
        }
        else if (file_info.dimension == 3)
        {
          (void)sprintf(end_buffer, "Number of Nodes: (%d,%d,%d)\n",
                        file_info.NX,file_info.NY,file_info.NZ);
          (void)sprintf(end_buffer, "Origin         : (%g,%g,%g)\n",
                        file_info.X0,file_info.Y0,file_info.Z0);
          (void)sprintf(end_buffer, "Mesh Dimensions: (%g,%g,%g)\n",
                        file_info.DX,file_info.DY,file_info.DZ);
          if (file_info.rotation)
          {
            (void)sprintf(end_buffer, "Rotation Angle around Z: %g\n",
                          file_info.angle_z);
            (void)sprintf(end_buffer, "Rotation Angle around Y: %g\n",
                          file_info.angle_y);
            (void)sprintf(end_buffer, "Rotation Angle around X: %g\n",
                          file_info.angle_x);
          }
          else
            (void)sprintf(end_buffer, "No Rotation\n");
        } /* end of 3D grid */
      } /* end of  grid type */

      /* special variables information */
      (void)sprintf(end_buffer, "Sample Number variable   : %s\n",
                    file_info.sn_var_name);
      if (file_info.type == GTX_FILE_TYPE_GRAVITY_LINES)
      {
        (void)sprintf(end_buffer, "Line Number variable     : %s\n",
                      file_info.ln_var_name);
        (void)sprintf(end_buffer, "Relative Number variable : %s\n",
                      file_info.rn_var_name);
      }
      (void)sprintf(end_buffer, "X Gravity Center variable: %s\n",
                    file_info.xg_var_name);
      (void)sprintf(end_buffer, "Y Gravity Center variable: %s\n",
                    file_info.yg_var_name);
      if (file_info.dimension == 3)
        (void)sprintf(end_buffer, "Z Gravity Center variable: %s\n",
                      file_info.zg_var_name);

      /* faults information */
      if (file_info.faulted)
      {
        fault_info = &file_info.fault_info;
        (void)sprintf(end_buffer, "\nFaulted File:\n");
        (void)sprintf(end_buffer, "2D Faults              : %c\n",
                      (fault_info->faults_2d)? 'Y': 'N');
        (void)sprintf(end_buffer, "Auxiliary Variable Used: %c\n",
                      (fault_info->aux_var_used)? 'Y': 'N');
        (void)sprintf(end_buffer, "Number of faults       : %d\n",
                      fault_info->faults_number);
        (void)sprintf(end_buffer, "Number of segments     : %d\n",
                      fault_info->segments_number);
        (void)sprintf(end_buffer, "Minimum Used Priority  : %d\n",
                      fault_info->min_fpriority);
        (void)sprintf(end_buffer, "Maximum Used Priority  : %d\n",
                      fault_info->max_fpriority);
        (void)sprintf(end_buffer, "Authorized Priority    : %d\n",
                      fault_info->auth_priority);
        if (GTXClientReadFaults(0,&fault_system))
          goto label_end;
      } /* fault information */
      break;

    case MODE_VAR:
      st_status_mess("Getting Variable Information...");
      if (strcmp(sel_str, current_var))
      {
        if (GTXClientSetVariable(sel_str)) goto label_end;
        (void)strcpy(current_var, sel_str);
        current_indice = -1;
      }
      if (GTXClientGetVariableInfo(&var_info)) goto label_end;

      (void)sprintf(buffer, "Variable  : %s\n", sel_str);
      (void)sprintf(end_buffer,"Type      : %s\n",
                    GTXvar_types[var_info.type]);
      if (var_info.type == GTX_VAR_TYPE_CHAR)
        (void)sprintf(end_buffer,"Alpha Length : %d\n", var_info.alpha_length);
      else
      {
        if (var_info.bit_length == 1)
          (void)sprintf(end_buffer, "Precision : 1 bit (selection)\n");
        else
          (void)sprintf(end_buffer, "Precision : %d bits\n",
                        var_info.bit_length);
        if (!var_info.implicit)
          (void)sprintf(end_buffer, "Unit      : %s\n", var_info.unit);
      }

      if (GTXClientGetVariableComment(&comment)) goto label_end;
      if (comment[0] != '\0')
        (void)sprintf(end_buffer,"Comment : %s\n", comment);
      comment = GTXClientFreePointer(comment);
      if (var_info.type == GTX_VAR_TYPE_MACRO)
      {
        if (GTXClientGetMacroIndicesWithAlpha(&nb_indices, &indices,
                                              &alpha_indices)) goto label_end;
        (void)sprintf(end_buffer, "Number of Indices : %d\n", nb_indices);
        if (nb_indices != 0)
        {
          if (alpha_indices)
            (void)sprintf(end_buffer, "Index  -> Alpha Index\n");
          else
            (void)sprintf(end_buffer, "Indices:\n");
          len = (int)strlen(buffer);
          for (i=0, j=0; i<nb_indices; i++)
          {
            if (len > 1000)
            {
              (void)strcat(&buffer[len], "...\n");
              break;
            }
            (void)sprintf(&buffer[len], "%06d", indices[i]);
            len += 6;
            if (alpha_indices)
            {
              (void)sprintf(&buffer[len], " -> %s\n", alpha_indices[i]);
              len += (int)strlen(&buffer[len]);
            }
            else
            {
              if (++j == 5)
              {
                buffer[len++] = '\n';
                buffer[len] = '\0';
                j = 0;
              }
            }
          }
        }
      }
      break;

    default:
      (void)fprintf(stderr, "Unknown Mode!!");
      exit(1);
  } /* end of switch */

  (void)printf(buffer);
label_end:
  indices = GTXClientFreePointer(indices);
  alpha_indices = GTXClientFreeStringArray(nb_indices,alpha_indices);
  if (sel_str != NULL) free(sel_str);
  return;
#undef end_buffer
}

/*****************************************************************************
**
** FUNCTION:    st_action_stat
**
** DESCRIPTION: Action attached to Statistics
**
** RETURNS:     None
**
** IN_ARGS:     int index : not used
**
*****************************************************************************/
static void st_action_stat(int index)
{
#define end_buffer &buffer[strlen(buffer)]
  GTXFileInfo file_info;
  GTXVariableInfo  var_info;
  char buffer[2000];
  int compress;

  gtx_long i,nech,n_kept,nb_def;
  double dbl_test_val;
  double *dbl_data = NULL;
  double min,max,mean;

  char *char_test_val = NULL;
  char **char_data = NULL;

  if (GTXClientGetFileInfo(&file_info)) goto label_end;
  if (GTXClientGetVariableInfo(&var_info)) goto label_end;

  compress = 1;                 /* always compress */

  nech=file_info.s_number;
  nb_def = 0;

  (void)sprintf(buffer, "Variable: %s\n",current_var);

  st_status_mess("Reading variable...");
  /* Alphanumeric variable reading */
  if (var_info.type == GTX_VAR_TYPE_CHAR)
  {
    if (GTXClientReadCharVariable(compress, &char_test_val,&n_kept,&char_data))
      goto label_end;
  }
  else                          /* all other variables */
    if (GTXClientReadDoubleVariable(compress, &dbl_test_val,&n_kept,&dbl_data))
      goto label_end;

  /* number of samples (not masked) */
  if (n_kept != nech)
    (void)sprintf(end_buffer, "Total Number of Samples: %"GTXLONG_FORMAT
                  "/%"GTXLONG_FORMAT"\n",n_kept,nech);
  else
    (void)sprintf(end_buffer, "Total Number of Samples: %"GTXLONG_FORMAT"\n",nech);

  if (var_info.type == GTX_VAR_TYPE_CHAR)
  {
    for (i=0; i<n_kept; i++)
      if (strcmp(char_data[i],char_test_val))
        nb_def++;
    (void)sprintf(end_buffer, "Number of Defined Samples: %"GTXLONG_FORMAT"\n",nb_def);
  }
  else if (var_info.type == GTX_VAR_TYPE_FLOAT &&
           var_info.bit_length == 1)
  {
    for (i=0; i<n_kept; i++)
      if (dbl_data[i] == 1)
        nb_def++;
    (void)sprintf(end_buffer, "Number of Selected Samples: %"GTXLONG_FORMAT"\n",nb_def);
  }
  else
  {
    min  = 1.e300;
    max  = -1.e300;
    mean = 0.;

    for (i=0; i<n_kept; i++)
      if  (dbl_data[i] != dbl_test_val)
      {
        nb_def++;
        mean += dbl_data[i];
        min = (dbl_data[i] < min)? dbl_data[i]: min;
        max = (dbl_data[i] > max)? dbl_data[i]: max;
      }
    mean /= nb_def;
    (void)sprintf(end_buffer, "Number of Defined Samples: %"GTXLONG_FORMAT"\n",nb_def);
    (void)sprintf(end_buffer, "Minimum Value: %g\n",min);
    (void)sprintf(end_buffer, "Maximum Value: %g\n",max);
    (void)sprintf(end_buffer, "Mean Value:    %g\n",mean);
  }

  (void)printf(buffer);

label_end:
  char_test_val = GTXClientFreePointer(char_test_val);
  char_data = GTXClientFreeStringArray(n_kept, char_data);
  dbl_data = GTXClientFreePointer(dbl_data);
#undef end_buffer
}

/*****************************************************************************
**
** FUNCTION:    st_menu
**
** DESCRIPTION: Create the menu
**
** RETURNS:     None
**
*****************************************************************************/
static void st_menu()
{
  struct act *act_ptr;
  int i,nb,index;
  char line[80];
  char cur_act[20];

  Actions[ACTION_RUN ].visible = 1;
  Actions[ACTION_CONN].visible = 1;
  Actions[ACTION_EXIT].visible = 1;

  while (1)
  {
    (void)printf("\n#*#*#*#*#*#*#*#*#*#*#*#*#\n");
    (void)printf("#*  Available actions  *#\n");
    (void)printf("#*#*#*#*#*#*#*#*#*#*#*#*#\n");
    for (i=0; i<nb_Action; i++)
    {
      act_ptr = &Actions[i];
      if (act_ptr->visible)
        (void)printf("%-4s%7s %s\n", act_ptr->action,
                     (act_ptr->need_index)? "index:": ":", act_ptr->title);
    }
    (void)printf("\nYour choice: ");

    fflush(stdout);
    if (fgets(line,80,stdin) == NULL) break;
    if (sscanf(line, "%19s%n",cur_act,&nb) != 1) continue;

    /* search for action */
    for (i=0; i<nb_Action; i++)
    {
      act_ptr = &Actions[i];
#if defined(winnt)
      if (act_ptr->visible && !_stricmp(cur_act, Actions[i].action))
#else
      if (act_ptr->visible && !strcasecmp(cur_act, Actions[i].action))
#endif
      {
        /* search for index if necessary */
        index = 0;
        if (act_ptr->need_index && sscanf(&line[nb], "%d", &index) != 1)
        {
          (void)fprintf(stderr,"action must be followed by a number");
          break;
        } /* need index and error in index */
        act_ptr->action_func(index);
        break;
      } /* if visible action found */
    } /* loop on actions */
    if (i == nb_Action)
      (void)fprintf(stderr,"Not a valid action: \"%s\"", cur_act);
  } /* end of infinite while */
}

/****************************************************************************
**
** FUNCTION:    main
**
** DESCRIPTION: client's main function
**
** RETURNS:     Nothing
**
** IN_ARGS:     int argv, char *argv : program arguments
**
*****************************************************************************/
int main(int argc,
         char *argv[])
{
  char *tmp_str;
  int i,error;

  error = 1;
  port = GTXClientGetDefaultPort();

  (void)strcpy(current_study,NONE_STRING);
  (void)strcpy(current_dir,  NONE_STRING);
  (void)strcpy(current_file, NONE_STRING);
  (void)strcpy(current_sel,  NONE_STRING);
  (void)strcpy(current_var,  NONE_STRING);
  current_indice = -1;

  (void)strcpy(host, DEFAULT_GTX_SERVERNAME);
  (void)strcpy(data_path, "");

  for (i=1; i<argc; i++)
  {
    if (!strcmp(argv[i], "-port"))
    {
      i++;
      if (i<argc)
      {
        port = (unsigned short)strtol(argv[i], &tmp_str, 10);
        if (port == 0 && tmp_str == argv[i])
        {
          (void)fprintf(stderr,"Invalid Port Number.\n");
          exit(1);
        }
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-host"))
    {
      i++;
      if (i<argc)
      {
        if (strlen(argv[i]) > MAX_HOSTNAME_LEN-1)
        {
          (void)fprintf(stderr,"Host Name must contain less than %d characters.\n",
                        MAX_HOSTNAME_LEN);
          exit(1);
        }
        (void)strcpy(host, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-path"))
    {
      i++;
      if (i<argc)
      {
        if (strlen(argv[i]) >= MAX_PATH_LEN)
        {
          (void)fprintf(stderr,"Path must contain less than %d characters.\n",
                        MAX_PATH_LEN);
          exit(1);
        }
        (void)strcpy(data_path, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-debug"))
      GTXClientDebug(1);
    else if (!strcmp(argv[i], "-help"))
    {
      st_usage(argv[0], 1);
      exit(0);
    }
  }

  st_status_mess("Disconnected");
  mode = MODE_INIT;

  st_menu();

  error = 0;

  /*
  ** Disconnect from the server
  */
  GTXClientDisconnect();

  return(error);
}
