/*****************************************************************************
 
            Copyright (c)2013 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/
 
/* $Id: write_files.c 14083 2009-03-27 08:18:41Z foucher $ */
 
#include <GTXClient.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define DEFAULT_GTX_SERVERNAME "localhost"

#define MAX_HOSTNAME_LEN 1024
#define MAX_PATH_LEN     1024
#define MAX_STUDY_LEN    50

/****************************************************************************
**
** FUNCTION: st_usage
**
** PURPOSE:  gives this program usage
**
** IN_ARGS:  exe_name: name of the executable (argv[0])
** IN_ARGS:  help    : 1 to obtain a complete help
**
*****************************************************************************/
static void st_usage(char *exe_name,
                     int help)
{
  (void)printf("Usage: %s -study XXXX [-help] [-host hostname] [-port port_number]\n",
               exe_name);
  (void)printf("          [-path data_path]");

  if (help)
  {
    (void)printf(" -help             : Print this message.\n");
    (void)printf(" -debug            : Print client/server messages.\n");
    (void)printf(" -host hostname    : Server's hostname.\n");
    (void)printf(" -port port_number : Port used between client and server\n");
    (void)printf(" -path data_path   : Initialize connection with given data_path.\n");
    (void)printf(" -sutdy data_path  : Name of an existing study to write in.\n");
    (void)printf("The default is to run a new server on a random port.");
    (void)printf("If -host or -port is specified, a server must have been started first on the given host/port");
  }
}

/****************************************************************************
**
** FUNCTION:    st_list_palettes
**
** DESCRIPTION: list the palettes that exists in the study
**
** RETURNS:     0 if OK, 1 in case of error
**
** ARGUMENTS:   dump_rgbs: 0 to dump only names, 1 to dump the whole contents
**
*****************************************************************************/
static int st_list_palettes(int dump_rgbs)
{
  int i,nb_palettes,icol,ncol,*rgbs = NULL;
  char **palettes;
  int error = 1;
  
  /*
  ** Printout list of Palettes
  */
  if (GTXClientPaletteGetList(&nb_palettes, &palettes))
    goto label_end;
  if (nb_palettes)
    printf("List of existing Palettes in the study:\n");
  else
    printf("No existing Palette in the study.\n");
  for (i = 0; i < nb_palettes; i++)
  {
    printf("- %s\n", palettes[i]);
    /* Printout palettes contents: */
    if (dump_rgbs)
    {
      if (GTXClientPaletteRead(palettes[i], &ncol, &rgbs))
        goto label_end;

      printf("    this palette has the following %d colors\n", ncol);
      for (icol = 0; icol < ncol; icol++)
        printf("    - R(%d), G(%d), B(%d)\n",
               rgbs[icol], rgbs[ncol+icol], rgbs[2*ncol+icol]);

      rgbs = GTXClientFreePointer(rgbs);
    }
  }

  error = 0;
label_end:
  palettes = GTXClientFreeStringArray(nb_palettes, palettes);
  rgbs = GTXClientFreePointer(rgbs);
  return (error);
}

/****************************************************************************
**
** FUNCTION:    st_create_palette
**
** DESCRIPTION: create our own palette in the study (6 colors)
**
** RETURNS:     0 if OK, 1 in case of error
**
** ARGUMENTS:   new_pal_name: Name of the palette to be created
**
*****************************************************************************/
static int st_create_palette(const char* new_pal_name)
{
  int icol, ncol, error;
  int rgbs[3*10];

  error = 1;

  printf("Creating Palette named %s...\n", new_pal_name);

  icol = 0;
  ncol = 6;
  rgbs[icol] = 255; rgbs[ncol+icol] =   0; rgbs[2*ncol+icol] =   0; icol++;  /* Red */
  rgbs[icol] =   0; rgbs[ncol+icol] = 255; rgbs[2*ncol+icol] =   0; icol++;  /* Green */
  rgbs[icol] =   0; rgbs[ncol+icol] =   0; rgbs[2*ncol+icol] = 255; icol++;  /* Blue */
  rgbs[icol] = 255; rgbs[ncol+icol] = 255; rgbs[2*ncol+icol] =   0; icol++;  /* Yellow */
  rgbs[icol] =   0; rgbs[ncol+icol] = 255; rgbs[2*ncol+icol] = 255; icol++;  /* Cyan */
  rgbs[icol] = 255; rgbs[ncol+icol] =   0; rgbs[2*ncol+icol] = 255; icol++;  /* Magenta */

  if (GTXClientPaletteWrite(new_pal_name, ncol, rgbs))
    goto label_end;

  error = 0;
label_end:
  return (error);
}

/****************************************************************************
**
** FUNCTION:    st_list_color scales
**
** DESCRIPTION: list the color scales that exists in the study
**
** RETURNS:     0 if OK, 1 in case of error
**
*****************************************************************************/
static int st_list_color_scales()
{
  int nb_cscales,i;
  char **cscales;
  int error = 1;

  if (GTXClientColorScaleGetList(&nb_cscales, &cscales))
    goto label_end;
  if (nb_cscales)
    printf("List of existing Color Scales in the study:\n");
  else
    printf("No existing Color Scale in the study.\n");
  for (i = 0; i < nb_cscales; i++)
    printf("- %s\n", cscales[i]);

  error = 0;
label_end:
  cscales = GTXClientFreeStringArray(nb_cscales, cscales);
  return (error);
}

/****************************************************************************
**
** FUNCTION:    st_create_manual_color_scale
**
** DESCRIPTION: create our own manual color scale in the study (6 colors)
**
** RETURNS:     0 if OK, 1 in case of error
**
** ARGUMENTS:   pal_name: Name of the palette to be created
** ARGUMENTS:   color_scale_name: Name of the color scale to be created
**
*****************************************************************************/
static int st_create_manual_color_scale(const char* pal_name,
                                        const char* color_scale_name)
{
  int icl, ncol, error, exists;
  int *rgbs = NULL;
  GTXColorScale cscale, cscale_read_back;

  printf("Creating Manual Color Scale named %s...\n", color_scale_name);

  error = 1;
  GTXClientColorScaleInit(&cscale);
  GTXClientColorScaleInit(&cscale_read_back);

  /* Check Palette contents */
  if (GTXClientPaletteRead(pal_name, &ncol, &rgbs))
    goto label_end;

  if (GTXClientColorScaleExists(color_scale_name, &exists))
    goto label_end;
  if (exists &&
      GTXClientColorScaleDelete(color_scale_name))
    goto label_end;

  cscale.type = GTX_CSCALE_TYPE_MANUAL;
  (void)strcpy(cscale.palette_name, pal_name);
  cscale.nclasses = ncol;
  cscale.bounds = malloc(sizeof(double)*(cscale.nclasses+1));
  cscale.pal_ind = malloc(sizeof(int)*cscale.nclasses);
  for (icl = 0; icl < cscale.nclasses; icl++)
  {
    cscale.bounds[icl] = icl*10.;
    cscale.pal_ind[icl] = icl;
  }
  cscale.bounds[cscale.nclasses] = cscale.nclasses*20.; /* make it irregular */

  cscale.undef_col[0]   = cscale.undef_col[1]    = cscale.undef_col[2]   = -1;
  cscale.lowval_col[0]  = cscale.lowval_col[1]   = cscale.undef_col[2]   = 50;
  cscale.highval_col[0] = cscale.highval_col[1]  = cscale.highval_col[2] = 150;

  if (GTXClientColorScaleNew(color_scale_name, &cscale))
    goto label_end;

  /* Just read it back (in another copy as allocations functions are different) to check everything works */
  if (GTXClientColorScaleRead(color_scale_name, &cscale_read_back))
    goto label_end;

  error = 0;
label_end:
  if (cscale.bounds)
    free(cscale.bounds);
  if (cscale.pal_ind)
    free(cscale.pal_ind);
  GTXClientColorScaleFree(&cscale_read_back);
  return (error);
}

/****************************************************************************
**
** FUNCTION:    st_create_facies_color_scale
**
** DESCRIPTION: create our own facies color scale in the study (3 colors)
**
** RETURNS:     0 if OK, 1 in case of error
**
** ARGUMENTS:   color_scale_name: Name of the color scale to be created
**
*****************************************************************************/
static int st_create_facies_color_scale(const char* color_scale_name)
{
  int icl, iint, error, exists;
  int *rgbs = NULL;
  GTXColorScale cscale;

  printf("Creating Facies Color Scale named %s...\n", color_scale_name);

  error = 1;
  GTXClientColorScaleInit(&cscale);

  if (GTXClientColorScaleExists(color_scale_name, &exists))
    goto label_end;
  if (exists &&
      GTXClientColorScaleDelete(color_scale_name))
    goto label_end;

  cscale.type = GTX_CSCALE_TYPE_FACIES;
  cscale.nclasses = 3;
  cscale.facies_int_nb = 6;
  cscale.facies_int_min = malloc(sizeof(double)*cscale.facies_int_nb);
  cscale.facies_int_max = malloc(sizeof(double)*cscale.facies_int_nb);
  cscale.facies_class_ind = malloc(sizeof(int)*cscale.facies_int_nb);
  cscale.facies_rgbs = malloc(sizeof(int)*cscale.nclasses*3);
  cscale.facies_labels = malloc(sizeof(char*)*cscale.nclasses);
  for (icl = 0; icl < 3; icl++)
    cscale.facies_labels[icl] = NULL;

  cscale.undef_col[0] = cscale.undef_col[1] = cscale.undef_col[2]  = -1;
  cscale.other_col[0] = cscale.other_col[1]  = cscale.other_col[2] = 50;

  for (iint = 0; iint < 6; iint++)
  {
    cscale.facies_int_min[iint] = iint*10.;
    cscale.facies_int_max[iint] = (iint+1)*10.;
    cscale.facies_class_ind[iint] = iint%cscale.nclasses;
  }
  for (icl = 0; icl < cscale.nclasses; icl++)
  {
    cscale.facies_labels[icl] = malloc(sizeof(char)*10);
    (void)sprintf(cscale.facies_labels[icl], "Facies %d", icl+1);
    cscale.facies_rgbs[icl] = 255;
    cscale.facies_rgbs[cscale.nclasses+icl] = 255*icl/cscale.nclasses;
    cscale.facies_rgbs[2*cscale.nclasses+icl] = 0;
  }

  if (GTXClientColorScaleNew(color_scale_name, &cscale))
    goto label_end;

  error = 0;
label_end:
  if (cscale.facies_int_min)
    free(cscale.facies_int_min);
  if (cscale.facies_int_max)
    free(cscale.facies_int_max);
  if (cscale.facies_class_ind)
    free(cscale.facies_class_ind);
  if (cscale.facies_rgbs)
    free(cscale.facies_rgbs);
  if (cscale.facies_labels)
  {
    for (icl = 0; icl < 3; icl++)
      if (cscale.facies_labels[icl])
        free(cscale.facies_labels[icl]);
    free(cscale.facies_labels);
  }
  return (error);
}

/****************************************************************************
**
** FUNCTION:    main
**
** DESCRIPTION: client's main function
**
** RETURNS:     Nothing
**
** ARGUMENTS:   int argv, char *argv : program arguments
**
*****************************************************************************/
int main(int argc,
         char *argv[])
{
  char *tmp_str, host[MAX_HOSTNAME_LEN];
  unsigned short port;
  int i,error,run_server;
  char data_path[MAX_PATH_LEN];
  char study_name[MAX_STUDY_LEN];
  char new_pal_name[50];

  error = 1;
  run_server = 1;
  port = GTXClientGetDefaultPort();

  (void)strcpy(host, DEFAULT_GTX_SERVERNAME);
  (void)strcpy(data_path, "");
  (void)strcpy(study_name, "");

  for (i = 1; i < argc; i++)
  {
    if (!strcmp(argv[i], "-port"))
    {
      i++;
      if (i < argc)
      {
        run_server = 0;
        port = (unsigned short)strtol(argv[i], &tmp_str, 10);
        if (port == 0 && tmp_str == argv[i])
        {
          (void)fprintf(stderr,"Invalid Port Number.");
          exit(1);
        }
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-host"))
    {
      i++;
      if (i<argc)
      {
        run_server = 0;
        if (strlen(argv[i]) > MAX_HOSTNAME_LEN-1)
        {
          (void)fprintf(stderr,"Host Name must contain less than %d characters.",
                        MAX_HOSTNAME_LEN);
          exit(1);
        }
        (void)strcpy(host, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-path"))
    {
      i++;
      if (i < argc)
      {
        if (strlen(argv[i]) > MAX_PATH_LEN-1)
        {
          (void)fprintf(stderr,"Path must contain less than %d characters.",
                        MAX_PATH_LEN);
          exit(1);
        }
        (void)strcpy(data_path, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-study"))
    {
      i++;
      if (i < argc)
      {
        if (strlen(argv[i]) > MAX_STUDY_LEN-1)
        {
          (void)fprintf(stderr,"Study name must contain less than %d characters.",
                        MAX_STUDY_LEN);
          exit(1);
        }
        (void)strcpy(study_name, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-debug"))
      GTXClientDebug(1);
    else if (!strcmp(argv[i], "-help"))
    {
      st_usage(argv[0], 1);
      exit(0);
    }
  }

  /*
  ** Connect to the server
  */
  if (GTXClientInitialize(GTX_CLIENT_VERSION))
    goto label_end;
  if (run_server)
  {
    port = 0;
    if (GTXClientRunGTXserver(&port))
      goto label_end;
  }
  if (GTXClientConnect(host, port, data_path))
    goto label_end;

  if (!GTXClientServerHasFeature(GTX_FEATURE_PALETTES, 1) ||
      !GTXClientServerHasFeature(GTX_FEATURE_COLOR_SCALES, 1))
    goto label_end;

  /*
  ** Set current study
  */
  if (GTXClientSetStudy(study_name))
    goto label_end;

  /* List Palettes */
  if (st_list_palettes(0)) goto label_end;

  /* Create our Own Palette */
  (void)strcpy(new_pal_name, "GTXserver Palette");
  if (st_create_palette(new_pal_name)) goto label_end;

  /* List Palettes */
  if (st_list_color_scales()) goto label_end;

  /* Create our own Color Scales */
  if (st_create_manual_color_scale(new_pal_name, "GTXserver Manual Color Scale"))
    goto label_end;
  if (st_create_facies_color_scale("GTXserver Facies Color Scale"))
    goto label_end;

  error = 0;

label_end:

  /*
  ** Disconnect from the server
  */
  GTXClientDisconnect();

  return(error);
}
