/*****************************************************************************
 
            Copyright (c)2005 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/
 
/* $Id: write_files.c 25953 2016-04-07 06:59:39Z foucher $ */
 
#include <GTXClient.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define GTX_STUDY        "GTXserver_Tests"
#define GTX_DIR2D        "write2D"
#define GTX_DIR3D        "write3D"
#define GTX_GRID_FILE    "grid"
#define GTX_POINTS_FILE  "points"
#define GTX_LINES_FILE   "lines"
#define GTX_HPOINTS_FILE "lines_header"
#define GTX_VARIABLE     "variable"
#define GTX_AVARIABLE    "avariable"
#define GTX_MACRO        "macro[xxxxx]"
#define GTX_MACRO_SEL    "macro_hole_sel[xxxxx]"
#define GTX_LID_VARIABLE "HoleId"

#define DEFAULT_GTX_SERVERNAME "localhost"

#define MAX_HOSTNAME_LEN 1024
#define MAX_PATH_LEN     1024

#ifdef winnt
#define GTX_STUDY_PATH "C:\\" GTX_STUDY
#else
#define GTX_STUDY_PATH "/tmp/" GTX_STUDY
#endif

char *GTXvar_types[] =
{
  "Float",
  "Character",
  "X gravity center",
  "Y gravity center",
  "Z gravity center",
  "Macro"
};

/****************************************************************************
**
** FUNCTION: st_usage
**
** PURPOSE:  gives this program usage
**
** IN_ARGS:  exe_name: name of the executable (argv[0])
** IN_ARGS:  help    : 1 to obtain a complete help
**
*****************************************************************************/
static void st_usage(char *exe_name,
                     int help)
{
  (void)printf("Usage: %s [-help] [-host hostname] [-port port_number]\n",
               exe_name);
  (void)printf("          [-path data_path]");

  if (help)
  {
    (void)printf(" -help             : Print this message.\n");
    (void)printf(" -debug            : Print client/server messages.\n");
    (void)printf(" -host hostname    : Server's hostname.\n");
    (void)printf(" -port port_number : Port used between client and server\n");
    (void)printf(" -path data_path   : Initialize connection with given data_path.\n");
    (void)printf("The default is to run a new server on a random port.");
    (void)printf("If -host or -port is specified, a server must have been started first on the given host/port");
  }
}

/****************************************************************************
**
** FUNCTION:    st_is_in_list
**
** DESCRIPTION: check presence of an item in a list
**
** RETURNS:     0 if not there, 1 if present
**
** ARGUMENTS:   nlist: number of elements in the list
** ARGUMENTS:   list:  list of elements
** ARGUMENTS:   item:  element to check existence for
**
*****************************************************************************/
static int st_is_in_list(int    nlist,
                         char **list,
                         char  *item)
{
  int i;
  for (i = 0; i < nlist; i++)
    if (!strcmp(list[i], item))
      return(1);
  return(0);
}

/****************************************************************************
**
** FUNCTION: st_create_grid
**
** PURPOSE:  Creates a grid file
**
** RETURNS:  0 if Ok, 1 on error
**
** IN_ARGS:  dimension: file dimension 2 for 2D, 3 for 3D
**
*****************************************************************************/
static int st_create_grid(int dimension)
{
  int nsample,nx,ny,nz,error;
  double *array;
  char **list;
  int i,j,nb_list;
  int selection_ind[3] = {1,12,50};

  error = 1;
  array = NULL;
  nb_list = 0;
  list = NULL;
  nx = 20;
  ny = 15;
  nz = (dimension == 3)? 30: 1;
  nsample = nx*ny*nz;

  /*
  ** Create a new grid file
  ** if it already exists, it must be deleted before !!
  */
  if (GTXClientGetFileList(&nb_list, &list))
    goto label_end;
  if (st_is_in_list(nb_list, list, GTX_GRID_FILE) &&
      (GTXClientSetFile(GTX_GRID_FILE) || GTXClientDeleteFile()))
    goto label_end;
  if (GTXClientNewGridFile(GTX_GRID_FILE, dimension,
                           1., 2., 3.,
                           4., 5., 6.,
                           nx, ny, nz))
     goto label_end;
  list = GTXClientFreeStringArray(nb_list, list);

  /*
  ** Set current file
  */
  if (GTXClientSetFile(GTX_GRID_FILE))
    goto label_end;
 
  /*
  ** Modify Grid Attributes
  ** We could have created it with the right origin and mesh, it's a test
  ** However the only way to specify the orientation of the grid is
  ** after its creation, like there.
  */
  if (GTXClientGridModifyOrigin(1000., 2000., 3000.))
     goto label_end;
  if (GTXClientGridModifyMesh(100., 200., 300.))
     goto label_end;
  if (GTXClientGridModifyRotation(45., 0., 0.))
     goto label_end;
  
  /*
  ** Create a variable
  */
  if (GTXClientNewDoubleVariable(GTX_VARIABLE, 64,
                                 GTX_FORMAT_TYPE_DECIMAL, 10, 2, " "))
    goto label_end;

  /*
  ** Set the current variable
  */
  if (GTXClientSetVariable(GTX_VARIABLE))
    goto label_end;

  /*
  ** Write data in the variable
  */
  array = (double*)malloc(nsample*sizeof(double));
  if (array == NULL) goto label_end;
  for (i = 0; i < nsample; i++)
    array[i] = (double)(rand() % 1000);

  if (GTXClientWriteDoubleVariable(0, -9999., nsample, array))
    goto label_end;

  if (GTXClientNewDoubleVariable(GTX_MACRO, 32,GTX_FORMAT_TYPE_DECIMAL, 10, 2, " "))
    goto label_end;
  if (GTXClientSetVariable(GTX_MACRO))
    goto label_end;
  if (GTXClientAddMacroIndices(3, selection_ind))
    goto label_end;

  for (i = 0; i<3; i++)
  {
    if (GTXClientSetIndice(selection_ind[i]))
      goto label_end;
    for (j = 0; j<nsample;j++)
      array[j] = selection_ind[i]*1000 + (rand()%1000);
    if (GTXClientWriteDoubleVariable(0, -9999.,nsample, array))
      goto label_end;

  }
  error = 0;
label_end:
  list = GTXClientFreeStringArray(nb_list, list);
  if (array) free(array);
  return(error);
}

/****************************************************************************
**
** FUNCTION: st_create_points
**
** PURPOSE:  Creates a points file
**
** RETURNS:  0 if Ok, 1 on error
**
** IN_ARGS:  dimension: file dimension 2 for 2D, 3 for 3D
**
*****************************************************************************/
static int st_create_points(int dimension)
{
  int nsample,error;
  double *x,*y,*z, *array;
  char **list;
  int i,nb_list;

  error = 1;
  x = y = z = array = NULL;
  nb_list = 0;
  nsample = 50;

  /*
  ** Create a new points file
  */
  x = (double*)malloc(sizeof(double)*nsample);
  y = (double*)malloc(sizeof(double)*nsample);
  if (dimension == 3)
    z = (double*)malloc(sizeof(double)*nsample);
  array = (double*)malloc(sizeof(double)*nsample);

  if (x == NULL || y == NULL || array == NULL) goto label_end;

  for (i = 0; i < nsample; i++)
  {
    x[i] = (double)((rand() % 3500) - 1000);
    y[i] = (double)((rand() % 3500) + 2000);
    if (z)
      z[i] = (double)((rand() % 9000) + 3000);
    array[i] = (double)(rand() % 1000);
  }

  /*
  ** if it already exists, it must be deleted before !!
  */
  if (GTXClientGetFileList(&nb_list, &list))
    goto label_end;
  if (st_is_in_list(nb_list, list, GTX_POINTS_FILE) &&
      (GTXClientSetFile(GTX_POINTS_FILE) || GTXClientDeleteFile()))
    goto label_end;
  /* The file will have the right dimension according to the fact that
     z variable is NULL or not. */
  if (GTXClientNewPointsFile(GTX_POINTS_FILE, nsample, x, y, z))
     goto label_end;
  if (GTXClientSetFile(GTX_POINTS_FILE))
    goto label_end;
  if (GTXClientPointsFileAppend(nsample,x, y, z))
     goto label_end;
  list = GTXClientFreeStringArray(nb_list, list);

  /*
  ** Set current file
  */
  if (GTXClientSetFile(GTX_POINTS_FILE))
    goto label_end;
 
  /*
  ** Create a variable
  */
  if (GTXClientNewDoubleVariable(GTX_VARIABLE, 64, GTX_FORMAT_TYPE_DECIMAL,
                                 10, 2, " "))
    goto label_end;

  /*
  ** Set the current variable
  */
  if (GTXClientSetVariable(GTX_VARIABLE))
    goto label_end;

  /*
  ** Write only a subpart of it
  */
  if (GTXClientWriteDoubleVariableSub(5, nsample+4, 0, -9999., nsample, array))
    goto label_end;

  error = 0;
label_end:
  if (x) free(x);
  if (y) free(y);
  if (z) free(z);
  if (array) free(array);
  list = GTXClientFreeStringArray(nb_list, list);
  return(error);
}

/****************************************************************************
**
** FUNCTION: st_create_lines
**
** PURPOSE:  Creates a points + lines file
**
** RETURNS:  0 if Ok, 1 on error
**
** IN_ARGS:  dimension: file dimension 2 for 2D, 3 for 3D
**
** REMARKS:  We create both a points and a lines file but a lines file can
** REMARKS:  exist alone.
**
*****************************************************************************/
static int st_create_lines(int dimension)
{
  int error;
  double *xhead,*yhead,*zhead;
  double *x,*y,*z, *array,test_val;
  unsigned char *sel_array, *sel_array_client;
  char **hole_id,**list;
  int i,j,k,ind,nb_list;
  int nlines = 3;
  int nsample[3] = {5,10,20};
  int max_nsample, total_nsample;
  gtx_long retval_nbsample;
  int selection_ind[3] = {1,2,3};
  double *ret_data;
  int ret_nsample;
  GTXFileInfo file_info;

  error = 1;
  x = y = z = array = NULL;
  xhead = yhead = zhead = NULL;
  hole_id = NULL;
  nb_list = 0;
  max_nsample = 0;
  total_nsample = 0;
  ret_data = NULL;
  sel_array = NULL;
  sel_array_client = NULL;
  for (i = 0; i < nlines; i++)
  {
    total_nsample += nsample[i];
    if (nsample[i] > max_nsample)
      max_nsample = nsample[i];
  }

    
 /* Allocations, there are no checks here. It's dangerous but that's only
     for the test...*/
  xhead = (double*)malloc(sizeof(double)*nlines);
  yhead = (double*)malloc(sizeof(double)*nlines);
  if (dimension == 3)
    zhead = (double*)malloc(sizeof(double)*nlines);

  x = (double*)malloc(sizeof(double)*max_nsample);
  y = (double*)malloc(sizeof(double)*max_nsample);
  if (dimension == 3)
    z = (double*)malloc(sizeof(double)*max_nsample);
  array = (double*)malloc(sizeof(double)*max_nsample);
  sel_array = (unsigned char*)malloc(sizeof(unsigned char)*total_nsample);
  hole_id = (char**)malloc(sizeof(char*)*nlines);
  for (i = 0; i < nlines; i++)
    hole_id[i] = (char*)malloc(sizeof(char)*7);

  if (x == NULL || y == NULL || array == NULL) goto label_end;

  /*
  ** Create Head samples somewhere around the top of XOY section */
  for (i = 0; i < nlines; i++)
  {
    xhead[i] = (double)((rand() % 3500) - 1000);
    yhead[i] = (double)((rand() % 1000) + 4500);
    if (zhead)
      zhead[i] = (double)((rand() % 1000) + 11000);
    (void)sprintf(hole_id[i], "Hole %d", i+1);
  }

  /*
  ** Create a new header points file
  ** if it already exists, it must be deleted before !!
  */
  if (GTXClientGetFileList(&nb_list, &list))
    goto label_end;
  if (st_is_in_list(nb_list, list, GTX_HPOINTS_FILE) &&
      (GTXClientSetFile(GTX_HPOINTS_FILE) || GTXClientDeleteFile()))
    goto label_end;
  /* The file will have the right dimension according to the fact that
     z variable is NULL or not. */
  if (GTXClientNewPointsFile(GTX_HPOINTS_FILE, nlines, xhead, yhead, zhead))
     goto label_end;

  /*
  ** Add it a hole id variable
  */
  if (GTXClientSetFile(GTX_HPOINTS_FILE) ||
      GTXClientNewCharVariable(GTX_LID_VARIABLE, 6) ||
      GTXClientSetVariable(GTX_LID_VARIABLE) ||
      GTXClientWriteCharVariable(0, "", nlines,
                                 (const char * const *)hole_id))
    goto label_end;

  /*
  ** Create a new lines file
  ** if it already exists, it must be deleted before !!
  */
  if (st_is_in_list(nb_list, list, GTX_LINES_FILE) &&
      (GTXClientSetFile(GTX_LINES_FILE) || GTXClientDeleteFile()))
    goto label_end;
  if (GTXClientNewLinesFile(GTX_LINES_FILE, dimension))
     goto label_end;
  list = GTXClientFreeStringArray(nb_list, list);

  /*
  ** Set current file
  */
  if (GTXClientSetFile(GTX_LINES_FILE))
    goto label_end;
 
  for (i = 0; i < nlines; i++)
  {
    for (j = 0; j < nsample[i]; j++)
    {
      x[j] = xhead[i];
      if (dimension == 2)
        y[j] = ((j == 0)? yhead[i]: y[j-1])- (double)(rand() % 250);
      else
      {
        y[j] = yhead[i];
        z[j] = ((j == 0)? zhead[i]: z[j-1]) - (double)(rand() % 900);
      }
    }

    if (GTXClientLinesFileAddCoreLine(
          nsample[i], xhead[i], yhead[i], (zhead != NULL)? zhead[i]: 0.,
          x, y, z))
      goto label_end;
  }

  /*
  ** The link between files must be created afterwards because
  ** appending samples would delete it elsewhere.
  */
  if (GTXClientLinesFileCreateLink(GTX_HPOINTS_FILE))
    goto label_end;

  /*
  ** Create a variable
  */
  if (GTXClientNewDoubleVariable(GTX_VARIABLE, 64, GTX_FORMAT_TYPE_DECIMAL,
                                 10, 6, " "))
    goto label_end;

  /*
  ** Set the current variable
  */
  if (GTXClientSetVariable(GTX_VARIABLE))
    goto label_end;

  /* Write the variable contents.
  ** Please note that we are writing here using GTXClientWriteLineDoubleVariable
  ** because it is easier when you have only a few lines to write but it can be
  ** time consuming if you might have a lot of lines to write and should in this
  ** case consider using GTXClientWriteDoubleVariableSub.
  */
  for (i = 0; i < nlines; i++)
  {
    for (k = 0; k < nsample[i]; k++)
      array[k] = (rand() % 1000) / 1000.;
    if (GTXClientWriteLineDoubleVariable(i+1,-9999.,nsample[i],array)) 
      goto label_end; 
  }  
  if (GTXClientGetFileInfo(&file_info))
    goto label_end;

  /* read back for the example */
  for (i = 0; i < file_info.item_number; i++)
  {
    if (GTXClientReadLineDoubleVariable(i+1, &test_val, &ret_nsample, &ret_data))
      goto label_end;
    ret_data = GTXClientFreePointer(ret_data);
  }

  /*
  ** Create an alpha variable
  */
  if (GTXClientNewCharVariable(GTX_AVARIABLE, 10))
    goto label_end;

  /*
  ** Set the alpha variable as current and write to it
  */
  if (GTXClientSetVariable(GTX_AVARIABLE))
    goto label_end;

  for (i = 0; i < nlines; i++)
  {
    char **carray = NULL;
    carray = (char**)malloc(sizeof(char*)*nsample[i]);
    if (carray == NULL)
      goto label_end;
    for (k = 0; k < nsample[i]; k++)
      carray[k] = NULL;
    for (k = 0; k < nsample[i]; k++)
    {
      carray[k] = (char*)malloc(sizeof(char)*10);
      if (carray[k] == NULL) goto label_end;
      sprintf(carray[k], "%d_%d", i+1, k+1);
    }
    if (GTXClientWriteLineCharVariable(i+1,"",nsample[i],
      (const char * const *)carray)) 
      goto label_end;
    for (k = 0; k < nsample[i]; k++)
      free(carray[k]);
    free(carray);
  }  
  if (GTXClientGetFileInfo(&file_info))
    goto label_end;

  /* read back for the example */
  for (i = 0; i < file_info.item_number; i++)
  {
    char **cret_data = NULL;
    char *ctest_val = NULL;
    if (GTXClientReadLineCharVariable(i+1, &ctest_val, &ret_nsample, &cret_data))
      goto label_end;
    cret_data = GTXClientFreeStringArray(ret_nsample, cret_data);
    ctest_val = GTXClientFreePointer(ctest_val);
  }
  /*
  ** Create a macro variable with alpha indices
  ** It is a macro-selection with the hole-id being used as the alpha index
  */

  if (GTXClientNewDoubleVariable(GTX_MACRO_SEL, 1,GTX_FORMAT_TYPE_INTEGER, 3, 0, " "))
    goto label_end;
  if (GTXClientSetVariable(GTX_MACRO_SEL))
    goto label_end;

  if (GTXClientAddMacroIndicesWithAlpha(nlines, selection_ind, (const char* const *) hole_id))
    goto label_end;

  for (ind = 0; ind < nlines; ind++)
  {
    if (GTXClientSetAlphaIndice(hole_id[ind]))
      goto label_end;

    for (i = 0, k = 0; i < nlines; i++)
    {
      for (j = 0; j < nsample[i]; j++, k++)
        sel_array[k] = (ind == i);
    }

    if (GTXClientWriteSelectionVariable(total_nsample, sel_array))
      goto label_end;

    /* Just to test the selection read function*/
    if (GTXClientReadSelectionVariable(&retval_nbsample, &sel_array_client))
      goto label_end;
  }

  /*
  ** Set the hole id variable (in the Points File) as the Line Name
  */
  if (GTXClientSetFile(GTX_HPOINTS_FILE))
    goto label_end;
  if (GTXClientSetVariable(GTX_LID_VARIABLE))
    goto label_end;
  if (GTXClientSetVariableAsLineName())
    goto label_end;

  error = 0;
label_end:
  if (xhead) free(xhead);
  if (yhead) free(yhead);
  if (zhead) free(zhead);
  if (hole_id)
  {
    for (i = 0; i < nlines; i++)
      if (hole_id[i]) free(hole_id[i]);
    free(hole_id);
  }
  if (x) free(x);
  if (y) free(y);
  if (z) free(z);
  if (array) free(array);
  if (sel_array) free(sel_array);
  if (sel_array_client) GTXClientFreePointer(sel_array_client);
  if (ret_data)
    ret_data = GTXClientFreePointer(ret_data);
  list = GTXClientFreeStringArray(nb_list, list);
  return(error);
}

/****************************************************************************
**
** FUNCTION:    main
**
** DESCRIPTION: client's main function
**
** RETURNS:     Nothing
**
** ARGUMENTS:   int argv, char *argv : program arguments
**
*****************************************************************************/
int main(int argc,
         char *argv[])
{
  char *tmp_str, host[MAX_HOSTNAME_LEN];
  unsigned short port;
  int i,dim,error,run_server;
  char data_path[MAX_PATH_LEN];
  char *dir;

  int nb_list = 0;
  char **list = NULL;

  char **vars = NULL;
  int  *indices = NULL;

  error = 1;
  run_server = 1;
  port = GTXClientGetDefaultPort();

  (void)strcpy(host, DEFAULT_GTX_SERVERNAME);
  (void)strcpy(data_path, "");

  for (i = 1; i < argc; i++)
  {
    if (!strcmp(argv[i], "-port"))
    {
      i++;
      if (i < argc)
      {
        run_server = 0;
        port = (unsigned short)strtol(argv[i], &tmp_str, 10);
        if (port == 0 && tmp_str == argv[i])
        {
          (void)fprintf(stderr,"Invalid Port Number.");
          exit(1);
        }
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-host"))
    {
      i++;
      if (i<argc)
      {
        run_server = 0;
        if (strlen(argv[i]) > MAX_HOSTNAME_LEN-1)
        {
          (void)fprintf(stderr,"Host Name must contain less than %d characters.",
                        MAX_HOSTNAME_LEN);
          exit(1);
        }
        (void)strcpy(host, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-path"))
    {
      i++;
      if (i < argc)
      {
        if (strlen(argv[i]) > MAX_PATH_LEN-1)
        {
          (void)fprintf(stderr,"Path must contain less than %d characters.",
                        MAX_PATH_LEN);
          exit(1);
        }
        (void)strcpy(data_path, argv[i]);
      }
      else
      {
        st_usage(argv[0], 0);
        exit(1);
      }
    }
    else if (!strcmp(argv[i], "-debug"))
      GTXClientDebug(1);
    else if (!strcmp(argv[i], "-help"))
    {
      st_usage(argv[0], 1);
      exit(0);
    }
  }

  /*
  ** Connect to the server
  */
  if (GTXClientInitialize(GTX_CLIENT_VERSION))
    goto label_end;
  if (run_server)
  {
    port = 0;
    if (GTXClientRunGTXserver(&port))
      goto label_end;
  }
  if (GTXClientConnect(host, port, data_path))
    goto label_end;

  /*
  ** Set current unit
  */
  if (GTXClientSetUnit("m"))
    goto label_end;

  /*
  ** Set current study (create it if necessary and set)
  */
  if (GTXClientGetStudyList(&nb_list, &list))
    goto label_end;
  if (!st_is_in_list(nb_list, list, GTX_STUDY) &&
      GTXClientNewStudy(GTX_STUDY, GTX_STUDY_PATH))
    goto label_end;
  if (GTXClientSetStudy(GTX_STUDY))
    goto label_end;
  list = GTXClientFreeStringArray(nb_list, list);

  /* Create a 2D set and a 3D set of data */
  for (dim = 2; dim <= 3; dim++)
  {
    dir = (dim == 2)? GTX_DIR2D: GTX_DIR3D;

    /*
    ** Set current directory (create it if necessary and set)
    */
    if (GTXClientGetDirectoryList(&nb_list, &list))
      goto label_end;
    if (!st_is_in_list(nb_list, list, dir) &&
        GTXClientNewDirectory(dir))
      goto label_end;
    if (GTXClientSetDirectory(dir))
      goto label_end;
    list = GTXClientFreeStringArray(nb_list, list);

    if (st_create_grid(dim)) goto label_end;
    if (st_create_points(dim)) goto label_end;
    if (st_create_lines(dim)) goto label_end;
  }

  error = 0;

label_end:
  list = GTXClientFreeStringArray(nb_list, list);

  /*
  ** Disconnect from the server
  */
  GTXClientDisconnect();

  return(error);
}
