/*****************************************************************************

            Copyright (c)2008 Geovariances, Avon, France.

    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.

    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).

            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE

             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr

                        All Rights Reserved

*****************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <sstream>
#include <iomanip>
#include <iostream>
#include <GTXFileInfo.hpp>
#include <GTXFaultInfo.hpp>
#include <GTXVariableInfo.hpp>
#include <GTXDoubleData.hpp>

#define MAX_HOSTNAME_LEN 1024
#define MAX_PATH_LEN     1024

class Arguments
{
public:
  bool run_server;
  char host[MAX_HOSTNAME_LEN];
  unsigned short port;
  char data_path[MAX_PATH_LEN];
  
  Arguments()
  {
    run_server = 1;
    (void)strcpy(host, "localhost");
    (void)strcpy(data_path, "");
    port = GTXClientGetDefaultPort();
  }

  static void Usage(char *exe_name, bool help)
  {
    (void)printf("Usage: %s [-help] [-host hostname] [-port port_number]\n",
                 exe_name);
    (void)printf("          [-path data_path]");
    
    if (help)
    {
      (void)printf(" -help             : Print this message.\n");
      (void)printf(" -debug            : Print client/server messages.\n");
      (void)printf(" -host hostname    : Server's hostname.\n");
      (void)printf(" -port port_number : Port used between client and server\n");
      (void)printf(" -path data_path   : Initialize connection with given data_path.\n");
      (void)printf("The default is to run a new server on a random port.");
      (void)printf("If -host or -port is specified, a server must have been started first on the given host/port");
    }
  }
  
  void ParseCommandLine(int argc, char *argv[])
  {
    int i;
    char *tmp_str;

    for (i=1; i<argc; i++)
    {
      if (!strcmp(argv[i], "-port"))
      {
        i++;
        if (i<argc)
        {
          run_server = 0;
          port = (unsigned short)strtol(argv[i], &tmp_str, 10);
          if (port == 0 && tmp_str == argv[i])
          {
            (void)fprintf(stderr,"Invalid Port Number.");
            exit(1);
          }
        }
        else
        {
          Usage(argv[0], 0);
          exit(1);
        }
      }
      else if (!strcmp(argv[i], "-host"))
      {
        i++;
        if (i<argc)
        {
          run_server = 0;
          if (strlen(argv[i]) > MAX_HOSTNAME_LEN-1)
          {
            (void)fprintf(stderr,"Host Name must contain less than %d characters.",
                          MAX_HOSTNAME_LEN);
            exit(1);
          }
          (void)strcpy(host, argv[i]);
        }
        else
        {
          Usage(argv[0], 0);
          exit(1);
        }
      }
      else if (!strcmp(argv[i], "-path"))
      {
        i++;
        if (i<argc)
        {
          if (strlen(argv[i]) > MAX_PATH_LEN-1)
          {
            (void)fprintf(stderr,"Path must contain less than %d characters.",
                          MAX_PATH_LEN);
            exit(1);
          }
          (void)strcpy(data_path, argv[i]);
        }
        else
        {
          Usage(argv[0], 0);
          exit(1);
        }
      }
      else if (!strcmp(argv[i], "-debug"))
        GTXClientDebug(1);
      else if (!strcmp(argv[i], "-help"))
      {
        Usage(argv[0], 1);
        exit(0);
      }
    }
  }
};

//! Class containing variable statistics
class VarStats
{
public:
  //! constructor initializing stats
  VarStats()
  {
    min = 1.e300;
    max = -1.e300;
    mean = 0.;
    nsamples = 0;
  }
  //!  number of defined samples
  int nsamples;

  //! Maximum taken by the variable on defined samples
  double max;

  //! Minimum taken by the variable on defined samples
  double min;

  //! Mean of the defined samples
  double mean;
};

//! Class used to dump information on Files and Variables
class InfoDump
{
public:
  //! \brief Dump file information into a string
  //! \param client current GTXclient session
  //! \param nindent number of spaces to prefix each line
  //! \return composed string containing file information
  static std::string FileInfo(GTXClient *client,
                              int nindent)
  {
    std::string prefix;
    std::stringstream prefixss;
    std::stringstream buffer;
    for (int i = 0; i < nindent; i++) prefixss << " ";
    prefix = prefixss.str();

    GTXFileInfo finfo = client->GetFileInfo();

    const char *file_type;
    switch (finfo.GetFileType())
    {
      case GTXFileInfo::FILE_TYPE_POINTS:        file_type = "Points";  break;
      case GTXFileInfo::FILE_TYPE_GRAVITY_LINES: file_type = "Gravity Lines"; break;
      case GTXFileInfo::FILE_TYPE_CORE_LINES:    file_type = "Core Lines"; break;
      case GTXFileInfo::FILE_TYPE_GRID:          file_type = "Grid"; break;
      default: file_type = "Unknown"; break;
    }
    buffer << prefix << file_type << " File" << std::endl;
    buffer << prefix << "Dimension = " << finfo.GetDimension() << "D" << std::endl;
    buffer << prefix << "Number of samples = " << finfo.GetSampleNumber() << std::endl;
    if (finfo.GetFileType() == GTXFileInfo::FILE_TYPE_GRAVITY_LINES ||
        finfo.GetFileType() == GTXFileInfo::FILE_TYPE_CORE_LINES)
    {
      buffer << prefix << "Number of lines = " << finfo.GetItemNumber() << std::endl;
    }
    else if (finfo.GetFileType() == GTXFileInfo::FILE_TYPE_GRID)
    {
      if (finfo.GetDimension() == 2)
      {
        buffer << prefix << "Number of nodes = (" << finfo.GetGridNX() << "," << finfo.GetGridNY() << ")" << std::endl;
        buffer << prefix << "Origin          = (" << finfo.GetGridX0() << "," << finfo.GetGridY0() << ")" << std::endl;
        buffer << prefix << "Mesh dimensions = (" << finfo.GetGridDX() << "," << finfo.GetGridDY() << ")" << std::endl;
        if (finfo.GetGridRotatedFlag())
          buffer << prefix << "Rotation angle = " << finfo.GetGridAngleAroundZ() << std::endl;
        else
          buffer << prefix << "No rotation" << std::endl;
      }
      else
      {
        buffer << prefix << "Number of nodes = ("
               << finfo.GetGridNX() << ","
               << finfo.GetGridNY() << ","
               << finfo.GetGridNZ() << ")" << std::endl;
        buffer << prefix << "Origin          = ("
               << finfo.GetGridX0() << ","
               << finfo.GetGridY0() << ","
               << finfo.GetGridZ0() << ")" << std::endl;
        buffer << prefix << "Mesh dimensions = ("
               << finfo.GetGridDX() << ","
               << finfo.GetGridDY() << ","
               << finfo.GetGridDZ() << ")" << std::endl;
        if (finfo.GetGridRotatedFlag())
        {
          buffer << prefix << "Rotation angle around Z = " << finfo.GetGridAngleAroundZ() << std::endl;
          buffer << prefix << "Rotation angle around Y = " << finfo.GetGridAngleAroundY() << std::endl;
          buffer << prefix << "Rotation angle around X = " << finfo.GetGridAngleAroundX() << std::endl;
        }
        else
          buffer << prefix << "No rotation" << std::endl;
      } // end of 3D
    } // end of if Grid Type

    const char *comment = client->GetFileComment();
    if (strlen(comment) != 0)
      buffer << prefix << "Comment: " << comment << std::endl;

    // special variables information
    buffer << prefix << "Sample Number variable: " << finfo.GetSampleNumberVariableName() << std::endl;
    if (finfo.GetFileType() == GTXFileInfo::FILE_TYPE_GRAVITY_LINES ||
        finfo.GetFileType() == GTXFileInfo::FILE_TYPE_CORE_LINES)
    {
      buffer << prefix << "Line Number variable: " << finfo.GetLineNumberVariableName() << std::endl;
      buffer << prefix << "Relative Number variable: " << finfo.GetRelativeNumberVariableName() << std::endl;
    }
    if (finfo.GetFileType() == GTXFileInfo::FILE_TYPE_POINTS &&
        strcmp(finfo.GetLinkedFileName(), ""))
    {
      buffer << prefix << "Line Name variable: " << finfo.GetLineNameVariableName() << std::endl;
    }
    buffer << prefix << "X Gravity Center variable: " << finfo.GetXCoordinateVariableName() << std::endl;
    buffer << prefix << "Y Gravity Center variable: " << finfo.GetYCoordinateVariableName() << std::endl;
    if (finfo.GetDimension() == 3)
      buffer << prefix << "Z Gravity Center variable: " << finfo.GetZCoordinateVariableName() << std::endl;
    
    if (finfo.GetFileType() == GTXFileInfo::FILE_TYPE_CORE_LINES)
    {
      buffer << prefix << "X Core Begin variable: " << finfo.GetXCoreBeginVariableName() << std::endl;
      buffer << prefix << "Y Core Begin variable: " << finfo.GetYCoreBeginVariableName() << std::endl;
      if (finfo.GetDimension() == 3)                             
        buffer << prefix << "Z Core Begin variable: " << finfo.GetZCoreBeginVariableName() << std::endl;

      buffer << prefix << "X Core End variable: " << finfo.GetXCoreEndVariableName() << std::endl;
      buffer << prefix << "Y Core End variable: " << finfo.GetYCoreEndVariableName() << std::endl;
      if (finfo.GetDimension() == 3)                             
        buffer << prefix << "Z Core End variable: " << finfo.GetZCoreEndVariableName() << std::endl;
    }

    if (finfo.GetFaultedFlag())
    {
      GTXFaultInfo fault_info = finfo.GetFaultInfo();
      buffer << prefix << "\nFaulted File:" << std::endl;
      const char *flag = fault_info.GetFaults2DFlag() ? "Y": "N";
      buffer << prefix << "  2D Faults: " << flag << std::endl;
      flag = fault_info.GetAuxiliaryVarUsedFlag() ? "Y": "N";
      buffer << prefix << "  Auxiliary Variable Used: " << flag << std::endl;
      buffer << prefix << "  Number of faults: " <<
        fault_info.GetFaultsNumber() << std::endl;
      buffer << prefix << "  Number of segments: " <<
        fault_info.GetSegmentsNumber() << std::endl;
      buffer << prefix << "  Minimum Used Priority: " <<
        fault_info.GetMinimumPriority() << std::endl;
      buffer << prefix << "  Maximum Used Priority: " <<
        fault_info.GetMaximumPriority() << std::endl;
      buffer << prefix << "  Authorized Priority: " <<
        fault_info.GetAuthorizedPriority() << std::endl;

      // Exemple of Faults Reading
      /*GTXFaultSystem faultSystem =
        client->ReadFaults(fault_info.auth_priority);*/
    }
    return buffer.str();
  }

  //! \brief  Dump variable information/statistics into a string
  //! \param  client current GTXclient session
  //! \param  stats_mode 0 no stats, 1 stats if not macro,
  //          2 always stat (macro index must have been set)
  //! \param  nindent number of spaces to prefix each line
  //! \return composed string containing variable information
  static std::string VariableInfo(GTXClient *client,
                                  int stats_mode,
                                  int nindent)
  {
    std::string prefix;
    std::stringstream prefixss;
    std::stringstream buffer;
    for (int i = 0; i < nindent; i++) prefixss << " ";
    prefix = prefixss.str();

    const char *var_type;
    GTXVariableInfo vinfo = client->GetVariableInfo();
    switch (vinfo.GetVariableType())
    {
      case GTXVariableInfo::VAR_TYPE_FLOAT: var_type = "Float"; break;
      case GTXVariableInfo::VAR_TYPE_CHAR: var_type = "Character"; break;
      case GTXVariableInfo::VAR_TYPE_XG: var_type = "X Gravity Center"; break;
      case GTXVariableInfo::VAR_TYPE_YG: var_type = "Y Gravity Center"; break;
      case GTXVariableInfo::VAR_TYPE_ZG: var_type = "Z Gravity Center"; break;
      case GTXVariableInfo::VAR_TYPE_MACRO: var_type = "Macro"; break;
      default: var_type = "Unknown"; break;
    }
    
    buffer << prefix << "Type: " << var_type << std::endl;

    if (vinfo.GetVariableType() == GTXVariableInfo::VAR_TYPE_CHAR)
    {
      buffer << prefix << "Alpha Length: " << vinfo.GetAlphaLength() << std::endl;
    }
    else
    {
      if (vinfo.GetBitLength() == 1)
        buffer << prefix << "Precision: 1 bit (selection)" << std::endl;
      else
      {
        buffer << prefix << "Precision: " << vinfo.GetBitLength() << " bits" << std::endl;
        if (!vinfo.GetImplicitFlag())
          buffer << prefix << "Unit: " << vinfo.GetUnit() << std::endl;
      }
    }
    const char *comment = client->GetVariableComment();
    if (strlen(comment) != 0)
      buffer << prefix << "Comment: " << comment << std::endl;
    if (vinfo.GetVariableType() == GTXVariableInfo::VAR_TYPE_MACRO)
    {
      GTXIntArray indices = client->GetMacroIndices();
      GTXStringArray listMacroAlpha =  client->GetMacroAlphaIndices();

        buffer << prefix << "Number of Indices: " << indices.GetCount() << std::endl;

      if (indices.GetCount() != 0)
      {
        for (int m=0; m<indices.GetCount(); m++){
          if(listMacroAlpha.GetCount())
            buffer << prefix  <<std::setw(5) << std::setfill('0') << indices.GetValue(m) << "\t" << listMacroAlpha.GetValue(m) << std::endl;
          else
            buffer << prefix  <<std::setw(5) << std::setfill('0') << indices.GetValue(m) << std::endl;
        }
      }
    }
    if (((vinfo.GetVariableType() == GTXVariableInfo::VAR_TYPE_FLOAT ||
          vinfo.GetVariableType() == GTXVariableInfo::VAR_TYPE_XG ||
          vinfo.GetVariableType() == GTXVariableInfo::VAR_TYPE_YG ||
          vinfo.GetVariableType() == GTXVariableInfo::VAR_TYPE_ZG) && stats_mode != 0) ||
        (vinfo.GetVariableType() == GTXVariableInfo::VAR_TYPE_MACRO && stats_mode == 2))
    {
      VarStats stats = statOnVariable(client);
      buffer << prefix << "# Defined: " << stats.nsamples << std::endl;
      buffer << prefix << "Minimum: " << stats.min << std::endl;
      buffer << prefix << "Maximum: " << stats.max << std::endl;
      buffer << prefix << "Mean: " << stats.mean/stats.nsamples << std::endl;
    }
    return buffer.str();
  }

  //! \brief Read variable and computes its stats
  //! \param session current GTXclient session
  //! \return computed variable statistics
  static VarStats statOnVariable(GTXClient *session)
  {
    VarStats stats;
    GTXDoubleData vdouble = session->ReadDoubleVariable(false);
    const double *vals = vdouble.GetValues();
    for (int n=0; n<vdouble.GetCount(); n++)
    {
      double val = vals[n];
      if (val != vdouble.GetUndefinedValue())
      {
        stats.mean += val;
        stats.min = ((val < stats.min) ? val : stats.min);
        stats.max = ((val > stats.max) ? val : stats.max);
        stats.nsamples++;
      }
    }
    return stats;
  }
};
