#include <GTXClient.hpp>
#include <GTXError.hpp>
#include <stdio.h>
#include <GTXWireframeSystem.hpp>

#define GTX_STUDY      "GTXserver_Tests"
#define GTX_DIR        "Wireframes Dir"
#define GTX_FILE       "File"
#define GTX_VARIABLE   "Var"

#include "utils.hpp"
#ifdef winnt
#define GTX_STUDY_PATH "C:\\" GTX_STUDY
#else
#define GTX_STUDY_PATH "/tmp/" GTX_STUDY
#endif

int main(int argc,
         char *argv[])
{
  Arguments args;
  args.ParseCommandLine(argc, argv);

  GTXClient *client = new GTXClient();

  try
  {
    /*
    ** Connect to the server
    */
    if (args.run_server)
    {
      args.port = 0;
      args.port = client->RunGTXserver(args.port);
    }
    client->Connect(args.host, args.port, args.data_path);

    /*Set Study,directory....*/

    bool isStudy = false;
    GTXStringArray list =  client->GetStudyList();
    for (int i = 0; i < list.GetCount(); i++){
      if (strcmp(list.GetValue(i), GTX_STUDY)==0)
        isStudy = true;
    }
    if (!isStudy)
      client->NewStudy(GTX_STUDY, GTX_STUDY_PATH);
    client->SetStudy(GTX_STUDY);

    //search is the Directory existing, else create it and set it
    if (!client->DirectoryExists(GTX_DIR))
      client->NewDirectory(GTX_DIR);
    client->SetDirectory(GTX_DIR);

    // Build a cube oject
    GTXWireframeObject obj = GTXWireframeObject();
    for (int ix = 0; ix < 2; ix++)
      for (int iy = 0; iy < 2; iy++)
        for (int iz = 0; iz < 2; iz++)
          obj.AddVertex(ix*10.,iy*10,iz*10.);
    obj.AddTriangle(0,1,4);
    obj.AddTriangle(4,1,5);
    obj.AddTriangle(4,5,6);
    obj.AddTriangle(6,5,7);
    obj.AddTriangle(2,6,7);
    obj.AddTriangle(7,2,3);
    obj.AddTriangle(0,2,3);
    obj.AddTriangle(0,3,1);
    obj.AddTriangle(5,1,7);
    obj.AddTriangle(7,1,3);
    obj.AddTriangle(6,0,4);
    obj.AddTriangle(0,6,2);
    
    GTXWireframe wire = GTXWireframe();
    wire.SetName("Cube");
    wire.SetColorRGB(255,0,0);
    wire.AddObject(obj);

    GTXWireframeSystem wfsys = GTXWireframeSystem();
    wfsys.AddWireframe(wire);
        
    // Build a surface
    obj = GTXWireframeObject();
    int nx = 30;
    int ny = 10;
    for (int ix = 0; ix < nx; ix++)
      for (int iy = 0; iy < ny; iy++)
      {
        /* going straight from 15. to 15. + 10. in X, 0 to 10. in Y.
        ** and randomly between 4.5 and 5.5 in Z. */
        obj.AddVertex((double)ix/(nx-1)*10.+ 15.,
                      (double)iy/(ny-1)*10.,
                      4.5 + ((double)rand())/RAND_MAX * 1.);
        if (ix > 0 && iy > 0)
        {
          obj.AddTriangle((ix-1)*ny+(iy-1),(ix)*ny+(iy-1),(ix)*ny+(iy));
          obj.AddTriangle((ix-1)*ny+(iy-1),(ix)*ny+(iy),(ix-1)*ny+(iy));
        }
      }
    wire = GTXWireframe();
    wire.SetName("Surface");
    wire.SetColorRGB(0,0,255);
    wire.AddObject(obj);
    wfsys.AddWireframe(wire);

    //Add to the study
    if (client->FileExists(GTX_FILE)){
      client->SetFile(GTX_FILE);
      client->DeleteFile();
    }
    client->NewWireframesFile(GTX_FILE,wfsys);
      
    //Read Wireframe System
    client->SetFile(GTX_FILE);
    GTXWireframeSystem retwfsys = client->ReadWireframes();

    //Display information about the WireframeSystem
    printf("WireframesNumber :%d\n",retwfsys.GetWireframesNumber());
    for (int i = 0; i < retwfsys.GetWireframesNumber(); i++)
    {
      GTXWireframe wire = retwfsys.GetWireframe(i);
      printf("\n\tName :%s\n" , wire.GetName());
      printf("\tColor :[%d,%d,%d]\n" ,
             wire.GetColorR(), wire.GetColorG(), wire.GetColorB());
      printf("\tObjectsNumber :%d\n" , wire.GetObjectsNumber());

      for (int j = 0; j < wire.GetObjectsNumber(); j++)
      {
        GTXWireframeObject obj = wire.GetWireframeObject(j);
        printf("\tVerticesNumber :%d\n", obj.GetVerticesNumber());
        for (int k = 0; k < obj.GetVerticesNumber(); k++)
          printf("\t %-4d - X=%g, Y=%g, Z=%g\n", k,
                 obj.GetVertexX(k), obj.GetVertexY(k), obj.GetVertexZ(k));
        printf("\tTrianglesNumber :%d\n", obj.GetTrianglesNumber());
        for (int k = 0; k < obj.GetTrianglesNumber(); k++)
          printf("\t - Triangle Vertices(%d,%d,%d)\n",
                 obj.GetTrianglePt1(k), obj.GetTrianglePt2(k), obj.GetTrianglePt3(k));
      }
    }
    client->Disconnect();
  } catch (GTXError e)
  {
    fprintf(stderr, "GTXClient returned an error:\n  %s\n",
            e.GetMessage());
  }
  return (0);
}
