/*****************************************************************************
 
            Copyright (c)2005 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/
 
/* $Id: write_files.c 11791 2008-02-20 12:31:47Z marin $ */
 
#include <GTXClient.hpp>
#include <GTXCharData.hpp>
#include <GTXDoubleData.hpp>
#include <GTXVariableFormat.hpp>
#include <GTXError.hpp>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "utils.hpp"

#define GTX_STUDY        "GTXserver_Tests"
#define GTX_DIR2D        "write2D"
#define GTX_DIR3D        "write3D"
#define GTX_GRID_FILE    "grid"
#define GTX_POINTS_FILE  "points"
#define GTX_LINES_FILE   "lines"
#define GTX_HPOINTS_FILE "lines_header"
#define GTX_VARIABLE     "variable"
#define GTX_AVARIABLE    "avariable"
#define GTX_MACRO        "macro[xxxxx]"
#define GTX_MACRO_SEL    "macro_hole_sel[xxxxx]"
#define GTX_LID_VARIABLE "HoleId"

#define DEFAULT_GTX_SERVERNAME "localhost"

#define MAX_HOSTNAME_LEN 1024
#define MAX_PATH_LEN     1024

#ifdef winnt
#define GTX_STUDY_PATH "C:\\" GTX_STUDY
#else
#define GTX_STUDY_PATH "/tmp/" GTX_STUDY
#endif

const char *GTXvar_types[] =
{
  "Float",
  "Character",
  "X gravity center",
  "Y gravity center",
  "Z gravity center",
  "Macro"
};
GTXClient *client = new GTXClient();
GTXVariableFormat format;
GTXDoubleData *doubledata;
GTXCharData *chardata;

/****************************************************************************
**
** FUNCTION: st_usage
**
** PURPOSE:  gives this program usage
**
** IN_ARGS:  exe_name: name of the executable (argv[0])
** IN_ARGS:  help    : 1 to obtain a complete help
**
*****************************************************************************/
static void st_usage(char *exe_name,
                     int help)
{
  (void)printf("Usage: %s [-help] [-host hostname] [-port port_number]\n",
               exe_name);
  (void)printf("          [-path data_path]");

  if (help)
  {
    (void)printf(" -help             : Print this message.\n");
    (void)printf(" -debug            : Print client/server messages.\n");
    (void)printf(" -host hostname    : Server's hostname.\n");
    (void)printf(" -port port_number : Port used between client and server\n");
    (void)printf(" -path data_path   : Initialize connection with given data_path.\n");
    (void)printf("The default is to run a new server on a random port.");
    (void)printf("If -host or -port is specified, a server must have been started first on the given host/port");
  }
}

/****************************************************************************
**
** FUNCTION:    st_is_in_list
**
** DESCRIPTION: check presence of an item in a list
**
** RETURNS:     0 if not there, 1 if present
**
** ARGUMENTS:   list:  list of elements
** ARGUMENTS:   item:  element to check existence for
**
*****************************************************************************/
static int st_is_in_list(GTXStringArray *list,
                         const char  *item)
{
  int i;
  for (i = 0; i < list->GetCount(); i++)
    if (!strcmp(list->GetValue(i), item))
      return(1);
  return(0);
}

/****************************************************************************
**
** FUNCTION: st_create_grid
**
** PURPOSE:  Creates a grid file
**
** RETURNS:  0 if Ok, 1 on error
**
** IN_ARGS:  dimension: file dimension 2 for 2D, 3 for 3D
**
*****************************************************************************/
static int st_create_grid(int dimension)
{
  int nsample,nx,ny,nz,error;
  double *array;
  GTXStringArray list;
  int i,j,nb_list;
  int selection_ind[3] = {1,12,50};

  error = 1;
  array = NULL;
  nb_list = 0;
  nx = 20;
  ny = 15;
  nz = (dimension == 3)? 30: 1;
  nsample = nx*ny*nz;

  /*
  ** Create a new grid file
  ** if it already exists, it must be deleted before !!
  */
  list = client->GetFileList();
    
  if (st_is_in_list( &list, GTX_GRID_FILE))
  {
    client->SetFile(GTX_GRID_FILE);
    client->DeleteFile();
  }
  
  if (dimension == 3)
    client->NewGridFile3D(GTX_GRID_FILE,
                          1., 2., 3.,
                          4., 5., 6.,
                          nx, ny, nz);
  else
    client->NewGridFile2D(GTX_GRID_FILE,
                          1., 2., 
                          4., 5., 
                          nx, ny);


  /*
  ** Set current file
  */
  client->SetFile(GTX_GRID_FILE);
    
  
  /*
  ** Modify Grid Attributes
  ** We could have created it with the right origin and mesh, it's a test
  ** However the only way to specify the orientation of the grid is
  ** after its creation, like there.
  */
  client->GridModifyOrigin(1000., 2000., 3000.);
  
  client->GridModifyMesh(100., 200., 300.);
  
  client->GridModifyRotation(45., 0., 0.);
  
  
  /*
  ** Create a variable
  */

  format.SetType(GTXVariableFormat::FORMAT_TYPE_DECIMAL);
  format.SetLength(10);
  format.SetDigits(2);
  format.SetUnit(" ");
  client->NewDoubleVariable(GTX_VARIABLE, 64, format);
  
  
  /*
  ** Set the current variable
  */
  client->SetVariable(GTX_VARIABLE);
  
  /*
  ** Write data in the variable
  */
  array = (double*)malloc(nsample*sizeof(double));
  if (array == NULL) 
    for (i = 0; i < nsample; i++)
      array[i] = (double)(rand() % 1000);

  doubledata = new GTXDoubleData( nsample, array);
  doubledata->SetUndefinedValue(-9999.);
  client->WriteDoubleVariable(false, *doubledata);
  delete doubledata;

  format.SetType(GTXVariableFormat::FORMAT_TYPE_DECIMAL);
  format.SetLength(10);
  format.SetDigits(2);
  format.SetUnit(" ");
  client->NewDoubleVariable(GTX_MACRO, 32,format);
  
  client->SetVariable(GTX_MACRO);
  
  GTXIntArray IAselection_ind(3, selection_ind);
  client->AddMacroIndices(IAselection_ind);
  
  
  for (i = 0; i<3; i++)
  {
    client->SetIndice(selection_ind[i]);
    
    for (j = 0; j<nsample;j++)
      array[j] = selection_ind[i]*1000 + (rand()%1000);

  doubledata = new GTXDoubleData( nsample, array);
  doubledata->SetUndefinedValue(-9999.);
  client->WriteDoubleVariable(false, *doubledata);
  delete doubledata;
    
    
  }
  error = 0;
  
  if (array) free(array);
  return(error);
}

/****************************************************************************
 **
 ** FUNCTION: st_create_points
 **
 ** PURPOSE:  Creates a points file
 **
 ** RETURNS:  0 if Ok, 1 on error
 **
 ** IN_ARGS:  dimension: file dimension 2 for 2D, 3 for 3D
 **
 *****************************************************************************/
static int st_create_points(int dimension)
{
  int nsample,error;
  double *x,*y,*z, *array;
  GTXStringArray list;
  int i,nb_list;
  
  error = 1;
  x = y = z = array = NULL;
  nb_list = 0;
  nsample = 50;
  
  /*
  ** Create a new points file
  */
  x = (double*)malloc(sizeof(double)*nsample);
  y = (double*)malloc(sizeof(double)*nsample);
  if (dimension == 3)
    z = (double*)malloc(sizeof(double)*nsample);
  array = (double*)malloc(sizeof(double)*nsample);
  
    
  for (i = 0; i < nsample; i++)
  {
    x[i] = (double)((rand() % 3500) - 1000);
    y[i] = (double)((rand() % 3500) + 2000);
    if (z)
      z[i] = (double)((rand() % 9000) + 3000);
    array[i] = (double)(rand() % 1000);
  }
  
  /*
  ** if it already exists, it must be deleted before !!
  */
  list = client->GetFileList();
  
  if (st_is_in_list( &list, GTX_POINTS_FILE))
  {
    client->SetFile(GTX_POINTS_FILE);
    client->DeleteFile();
  }
  /* The file will have the right dimension according to the fact that
     z variable is NULL or not. */
  if (dimension ==2)
    client->NewPointsFile2D(GTX_POINTS_FILE, nsample, x, y);
  else
    client->NewPointsFile3D(GTX_POINTS_FILE, nsample, x, y, z);
    
  
  client->SetFile(GTX_POINTS_FILE);
  
  if (dimension ==2)
    client->PointsFileAppend2D(nsample,x, y);
  else
    client->PointsFileAppend3D(nsample,x, y, z);
  
  
  
  /*
  ** Set current file
  */
  client->SetFile(GTX_POINTS_FILE);
  
  
  /*
  ** Create a variable
  */
  format.SetType(GTXVariableFormat::FORMAT_TYPE_DECIMAL);
  format.SetLength(10);
  format.SetDigits(2);
  format.SetUnit(" ");
  client->NewDoubleVariable(GTX_VARIABLE, 64, format);
  
  
  /*
  ** Set the current variable
  */
  client->SetVariable(GTX_VARIABLE);
  
  
  /*
  ** Write only a subpart of it
  */
  doubledata = new GTXDoubleData( nsample, array);
  doubledata->SetUndefinedValue(-9999.);
  client->WriteDoubleVariableSub((gtx_long)5,(gtx_long) nsample+4, false, *doubledata);
  delete doubledata;
  
  error = 0;
  
  if (x) free(x);
  if (y) free(y);
  if (z) free(z);
  if (array) free(array);
  
  return(error);
}

/****************************************************************************
 **
 ** FUNCTION: st_create_lines
 **
 ** PURPOSE:  Creates a points + lines file
 **
 ** RETURNS:  0 if Ok, 1 on error
 **
 ** IN_ARGS:  dimension: file dimension 2 for 2D, 3 for 3D
 **
 ** REMARKS:  We create both a points and a lines file but a lines file can
 ** REMARKS:  exist alone.
 **
 *****************************************************************************/
static int st_create_lines(int dimension)
{
  int error;
  double *xhead,*yhead,*zhead;
  double *x,*y,*z, *array;
  unsigned char *sel_array;
  char **hole_id;
  GTXStringArray list;
  int i,j,k,ind,nb_list;
  int nlines = 3;
  int nsample[3] = {5,10,20};
  int max_nsample, total_nsample;
  int selection_ind[3] = {1,2,3};
  error = 1;
  x = y = z = array = NULL;
  xhead = yhead = zhead = NULL;
  hole_id = NULL;
  nb_list = 0;
  max_nsample = 0;
  total_nsample = 0;
  for (i = 0; i < nlines; i++)
  {
    total_nsample += nsample[i];
    if (nsample[i] > max_nsample)
      max_nsample = nsample[i];
  }
  
  /* Allocations, there are no checks here. It's dangerous but that's only
     for the test...*/
  xhead = (double*)malloc(sizeof(double)*nlines);
  yhead = (double*)malloc(sizeof(double)*nlines);
  if (dimension == 3)
    zhead = (double*)malloc(sizeof(double)*nlines);
  
  x = (double*)malloc(sizeof(double)*max_nsample);
  y = (double*)malloc(sizeof(double)*max_nsample);
  if (dimension == 3)
    z = (double*)malloc(sizeof(double)*max_nsample);
  array = (double*)malloc(sizeof(double)*max_nsample);
  sel_array = (unsigned char*)malloc(sizeof(unsigned char)*total_nsample);
  hole_id = (char**)malloc(sizeof(char*)*nlines);
  for (i = 0; i < nlines; i++)
    hole_id[i] = (char*)malloc(sizeof(char)*7);
  
  /*
  ** Create Head samples somewhere around the top of XOY section */
  for (i = 0; i < nlines; i++)
  {
    xhead[i] = (double)((rand() % 3500) - 1000);
    yhead[i] = (double)((rand() % 1000) + 4500);
    if (zhead)
      zhead[i] = (double)((rand() % 1000) + 11000);
    (void)sprintf(hole_id[i], "Hole %d", i+1);
  }
  
  /*
  ** Create a new header points file
  ** if it already exists, it must be deleted before !!
  */
  list = client->GetFileList();
  
  if (st_is_in_list( &list, GTX_HPOINTS_FILE))
  {
    client->SetFile(GTX_HPOINTS_FILE);
    client->DeleteFile();
  }
  /* The file will have the right dimension according to the fact that
     z variable is NULL or not. */
  if (dimension ==2)
    client->NewPointsFile2D(GTX_HPOINTS_FILE, nlines, xhead, yhead);
  else
    client->NewPointsFile3D(GTX_HPOINTS_FILE, nlines, xhead, yhead, zhead);
  
  
  /*
  ** Add it a hole id variable
  */
  client->SetFile(GTX_HPOINTS_FILE);
  client->NewCharVariable(GTX_LID_VARIABLE, 6);
  client->SetVariable(GTX_LID_VARIABLE);

  chardata = new GTXCharData(nlines, hole_id);
  client->WriteCharVariable(0,  *chardata);
  delete chardata;
  
  /*
  ** Create a new lines file
  ** if it already exists, it must be deleted before !!
  */
  if (st_is_in_list( &list, GTX_LINES_FILE))
  {
    client->SetFile(GTX_LINES_FILE);
    client->DeleteFile();
  }
  
  client->NewLinesFile(GTX_LINES_FILE, dimension);
  
  /*
  ** Set current file
  */
  client->SetFile(GTX_LINES_FILE);
  
  for (i = 0; i < nlines; i++)
  {
    for (j = 0; j < nsample[i]; j++)
    {
      x[j] = xhead[i];
      if (dimension == 2)
        y[j] = ((j == 0)? yhead[i]: y[j-1])- (double)(rand() % 250);
      else
      {
        y[j] = yhead[i];
        z[j] = ((j == 0)? zhead[i]: z[j-1]) - (double)(rand() % 900);
      }
    }

    if (dimension == 2)
      client->LinesFileAddCoreLine2D(
        nsample[i], xhead[i], yhead[i], x, y);
    else
      client->LinesFileAddCoreLine3D(
        nsample[i], xhead[i], yhead[i], (zhead != NULL)? zhead[i]: 0.,
        x, y, z);
  }
  
  /*
  ** The link between files must be created afterwards because
  ** appending samples would delete it elsewhere.
  */
  client->LinesFileCreateLink(GTX_HPOINTS_FILE);
  
  /*
  ** Create a float variable
  */
  format.SetType(GTXVariableFormat::FORMAT_TYPE_DECIMAL);
  format.SetLength(10);
  format.SetDigits(6);
  format.SetUnit(" ");
  client->NewDoubleVariable(GTX_VARIABLE, 64, format);
  
  /*
  ** Set the float variable as current and write it.
  ** Please note that we are writing here using WriteLineDoubleVariable
  ** because it is easier when you have only a few lines to write but it can be
  ** time consuming if you might have a lot of lines to write and should in this
  ** case consider using WriteDoubleVariableSub.
  */
  client->SetVariable(GTX_VARIABLE);
  
  for (i = 0; i < nlines; i++)
  {
    for (j = 0; j < nsample[i]; j++)
      array[j] = (rand() % 1000) / 1000.;

    doubledata = new GTXDoubleData(nsample[i], array);
    doubledata->SetUndefinedValue(-9999.);
    client->WriteLineDoubleVariable(i+1, *doubledata);
    delete doubledata;
  }

  /*
  ** read it back for the example
  */
  GTXFileInfo finfo = client->GetFileInfo();
  for (i = 0; i < finfo.GetItemNumber(); i++)
  {
    GTXDoubleData ddata;
    ddata = client->ReadLineDoubleVariable(i+1);
  }
  
  /*
  ** Create an alpha variable
  */
  client->NewCharVariable(GTX_AVARIABLE, 10);
  client->SetVariable(GTX_AVARIABLE);

  /*
  ** Set the alpha variable as current and write it.
  */
  for (i = 0; i < nlines; i++)
  {
    GTXCharData cdata;
    cdata.Resize(nsample[i]);
    char tmp[10];
    for (j = 0; j < nsample[i]; j++)
    {
      sprintf(tmp, "%d_%d", i+1, j+1);
      cdata.SetValue(j, tmp);
    }
    client->WriteLineCharVariable(i+1, cdata);
  }

  /*
  ** read it back for the example
  */
  for (i = 0; i < finfo.GetItemNumber(); i++)
  {
    GTXCharData cdata2;
    cdata2 = client->ReadLineCharVariable(i+1);
  }

  /*
  ** Create a macro variable with alpha indices
  ** It is a macro-selection with the hole-id being used as the alpha index
  */
  format.SetType(GTXVariableFormat::FORMAT_TYPE_INTEGER);
  format.SetLength(3);
  format.SetDigits(0);
  format.SetUnit(" ");
  client->NewDoubleVariable(GTX_MACRO_SEL, 1,format);
  
  client->SetVariable(GTX_MACRO_SEL);
  
  GTXIntArray IAindices(nlines, selection_ind);
  GTXStringArray SAalpha_indices(nlines, hole_id);
  
  client->AddMacroIndicesWithAlpha(IAindices, SAalpha_indices);
  
  for (ind = 0; ind < nlines; ind++)
  {
    client->SetAlphaIndice(hole_id[ind]);
    
    for (i = 0, k = 0; i < nlines; i++)
    {
      for (j = 0; j < nsample[i]; j++, k++)
        sel_array[k] = (ind == i);
    }

    GTXByteArray BAsel_array(total_nsample, sel_array);
    client->WriteSelectionVariable(BAsel_array);
    
    /* Just to test the selection read function*/
    GTXByteArray readtest;
    readtest = client->ReadSelectionVariable();
  }
  
  /*
  ** Set the hole id variable (in the Points File) as the Line Name
  */
  client->SetFile(GTX_HPOINTS_FILE);
  client->SetVariable(GTX_LID_VARIABLE);
  client->SetVariableAsLineName();

  error = 0;
  
  if (xhead) free(xhead);
  if (yhead) free(yhead);
  if (zhead) free(zhead);
  if (hole_id)
  {
    for (i = 0; i < nlines; i++)
      if (hole_id[i]) free(hole_id[i]);
    free(hole_id);
  }
  if (x) free(x);
  if (y) free(y);
  if (z) free(z);
  if (array) free(array);
  if (sel_array) free(sel_array);
  
  return(error);
}

/****************************************************************************
 **
 ** FUNCTION:    main
 **
 ** DESCRIPTION: client's main function
 **
 ** RETURNS:     Nothing
 **
 ** ARGUMENTS:   int argv, char *argv : program arguments
 **
 *****************************************************************************/
int main(int argc,
         char *argv[])
{
  int dim,error;
  const char *dir;


  char **vars = NULL;
  int  *indices = NULL;

  error = 1;

  GTXStringArray list;
  Arguments args;
  args.ParseCommandLine(argc, argv);

  try
  {
    /*
    ** Connect to the server
    */
    if (args.run_server)
    {
      args.port = 0;
      args.port = client->RunGTXserver(args.port);
    }
    client->Connect(args.host, args.port, args.data_path);
  

    /*
    ** Set current unit
    */
    client->SetUnit("m");
    
    /*
    ** Set current study (create it if necessary and set)
    */
    list = client->GetStudyList();
  
    if(!st_is_in_list( &list, GTX_STUDY))
      client->NewStudy(GTX_STUDY, GTX_STUDY_PATH);
  
    client->SetStudy(GTX_STUDY);
  
    /* Create a 2D set and a 3D set of data */
    for (dim = 2; dim <= 3; dim++)
    {
      dir = (dim == 2)? GTX_DIR2D: GTX_DIR3D;
    
      /*
      ** Set current directory (create it if necessary and set)
      */
      list = client->GetDirectoryList();
    
      if (!st_is_in_list( &list, dir))
        client->NewDirectory(dir);

      client->SetDirectory(dir);

      st_create_grid(dim);
      st_create_points(dim);
      st_create_lines(dim);
    }

    /*
    ** Disconnect from the server
    */
    client->Disconnect();
    error = 0;
  }
  catch (GTXError e)
  {
    printf("GTXError: %s\n", e.GetMessage());
  }

  return(error);
}
