/*****************************************************************************

            Copyright (c)2008 Geovariances, Avon, France.

    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.

    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).

            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE

             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr

                        All Rights Reserved

*****************************************************************************/

using System;
using System.IO;
using com.geovariances.gtxclient;
using System.Reflection;
using System.Runtime.CompilerServices;

/// <summary>
/// Class containing variable statistics
/// </summary>
class VarStats
{
  /// <summary>
  /// constructor initializing stats
  /// </summary>
  public VarStats()
  {
    min = 1e300;
    max = -1e300;
    mean = 0;
    nsamples = 0;
  }
  /// <summary>
  ///  number of defined samples
  /// </summary>
  public int nsamples;

  /// <summary>
  /// Maximum taken by the variable on defined samples
  /// </summary>
  public double max;

  /// <summary>
  /// Minimum taken by the variable on defined samples
  /// </summary>
  public double min;

  /// <summary>
  /// Mean of the defined samples
  /// </summary>
  public double mean;
}

/// <summary>
/// Class used to dump information on Files and Variables
/// </summary>
class InfoDump
{
  /// <summary>
  /// Dump file information into a string
  /// </summary>
  /// <param name="client">current GTXclient session</param>
  /// <param name="nindent">number of spaces to prefix each line</param>
  /// <returns>composed string containing file information</returns>
  public static string FileInfo(GTXClient client,
    int nindent)
  {
    string prefix = "";
    string buffer = "";
    for (int i = 0; i < nindent; i++) prefix += " ";

    GTXFileInfo finfo = client.GetFileInfo();

    String file_type;
    switch (finfo.GetFileType())
    {
      case GTXFileInfo.FileType.FILE_TYPE_POINTS: file_type = "Points";  break;
      case GTXFileInfo.FileType.FILE_TYPE_GRAVITY_LINES: file_type = "Lines"; break;
      case GTXFileInfo.FileType.FILE_TYPE_GRID: file_type = "Grid"; break;
      default: file_type = "Unknown"; break;
    }
    buffer += prefix + file_type + " File" + "\n";
    buffer += prefix + "Dimension = " + finfo.GetDimension() + "D" + "\n";
    buffer += prefix + "Number of samples = " + finfo.GetSampleNumber() + "\n";
    if (finfo.GetFileType() == GTXFileInfo.FileType.FILE_TYPE_GRAVITY_LINES ||
        finfo.GetFileType() == GTXFileInfo.FileType.FILE_TYPE_CORE_LINES)

    {
      buffer += prefix + "Number of lines = " + finfo.GetItemNumber() + "\n";
    }
    else if (finfo.GetFileType() == GTXFileInfo.FileType.FILE_TYPE_GRID)
    {
      if (finfo.GetDimension() == 2)
      {
        buffer += prefix + "Number of nodes = (" + finfo.GetGridNX() + "," + finfo.GetGridNY() + ")" + "\n";
        buffer += prefix + "Origin          = (" + finfo.GetGridX0() + "," + finfo.GetGridY0() + ")" + "\n";
        buffer += prefix + "Mesh dimensions = (" + finfo.GetGridDX() + "," + finfo.GetGridDY() + ")" + "\n";
        if (finfo.GetGridRotatedFlag())
          buffer += prefix + "Rotation angle = " + finfo.GetGridAngleAroundZ() + "\n";
        else
          buffer += prefix + "No rotation" + "\n";
      }
      else
      {
        buffer += prefix + "Number of nodes = (" + finfo.GetGridNX() + "," + finfo.GetGridNY() + "," + finfo.GetGridNZ() +
          ")\n";
        buffer += prefix + "Origin          = (" + finfo.GetGridX0() + "," + finfo.GetGridY0() + "," + finfo.GetGridZ0() +
          ")\n";
        buffer += prefix + "Mesh dimensions = (" + finfo.GetGridDX() + "," + finfo.GetGridDY() + "," + finfo.GetGridDZ() +
          ")\n";
        if (finfo.GetGridRotatedFlag())
        {
          buffer += prefix + "Rotation angle around Z = " + finfo.GetGridAngleAroundZ() + "\n";
          buffer += prefix + "Rotation angle around Y = " + finfo.GetGridAngleAroundY() + "\n";
          buffer += prefix + "Rotation angle around X = " + finfo.GetGridAngleAroundX() + "\n";
        }
        else
          buffer += prefix + "No rotation" + "\n";
      } // end of 3D
    } // end of if Grid Type

    String comment = client.GetFileComment();
    if (comment.Length != 0)
      buffer += prefix + "Comment: " + comment + "\n";

    // special variables information
    buffer += prefix + "Sample Number variable: " + finfo.GetSampleNumberVariableName() + "\n";
    if (finfo.GetFileType() == GTXFileInfo.FileType.FILE_TYPE_GRAVITY_LINES ||  
        finfo.GetFileType() == GTXFileInfo.FileType.FILE_TYPE_CORE_LINES)
    {
      buffer += prefix + "Line Number variable: " + finfo.GetLineNumberVariableName() + "\n";
      buffer += prefix + "Relative Number variable: " + finfo.GetRelativeNumberVariableName() + "\n";
    }
    buffer += prefix + "X Gravity Center variable: " + finfo.GetXCoordinateVariableName() + "\n";
    buffer += prefix + "Y Gravity Center variable: " + finfo.GetYCoordinateVariableName() + "\n";
    if (finfo.GetDimension() == 3)
      buffer += prefix + "Z Gravity Center variable: " + finfo.GetZCoordinateVariableName() + "\n";


     if (finfo.GetFileType() == GTXFileInfo.FileType.FILE_TYPE_CORE_LINES)
     {
       buffer += "X Core Begin variable: " + finfo.GetXCoreBeginVariableName() + "\n";
       buffer += "Y Core Begin variable: " + finfo.GetYCoreBeginVariableName() + "\n";
       if (finfo.GetDimension() == 3)                             
         buffer += "Z Core Begin variable: " + finfo.GetZCoreBeginVariableName() + "\n";
       
       buffer += "X Core End variable: " + finfo.GetXCoreEndVariableName() + "\n";
       buffer += "Y Core End variable: " + finfo.GetYCoreEndVariableName() + "\n";
       if (finfo.GetDimension() == 3)                             
         buffer += "Z Core End variable: " + finfo.GetZCoreEndVariableName() + "\n";
     }
            

    if (finfo.GetFaultedFlag())
    {
      GTXFaultInfo fault_info = finfo.GetFaultInfo();
      buffer += prefix + "\nFaulted File:" + "\n";
      buffer += prefix + "  2D Faults: " +
        (fault_info.GetFaults2DFlag() ? "Y": "N" + "\n");
      buffer += prefix + "  Auxiliary Variable Used: " +
        (fault_info.GetAuxiliaryVarUsedFlag() ? "Y": "N" + "\n");
      buffer += prefix + "  Number of faults: " +
        fault_info.GetFaultsNumber() + "\n";
      buffer += prefix + "  Number of segments: " +
        fault_info.GetSegmentsNumber() + "\n";
      buffer += prefix + "  Minimum Used Priority: " +
        fault_info.GetMinimumPriority() + "\n";
      buffer += prefix + "  Maximum Used Priority: " +
        fault_info.GetMaximumPriority() + "\n";
      buffer += prefix + "  Authorized Priority: " +
        fault_info.GetAuthorizedPriority() + "\n";

      // Exemple of Faults Reading
      /*GTXFaultSystem faultSystem =
        client.ReadFaults(fault_info.auth_priority);*/
    }
    return buffer;
  }

  /// <summary>
  /// Dump variable information/statistics into a string
  /// </summary>
  /// <param name="client">current GTXclient session</param>
  /// <param name="need_stats">if true, variable is read and stats are reported</param>
  /// <param name="nindent">number of spaces to prefix each line</param>
  /// <returns>composed string containing variable information</returns>
  public static string VariableInfo(GTXClient client,
    bool need_stats,
    int nindent)
  {
    string prefix = "";
    string buffer = "";
    for (int i = 0; i < nindent; i++) prefix += " ";

    String var_type;
    GTXVariableInfo vinfo = client.GetVariableInfo();
    switch (vinfo.GetVariableType())
    {
      case GTXVariableInfo.VariableType.VAR_TYPE_FLOAT: var_type = "Float"; break;
      case GTXVariableInfo.VariableType.VAR_TYPE_CHAR: var_type = "Character"; break;
      case GTXVariableInfo.VariableType.VAR_TYPE_XG: var_type = "X Gravity Center"; break;
      case GTXVariableInfo.VariableType.VAR_TYPE_YG: var_type = "Y Gravity Center"; break;
      case GTXVariableInfo.VariableType.VAR_TYPE_ZG: var_type = "Z Gravity Center"; break;
      case GTXVariableInfo.VariableType.VAR_TYPE_MACRO: var_type = "Macro"; break;
      default: var_type = "Unknown"; break;
    }
    
    buffer += prefix + "Type: " + var_type + "\n";

    if (vinfo.GetVariableType() == GTXVariableInfo.VariableType.VAR_TYPE_CHAR)
    {
      buffer += prefix + "Alpha Length: " + vinfo.GetAlphaLength() + "\n";
    }
    else
    {
      if (vinfo.GetBitLength() == 1)
        buffer += prefix + "Precision: 1 bit (selection)" + "\n";
      else
      {
        buffer += prefix + "Precision: " + vinfo.GetBitLength() + " bits" + "\n";
        if (!vinfo.GetImplicitFlag())
          buffer += prefix + "Unit: " + vinfo.GetUnit() + "\n";
      }
    }
    string comment = client.GetVariableComment();
    if (comment.Length != 0)
      buffer += prefix + "Comment: " + comment + "\n";
    if (vinfo.GetVariableType() == GTXVariableInfo.VariableType.VAR_TYPE_MACRO)
    {
      int[] indices = client.GetMacroIndices().GetValues();
      buffer += prefix + "Number of Indices: " + indices.Length + "\n";
      if (indices.Length != 0)
      {
        for (int m=0; m<indices.Length; m++)
          buffer += prefix + indices[m] + "\n";
      }
    }
    if (vinfo.GetVariableType() == GTXVariableInfo.VariableType.VAR_TYPE_FLOAT && need_stats)
    {
      VarStats stats = statOnVariable(client);
      buffer += prefix + "# Defined: " + stats.nsamples + "\n";
      buffer += prefix + "Minimum: " + stats.min + "\n";
      buffer += prefix + "Maximum: " + stats.max + "\n";
      buffer += prefix + "Mean: " + stats.mean/stats.nsamples + "\n";
    }
    return buffer;
  }

  /// <summary>
  /// Read variable and computes its stats
  /// </summary>
  /// <param name="session">current GTXclient session</param>
  /// <returns>computed variable statistics</returns>
  public static VarStats statOnVariable(GTXClient session)
  {
    VarStats stats = new VarStats();
    GTXDoubleData vdouble = session.ReadDoubleVariable(false);
    double[] vals = vdouble.GetValues();
    for (int n=0; n<vals.Length; n++)
    {
      double val = vals[n];
      if (val != vdouble.GetUndefinedValue())
      {
        stats.mean += val;
        stats.min = ((val < stats.min) ? val : stats.min);
        stats.max = ((val > stats.max) ? val : stats.max);
        stats.nsamples++;
      }
    }
    return stats;
  }
}
