#include <stdio.h>
#include <dlfcn.h>
#include <stdlib.h>
#include <unistd.h> /* sleep() on solaris */
#include <string.h> /* strcpy() on solaris */
#include "GTXClient.h" /* GTX_VAR_TYPE_xxx */
#include "GTXClientP.h"

void *handle_lib; /* Handle de library */
int CompressFlag = 0;
int GTXState;

/******************************************************************************
**
** Chargement de la library libgtxclient.so 
**
******************************************************************************/
static void *stDynloadOpen(const char *library_name)

{
 void *mylib;
 char full_name[500];
 
 /* Chemin de la librairie GTXserver */
 sprintf(full_name,
         "/bibli/Export/isatis_export/lib/GTXserver-1.0.0/solaris/lib%s.so",
         library_name);
 
 mylib = dlopen(full_name, RTLD_NOW);
 if( mylib == NULL)
 {
   fprintf(stderr, "Unable to load: %s", full_name);
   printf("%s\n", dlerror());
   return(NULL);
 }
 return(mylib);
}

/******************************************************************************
**
**               Dechargement de la librarie
**
******************************************************************************/
static void stDynloadClose(void *library)

{
 dlclose(library);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientConnect
**
** DESCRIPTION: runs a mono session GTXserver to read isatis files
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** IN_ARGS:     char **path   : path to the Isatis File System to be accessed
**
**
** REMARKS:     A session must be closed using GTXClientDisconnect to free
** REMARKS:     memory allocations...
**
*****************************************************************************/
void S_GTXClientConnect(char **path,
                        int   *retcode)

{
  int (*GTXClientConnect)(char *, int, char *);
  int  portmap;
  char string[100];
  extern double drand48();

  /* Initializations*/
  *retcode = -1;
  
  /* Chargement de libgtxclient.so pour une connection au GTXserver*/
  handle_lib = stDynloadOpen("gtxclient");  
  
  /* Recuperation de l'adresse de la routine GTXClientConnect */
  GTXClientConnect = (int (*)(char *, int, char *)) 
    dlsym(handle_lib, "GTXClientConnect");
  if (GTXClientConnect == NULL) return;
  
  /* Creating the path (if not defined) */
  if (! strcmp(path[0],""))
    (void) sprintf(path[0],"%s/Isatis40",getenv("HOME"));

  /* Launching the GTXserver application */
  printf("Launching GTXserver ... This may take some time\n");
  portmap = (int) (1000. + 1000. * drand48());
  sprintf (string,"GTXserver -port %d",portmap);
  system(string);
  sleep(1);
  
  /* Connecting to GTXserver */
  printf("\nConnecting to GTXserver ... (Port=%d)\n",portmap);
  if(GTXClientConnect("localhost",(int) portmap, path[0]))
  {
    printf("Unable to connect GTXserver on path=%s\n",path[0]);
    *retcode = 1;
    return;
  }
  else
  {
    printf("Connecting to GTXserver ... Done\n");
    *retcode = 0;
  }
  
  return;
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientConnectExisting
**
** DESCRIPTION: connect to an already running GTXserver to read isatis files
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** IN_ARGS:     char **IP_add : hostname or IP address "aaa.bbb.ccc.ddd" of
** IN_ARGS:                    GTXserver machine
** IN_ARGS:     int *port     : port connected to the running GTXserver
** IN_ARGS:     char **path   : path to the Isatis File System to be accessed
**
**
** REMARKS:     A session must be closed using GTXClientDisconnect to free
** REMARKS:     memory allocations...
**
*****************************************************************************/
void S_GTXClientConnectExisting(char **IP_add,
                                int   *port,
                                char **path,
                                int   *retcode)

{
  int (*GTXClientConnect)(char *, int, char *);

  /* Initializations*/
  *retcode = -1;
  
  /* Chargement de libgtxclient.so pour une connection au GTXserver*/
  handle_lib = stDynloadOpen("gtxclient");  
  
  /* Recuperation de l'adresse de la routine GTXClientConnect */
  GTXClientConnect = (int (*)(char *, int, char *)) 
    dlsym(handle_lib, "GTXClientConnect");
  if (GTXClientConnect == NULL) return;
  
  /* Creating the path (if not defined) */
  if (! strcmp(*path,""))
    (void)sprintf(*path, "%s/Isatis40",getenv("HOME"));

  /* Connecting to GTXserver */
  printf("\nConnecting to GTXserver ... (Host=%s Port=%d)\n",
         *IP_add, *port);
  if (GTXClientConnect(*IP_add, *port, *path))
  {
    printf("Unable to connect GTXserver on path=%s\n",*path);
    *retcode = 1;
    return;
  }
  else
  {
    printf("Connecting to GTXserver ... Done\n");
    *retcode = 0;
  }
  
  return;
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientDisconnect
**
** DESCRIPTION: close a session
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** OUT_ARGS:    None
**
*****************************************************************************/
int S_GTXClientDisconnect(void)

{
  int (*GTXClientDisconnect)(void);
  int error_code;

  GTXClientDisconnect = (int (*)(void)) 
    dlsym(handle_lib, "GTXClientDisconnect");
  if (GTXClientDisconnect == NULL) return(-1);
 
  printf("\nDisconnecting to GTXserver ... ");
  if((error_code = GTXClientDisconnect()) != 0)
  {
    printf("\nUnable to disconnect GTXserver\n");
    return(error_code);
  } 
  else
  {
    printf("Disconnection to GTXserver ... Done\n");
  }  

  /* Dechargement de la library */
  stDynloadClose(handle_lib);

  return(error_code);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientGetStudyList
**
** DESCRIPTION: get study list in the Isatis file system
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** OUT_ARGS:    int *nb_study   : number of directories
** OUT_ARGS:    char **studies : study list
**
*****************************************************************************/
int S_GTXClientGetStudyList(long *nb_study,
                            char **studies)

{
  int (*GTXClientGetStudyList)(int *, char ***);
  int i;

  GTXClientGetStudyList = (int (*)(int *, char ***)) 
    dlsym(handle_lib, "GTXClientGetStudyList");
  if (GTXClientGetStudyList == NULL) return(-1);

  if((GTXState = GTXClientGetStudyList((int *) nb_study, &studies)) !=0)
    return(GTXState);
 
  for (i=0; i<(int) *nb_study; i++) free(studies[i]);
  free(studies);
  studies = NULL;
  
  return(0);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientSetStudy
**
** DESCRIPTION: set the name of the current study
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** IN_ARGS:     char **study: study name
**
*****************************************************************************/
int S_GTXClientSetStudy(char **study)

{
  int (*GTXClientSetStudy)(char *); 
  
  GTXClientSetStudy = (int (*)(char *)) 
    dlsym(handle_lib, "GTXClientSetStudy");
  if (GTXClientSetStudy == NULL) return(-1);

  if((GTXState = GTXClientSetStudy(study[0])) != 0) 
    return(GTXState);

  return(0); 
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientGetDirectoryList
**
** DESCRIPTION: get directory list in the current study
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** OUT_ARGS:    long *nb_dir          : number of directories
** OUT_ARGS:    char **directories    : directory list
**
*****************************************************************************/
int S_GTXClientGetDirectoryList(long *nb_dir,
                                char **directories)

{
  int (*GTXClientGetDirectoryList)(int *, char ***);
  int i;

  GTXClientGetDirectoryList = (int (*)(int *, char ***)) 
    dlsym(handle_lib, "GTXClientGetDirectoryList");
  if (GTXClientGetDirectoryList == NULL) return(-1);

  if((GTXState = GTXClientGetDirectoryList((int *) nb_dir, &directories)) != 0)
    return(GTXState);

  for (i=0; i<(int) *nb_dir; i++) free(directories[i]);
  free(directories);
  directories = NULL;
  
  return(0);
}

/****************************************************************************
**
** FUNCTION:   S_GTXClientSetDirectory
**
** DESCRIPTION: set current directory in the current study
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** IN_ARGS:     char **dir: directory name
**
*****************************************************************************/
int S_GTXClientSetDirectory(char **dir)

{
  int (*GTXClientSetDirectory)(char *); 

  GTXClientSetDirectory = (int (*)(char *)) 
    dlsym(handle_lib, "GTXClientSetDirectory");
  if (GTXClientSetDirectory == NULL) return(-1);
  
  if((GTXState = GTXClientSetDirectory(dir[0])) != 0) return(GTXState);
  
  return(0);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientGetFileList
**
** DESCRIPTION: get file list in the current directory
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** OUT_ARGS:    long *nb_file      : number of directories
** OUT_ARGS:    char **files       : directory list
**
*****************************************************************************/
int S_GTXClientGetFileList(long *nb_file,
                           char **files)

{
  int (*GTXClientGetFileList)(int *, char ***);
  int i;
  
  GTXClientGetFileList = (int (*)(int *, char ***)) 
    dlsym(handle_lib, "GTXClientGetFileList");
  if (GTXClientGetFileList == NULL) return(-1);
  
  if((GTXState = GTXClientGetFileList((int *) nb_file, &files)) != 0)
    return(GTXState);
  
  for (i=0; i<(int) *nb_file; i++) free(files[i]);
  free(files);
  files = NULL;

  return(0);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientSetFile
**
** DESCRIPTION: set current file in the current directory
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** IN_ARGS:     char **file: directory name
**
*****************************************************************************/
int S_GTXClientSetFile(char **file)

{
  int (*GTXClientSetFile)(char *); 

  GTXClientSetFile = (int (*)(char *)) 
    dlsym(handle_lib, "GTXClientSetFile");
  if (GTXClientSetFile == NULL) return(-1);
  
  if((GTXState = GTXClientSetFile(file[0])) != 0) 
    return(GTXState);
  
  return(0);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientGetVariableList
**
** DESCRIPTION: get variable list in the current file
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** OUT_ARGS:    int *nb_var  : number of variables
** OUT_ARGS:    char ***vars : variable list
**
*****************************************************************************/
int S_GTXClientGetVariableList(long *nb_var,
                               char **vars)

{
  int (*GTXClientGetVariableList)(int *, char ***);
  int i;

  GTXClientGetVariableList = (int (*)(int *, char ***)) 
    dlsym(handle_lib, "GTXClientGetVariableList");
  if (GTXClientGetVariableList == NULL) return(-1);
  
  if((GTXState = GTXClientGetVariableList((int *) nb_var, &vars)) != 0)
    return(GTXState);
  
  for (i=0; i<(int) *nb_var; i++) free(vars[i]);
  free(vars);
  vars = NULL;

  return(0);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientSetVariable
**
** DESCRIPTION: set current variable in the current file
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** IN_ARGS:     char **var: variable name
**
*****************************************************************************/
int S_GTXClientSetVariable(char **var)

{
  int (*GTXClientSetVariable)(char *); 

  GTXClientSetVariable = (int (*)(char *)) 
    dlsym(handle_lib, "GTXClientSetVariable");
  if (GTXClientSetVariable == NULL) return(-1);
  
  if((GTXState = GTXClientSetVariable(var[0])) !=0 ) 
    return(GTXState);
  
  return(0);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientReadVarCount
**
** DESCRIPTION: read the current number of float variable sample
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** IN_ARGS:     long *flag_compress : samples masked by the selection are
** IN_ARGS:                           transformed into test values if 0,
** IN_ARGS:                           or do not appear in the output array
** IN_ARGS:                           if this flag is set to 1
**
** OUT_ARGS:    long       *nb_val  : number of kept samples
**
*****************************************************************************/
int S_GTXClientReadVarCount(long   *flag_compress,
                            long   *nb_val)

{
  int (*GTXClientReadDoubleVariable)(int, double *, int *, double **); 
  int i;
  double *Tdbl_data;
  double test_val = 0;
  
  GTXClientReadDoubleVariable = (int (*)(int, double *,int *, double **)) 
    dlsym(handle_lib, "GTXClientReadDoubleVariable");
  if (GTXClientReadDoubleVariable == NULL) return(-1);
  
  if((GTXState = GTXClientReadDoubleVariable((int) *flag_compress, &test_val, 
                                             (int *) nb_val, 
                                             &Tdbl_data)) != 0)
    return(GTXState);
  
  free(Tdbl_data);   
  
  return(0);
}

/****************************************************************************
**
** FUNCTION:    S_GTXClientReadDoubleVariable
**
** DESCRIPTION: read the current float variable
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
**
** IN_ARGS:     long *flag_compress : samples masked by the selection are
** IN_ARGS:                           transformed into test values if 0,
** IN_ARGS:                           or do not appear in the output array
** IN_ARGS:                           if this flag is set to 1
**
** OUT_ARGS:    double    *test_val : value used for undefined samples
** OUT_ARGS:    long      *nb_val   : number of kept samples
** OUT_ARGS:    double    *dbl_data : array of numeric variable samples
**
*****************************************************************************/
int S_GTXClientReadDoubleVariable(long   *flag_compress,
                                  double *test_val,
                                  long   *nb_val,
                                  double *dbl_data)

{
  int (*GTXClientReadDoubleVariable)(int, double *, int *, double **); 
  int i;
  double *Tdbl_data;

  GTXClientReadDoubleVariable = (int (*)(int, double *,int *, double **)) 
    dlsym(handle_lib, "GTXClientReadDoubleVariable");
  if (GTXClientReadDoubleVariable == NULL) return(-1);
  
  if((GTXState = GTXClientReadDoubleVariable((int) *flag_compress, test_val, 
                                             (int *) nb_val, &Tdbl_data)) != 0)
    return(GTXState);
  
  for(i=0;i<(int) *nb_val;i++) dbl_data[i] = Tdbl_data[i]; 
  free(Tdbl_data); 
  
  return(0);
}

/****************************************************************************
**
** FUNCTION:    GTXClientSSReadTarget
**
** DESCRIPTION: Browse Isatis tree structure (SPlus Function)
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
** RETURNS:                     4 in case of dlsym error
**
** OUT_ARGS:    long *Sstudy     : Number of study request by user in Splus
** OUT_ARGS:    long *Sdirectory : Number of directory request by user in SPlus
** OUT_ARGS:    long *Sfile      : Number of file request by user in SPlus
**
*****************************************************************************/
int SSReadTarget(long *Sstudy, 
                 long *Sdirectory,
                 long *Sfile)

{
 int nb_study, nb_dir, nb_file;
 char **studies, **directories, **files;
 char Pstudies[50], Pdirectories[50], Pfiles[50];
 int i, var_browse, end_browse, Nstudy, Ndirectory, Nfile;

 int (*GTXClientGetStudyList)(int *, char ***);
 int (*GTXClientSetStudy)(char *); 
 int (*GTXClientGetDirectoryList)(int *, char ***);
 int (*GTXClientSetDirectory)(char *);
 int (*GTXClientGetFileList)(int *, char ***);
 int (*GTXClientSetFile)(char *);

 studies = directories = files = NULL;

 /* Variables servant au browsage de la l'aorborescence d'Isatis */
 end_browse = 1;
 var_browse = 0;
 
 while(end_browse)
 { 
   switch(var_browse)
   {
     case 0:                  /* Study Level */
     {       
     label_study_inquiry:
       GTXClientGetStudyList = (int (*)(int *, char ***)) 
         dlsym(handle_lib, "GTXClientGetStudyList");
       if (GTXClientGetStudyList == NULL) return(-1);
       
       if((GTXState = GTXClientGetStudyList(&nb_study, &studies)) != 0)
         return(GTXState); 
       
       /* Study Inquiry */
       
       if (*Sstudy < 1 || *Sstudy > nb_study)
       {
         
         /* Interactive */
         
         printf("\n");
         printf("Study List\n");
         printf("Number | Study Name\n");
         printf("-------|-----------\n");
         for(i=0; i<(int) nb_study; i++)
           printf("%6d | %s\n",i+1, studies[i]);
         do
         {
           printf("Enter the study number (between 1 and %d): ", 
                  nb_study);
           scanf("%d", &Nstudy);
         } while(Nstudy < 1 || Nstudy > nb_study);
       }
       else
       {
         Nstudy = (int) *Sstudy;
       }     
       
       GTXClientSetStudy = (int (*)(char *))
         dlsym(handle_lib, "GTXClientSetStudy");
       if (GTXClientSetStudy == NULL) return(-1);
       
       printf("\n");
       printf("Selected study is: %s \n",studies[Nstudy-1]); 
       strcpy(Pstudies, studies[Nstudy-1]);
       if ((GTXState = GTXClientSetStudy(studies[Nstudy-1])) != 0)
         return(GTXState);
       
       for (i=0; i< nb_study; i++) free(studies[i]);
       free(studies);
       var_browse = 1;
       break;
     }

     case 1:                  /* Directory Level */
     {
     label_directory_inquiry:
       GTXClientGetDirectoryList = (int (*)(int *, char ***)) 
         dlsym(handle_lib, "GTXClientGetDirectoryList");
       if (GTXClientGetDirectoryList == NULL) return(-1);
       
       if((GTXState = GTXClientGetDirectoryList(&nb_dir, &directories)) != 0)
         return(GTXState); 
       
       /* Directory Inquiry */
       
       if (*Sdirectory < 1 || *Sdirectory > nb_dir)
       {
         
         /* Interactive */
         
         printf("\n");
         printf("Directory List\n");
         printf("Number | Directory Name\n");
         printf("-------|-----------\n");
         printf("     0 | (back to the Study Level)\n");        
         for(i=0;i< nb_dir;i++)
           printf("%6d | %s\n",i+1, directories[i]);
         do
         {
           printf("Enter the directory number (between 0 and %d): ",
                  nb_dir);
           scanf("%d", &Ndirectory);
         } while(Ndirectory < 0 || Ndirectory > nb_dir);
       }
       else
       {
         Ndirectory = (int) *Sdirectory;
       }     
       
       if (Ndirectory == 0)
       {
         
         /* Return to the study level */
         
         *Sstudy = *Sdirectory = *Sfile = -1;
         var_browse  = 0;
         end_browse  = 1;
         goto label_study_inquiry;
       }
       
       GTXClientSetDirectory = (int (*)(char *)) 
         dlsym(handle_lib, "GTXClientSetDirectory");
       if (GTXClientSetDirectory == NULL) return(-1);
       
       printf("Selected directory is: %s \n",
              directories[Ndirectory-1]);        
       strcpy(Pdirectories, directories[Ndirectory-1]);
       
       if((GTXState = GTXClientSetDirectory(directories[Ndirectory-1])) != 0)
         return(GTXState); 
       
       for (i=0; i< nb_dir; i++) free(directories[i]);
       free(directories); 
       var_browse = 2;
       break;
     }

     case 2:                  /* File Level */
     {    
     label_file_inquiry:
       GTXClientGetFileList = (int (*)(int *, char ***)) 
         dlsym(handle_lib, "GTXClientGetFileList");
       if (GTXClientGetFileList == NULL) return(-1);
       
       if((GTXState = GTXClientGetFileList(&nb_file, &files)) != 0)
         return(GTXState); 
       
       /* File Inquiry */
       
       if (*Sfile < 1 || *Sfile > nb_dir)
       {
         
         /* Interactive */
         
         printf("\nFile List\n");
         printf("Number | file Name\n");
         printf("-------|-----------\n");
         printf("     0 | (back to the Directory Level)\n");
         for(i=0;i< nb_file;i++)
           printf("%6d | %s\n",i+1, files[i]);
         do
         {
           printf("Enter the file number (between 0 and %d): ",
                  nb_file);
           scanf("%d", &Nfile);
         } while (Nfile < 0 || Nfile > nb_file);
       }
       else
       {
         Nfile = (int) *Sfile;
       }     
       
       if (Nfile == 0)
       {
         *Sstudy = *Sdirectory = *Sfile = -1;
         var_browse = 1;
         end_browse = 1;
         goto label_directory_inquiry;
       }
       
       GTXClientSetFile = (int (*)(char *)) 
         dlsym(handle_lib, "GTXClientSetFile");
       if (GTXClientSetFile == NULL) return(-1);
       
       printf("Selected file is: %s \n",files[Nfile-1]); 
       strcpy(Pfiles, files[Nfile-1]);
       
       printf("\n");
       printf("Current Isatis Path\n");
       printf("-------------------\n");
       printf("Study    : %s\n",Pstudies);
       printf("Directory: %s\n",Pdirectories);
       printf("File     : %s\n",Pfiles);
       printf("\n");
       GTXClientSetFile(files[Nfile-1]);
       
       for (i=0; i< nb_file; i++) free(files[i]);
       free(files); 
       end_browse = 0;
       break;
     }
   }
 }
return(0);
}

/****************************************************************************
**
** FUNCTION:    SetValue
**
** DESCRIPTION: Select and read Isatis variable.
**
** RETURNS:     int error_code: 0 if Ok,
** RETURNS:                     1 if an error occured during initialization
** RETURNS:                     2 if a network error occured
** RETURNS:                     3 in case of server wrong reply
** RETURNS:                     4 in case of dlsym error
**
*****************************************************************************/
int SetValue(long *nb_val, long *flag_compress)

{
  int *select_var,*float_var;
  int i, j, nb_util, Pvar, Nvar, nb_var, flag, GTXState;
  char **variables;
  double test_val;

  int (*GTXClientGetVariableList)(int *, char ***);
  int (*GTXClientSetVariable)(char *);
  int (*GTXClientGetVariableInfo)(GTXVariableInfo *);
  int (*GTXClientSetSelection)(char *);

  GTXVariableInfo TGTXVariableInfo;

  variables = NULL;
  *flag_compress = 0;
  
  GTXClientGetVariableList = (int (*)(int *, char ***)) 
    dlsym(handle_lib, "GTXClientGetVariableList");
  if (GTXClientGetVariableList == NULL) return(-1);
  
  if((GTXState = GTXClientGetVariableList(&nb_var, &variables)) != 0)
    return(GTXState);

  /* No variable available in the file */

  if(nb_var == 0) 
  {
    printf("No variable in this file");
    return(-1);
  }
  
  GTXClientSetVariable = (int (*)(char *)) 
    dlsym(handle_lib, "GTXClientSetVariable");
  if (GTXClientSetVariable == NULL) return(-1);
  
  /* Core allocation */
  
  select_var = (int *) malloc(sizeof(int) * nb_var);
  float_var = (int *) malloc(sizeof(int) * nb_var);
  
  /* Look for numeric variables */
  
  for (i=0; i<nb_var; i++)
  {      
    GTXClientSetVariable = (int (*)(char *)) 
      dlsym(handle_lib, "GTXClientSetVariable");
    if (GTXClientSetVariable == NULL) return(-1);
    
    if((GTXState = GTXClientSetVariable(variables[i])) != 0)
      return(GTXState);   
    
    GTXClientGetVariableInfo = (int (*)(GTXVariableInfo *)) 
      dlsym(handle_lib, "GTXClientGetVariableInfo");
    if (GTXClientGetVariableInfo == NULL) return(-1);
    
    if((GTXState = GTXClientGetVariableInfo(&TGTXVariableInfo)) != 0)
      return(GTXState);   
    
    /* Check the variable type */
    
    if(TGTXVariableInfo.bit_length == 1 && 
       TGTXVariableInfo.type != GTX_VAR_TYPE_CHAR)
    {
      select_var[i] = 1;
      float_var[i] = 0;
    }
    
    if(TGTXVariableInfo.type != GTX_VAR_TYPE_CHAR && 
       TGTXVariableInfo.bit_length != 1)
    {
      select_var[i] = 0;
      float_var[i] = 1;
    }      
  }
  
  /***************************/
  /* Choice of the variables */
  /***************************/
  
  printf("\nVariable List\n");
  printf("Number | Variable Name\n");
  printf("-------|--------------\n");
  for(i=j=0; i<nb_var;i++) 
  {
    if(float_var[i] != 1) continue;
    j++;
    printf("%6d | %s\n",j,variables[i]);
  }
  nb_util = j;
  
  do
  {
    printf("Enter the variable number (between 1 and %d): ",nb_util);
    scanf("%d", &Pvar);
  } while(Pvar < 1 || Pvar > nb_util);
  
  for(i=j=0; i<nb_var;i++) 
  {
    if(float_var[i] != 1) continue;
    j++;
    if (j == Pvar) 
    {
      Nvar = i;
      break;
    }
  }
  
  GTXClientSetVariable = (int (*)(char *)) 
    dlsym(handle_lib, "GTXClientSetVariable");
  if (GTXClientSetVariable == NULL) return(-1);
  
  printf("Selected Variable is %s\n", variables[Nvar]);
  if((GTXState = GTXClientSetVariable(variables[Nvar])) != 0) return(GTXState);
  
  /********************/
  /* Selection choice */
  /********************/
  
  printf("\nSelection List\n");
  printf("Number | Selection Name\n");
  printf("-------|---------------\n");
  printf("     0 | (no selection)\n");
  for(i=j=0; i<nb_var;i++)
  {
    if(select_var[i] != 1) continue;
    j++;
    printf("%6d | %s\n",j,variables[i]);
  }
  nb_util = j;
  
  do
  {
    printf("Enter the selection number (between 0 and %d): ",nb_util);
    scanf("%d", &Pvar);
  } while(Pvar < 0 || Pvar > nb_util);
  
  if (Pvar > 0)
  {
    
    /* Case when a selection is selected */
    
    for(i=j=0; i<nb_var;i++) 
    {
      if(select_var[i] != 1) continue;
      j++;
      if (j == Pvar) 
      {
        Nvar = i;
        break;
      }
    }
    
    GTXClientSetSelection = (int (*)(char *)) 
      dlsym(handle_lib, "GTXClientSetSelection");
    if (GTXClientSetSelection == NULL) return(-1);
    
    if((GTXState = GTXClientSetSelection(variables[Nvar])) != 0)
      return(GTXState);
    
    printf("\n");
    printf("A selection has been required\n");
    printf("Should the masked samples be skipped     : enter 1\n");
    printf("or be replaced by the value [1.2345e_308]: enter 0\n");
    do
    {
      scanf("%d", flag_compress);
    } while(*flag_compress != 0 && *flag_compress != 1);
  }
  else
  {
    
    /* No selection */
    
    *flag_compress = 0;
  }
  
  /* Core deallocation */
  
  free(variables);
  free(select_var);
  free(float_var);
  
  /* Get the count of active samples */
  
  if(GTXState = S_GTXClientReadVarCount(flag_compress, nb_val) != 0)
    return(GTXState);
  printf("Count of active values = %d\n",(int) (*nb_val));
  return(0);
}

/*****************************************************************************
**
** FUNCTION:       stCreateVariable
**
** DESCRIPTION:    create a new isatis variable
**
** RETURNS:        int error_code: 0 if ok,
**                                1 if an error occured during initialization
**                                2 if a network error occured
**                                3 in case of server wrong reply
**
** IN_ARGS:        char *var_name name of variable to create
**                 int bitlength
**                 char *format_unit format of the new variable
** OUT_ARGS:
**
** REMARKS:
**
*****************************************************************************/
static void stCreateVariable(char *VarName, 
                             int   BitLength,
                             char *FormatUnit,
                             int  *retcode)

{
  int i,nb_var,choix,found,calc;
  char **var_list = NULL;
  GTXVariableInfo var_info;
  int (*GTXClientGetVariableList)(int *, char ***);
  int (*GTXClientSetVariable)(char *); 
  int (*GTXClientGetVariableInfo)(GTXVariableInfo *);
  int (*GTXClientNewDoubleVariable)(char *,int ,GTX_FORMAT_TYPE,
                                    int, int, char *);
  GTX_FORMAT_TYPE FormatType = GTX_FORMAT_TYPE_DECIMAL;
  int FormatLength = 10;
  int FormatDigits = 2;
  static int FlagOverwrite = 0;
  
  /* Initializations */
  *retcode = 1;
  
  /* Bit Length */
  do
  {
    calc=1;
    if(calc != BitLength)  
      do
      {
        calc=calc*2;            
      }
      while(BitLength!=calc && calc<=64);
    if(BitLength!=calc) 
    {
      printf("wrong bit length:%d\nenter new bit length: ",BitLength);
      scanf("%d",&BitLength);
    }
  }
  while(BitLength !=calc);
  
  /* Format Unit */
  
  if(! strcmp(FormatUnit,""))
  {
    do
    {
      printf("Enter the Format type:\n");
      printf(" 1 - Length Variable\n");
      printf(" 2 - Specific Format\n");
      scanf("%d",&choix);
    }
    while(choix != 1 && choix != 2);
    if(choix == 2)
    {
      printf("Enter the specific format : ");
      scanf("%s",FormatUnit);
    }
    else
    {
      (void) strcpy(FormatUnit,NULL);
    }
  }
  
  /* Get the list of the existing variables */
  
  GTXClientGetVariableList = (int (*)(int *, char ***)) 
    dlsym(handle_lib, "GTXClientGetVariableList");
  if (GTXClientGetVariableList == NULL) return;
  if ((GTXState = GTXClientGetVariableList(&nb_var,&var_list)) != 0) return;
  
  /* Check if the variable already exists */
  
  for(i=0,found= -1; i<nb_var && found<0;i++)
    if (! strcmp(VarName,var_list[i])) found = i;
  
  if (found > 0)
  {

    /* The variable already exists */
    
    GTXClientSetVariable = (int (*)(char *)) 
      dlsym(handle_lib, "GTXClientSetVariable");
    if (GTXClientSetVariable == NULL) return;
    GTXClientSetVariable(VarName);
    
    GTXClientGetVariableInfo = (int (*)(GTXVariableInfo *)) 
      dlsym(handle_lib, "GTXClientGetVariableInfo");
    if (GTXClientGetVariableInfo == NULL) return;
    GTXClientGetVariableInfo(&var_info);
    
    /* Dispatch according to the existing variable type */
    
    switch(var_info.type)
    {
      case GTX_VAR_TYPE_FLOAT: 
      {
        if(var_info.bit_length != BitLength) 
        {
          printf("\n");
          printf("The variable %s already exists\n",VarName);
          printf("Its bitlength (%d) is different from the new one (%d)\n",
                 var_info.bit_length,BitLength);
          
          /* If the application does not overwrite variable: it is an error */
          
          if (! FlagOverwrite)
          {
            printf("The current version of the interface does not allow\n");
            printf("the deletion of an already existing variable\n");
            printf("The WRITE action is cancelled\n");
            goto label_end;
          }
          do {
            printf("Enter 0 to quit or 1 to overwrite the old bitlength\n");
            scanf("%d",&choix);
          } while (choix != 0 && choix != 1);
          if (choix == 1) found = -1;
          break;
        }
        else
          found = -2;
        break;     
      }
      
      case GTX_VAR_TYPE_XG:
      case GTX_VAR_TYPE_YG:
      case GTX_VAR_TYPE_ZG: 
      {
        printf("You may not overwrite the READONLY variable\n",VarName);
        break; 
      }

      case GTX_VAR_TYPE_MACRO: 
      {
        printf("You are about to overwrite a MACRO variable\n");
          
        /* If the application does not overwrite variable: it is an error */
        
        if (! FlagOverwrite)
        {
          printf("The current version of the interface does not allow\n");
          printf("the deletion of an already existing variable\n");
          printf("The WRITE action is cancelled\n");
          goto label_end;
        }
        do {
          printf("Enter 0 to quit or 1 to overwrite the MACRO variable\n");
          scanf("%d",&choix);
        } while (choix != 0 && choix != 1);
        if (choix == 1) found = -1; 
        break;
      }
    }
  }
  
  /* Abort */
  
  if (found >= 0)
  {
    printf("\nThe Write operation is cancelled\n");
    goto label_end;
  }
  
  /* Create a new variable (if necessary) */
  
  if (found == -1) 
  {
    printf("Creation de la variable\n");
    GTXClientNewDoubleVariable = (int (*)(char *,int ,GTX_FORMAT_TYPE,
                                          int,int,char *)) 
      dlsym(handle_lib, "GTXClientNewDoubleVariable");
    if (GTXClientNewDoubleVariable == NULL) return;
    if((GTXState = GTXClientNewDoubleVariable(VarName,BitLength,FormatType,
                                              FormatLength,FormatDigits,
                                              FormatUnit)) != 0)
    {
      printf("\nCreation failed.\n");
      goto label_end;
    }
  }
  
  GTXClientSetVariable = (int (*)(char *)) 
    dlsym(handle_lib, "GTXClientSetVariable");
  if (GTXClientSetVariable == NULL) return;
  GTXClientSetVariable(VarName);
  *retcode = 0;
  return;
  
label_end:                   /*liberation de la memoire*/
  if(var_list != NULL)
  {
    for(i=0;i<nb_var;i++) free(var_list[i]);
    free(var_list);
  }
  
  return;
}  

/*****************************************************************************
**
** FUNCTION:       S_GTXClientNewVariable
**
** DESCRIPTION:    "transtypage entre splus et isatis"
**
** RETURNS:        int error_code: 0 if ok,
**                                1 if an error occured during initialization
**                                2 if a network error occured
**                                3 in case of server wrong reply
**
** IN_ARGS:        char **var_name name of variable to create
**                 long *bitlength
**                 char **format_unit format of the new variable
** OUT_ARGS:
**
** REMARKS:
**
*****************************************************************************/
void S_GTXClientNewVariable(char **var_name,
                            long  *bit_length,
                            char **format_unit,
                            int   *retcode)
{
  /* Initializations */

  stCreateVariable(*var_name,(int)*bit_length,*format_unit,retcode);

  return;
}

/*****************************************************************************
 **
 ** FUNCTION:       stWriteVariable
 **
 ** DESCRIPTION:    write a splus variable in the new isatis variable
 **
 ** RETURNS:        int error_code: 0 if ok,
 **                                1 if an error occured during initialization
 **                                2 if a network error occured
 **                                3 in case of server wrong reply
 **
 ** IN_ARGS:        double *dbl_data array with the splus variable
 **                 int nb_val number of samples
 **                
 ** OUT_ARGS:
 **
 ** REMARKS:
 **
 ****************************************************************************/
static int stWriteVariable(double *dbl_data, 
                           int     nb_val)

{
  int (*GTXClientWriteDoubleVariable)(int, double,  int, double *); 
  int flag_compress = 0;
  double test_val = 0;
  
  /* The variable exists: Copie its contents */

  GTXClientWriteDoubleVariable = (int (*)(int, double, int, double *))
    dlsym(handle_lib, "GTXClientWriteDoubleVariable");
  if (GTXClientWriteDoubleVariable == NULL) return(-1);
  
  if (GTXState = GTXClientWriteDoubleVariable(flag_compress,test_val,
                                              nb_val,dbl_data))
    {
      printf("Write variable failed.\n");
      return(GTXState);
    }
  
  return(0);
}
  
/*****************************************************************************
 ** FUNCTION:       S_GTXClientWriteVariable
 **
 ** DESCRIPTION:    "trasntypage entre splus et isatis"
 **
 ** RETURNS:        int error_code: 0 if ok,
 **                    1 if an error occured during initialization
 **                       2 if a network error occured
 **                       3 in case of server wrong reply
 **
 ** IN_ARGS:        double *dbl_data array with the splus variable
 **                 double * nb_val number of samples
 ** OUT_ARGS:
 **
 ** REMARKS:
 **
 ****************************************************************************/
int S_GTXClientWriteVariable(double *db_data, 
                             int    *nb_val)

{
  int i;

  if ((GTXState = stWriteVariable(db_data, (int) *nb_val)) != 0)
    return(GTXState);
  
  return(0);
}

