/*****************************************************************************
 
            Copyright (c)2009 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

/* $Id: GTXClient.h 25941 2016-04-05 15:10:18Z foucher $ */

/*!
 * \file GTXClient.h
 * \brief GTXserver C API
 */

/*!
 * \mainpage GTXClient C API
 *
 * \section what What are GTXserver and GTXClient ?
 *
 * You should first read the
 * <a href="../index.html" target="_blank">introduction</a> to know it.
 *
 * \section rn Where can I find GTXserver Release Notes ?
 *
 * GTXserver Release Notes are only available on Geovariances Website in the Software / Free Software / GTXserver section.
 *
 * \section detail GTXClient API
 *
 * The main API is described in the documentation of the main include file:
 * \ref GTXClient.h.
 *
 * \section new_lines_api New Lines API
 *
 * Isatis version 9.0 now has a separate format format for Gravity Lines & Core Lines.
 * GTXserver version 9.0 also has the distiction in the API. To Create a Lines file, you must use
 * GTXClientNewLinesFile then add lines in it using: GTXClientLinesFileAddGravityLine for gravity lines
 * and GTXClientLinesFileAddCoreLine for core lines.
 * If you were using previous version of the API, you may have been using GTXClientLinesFileAddLineCores
 * which has been removed. If you really want to quick port to the new version, simply use
 * GTXClientLinesFileAddLineCoresOld but be aware that this creates old style core lines that the user
 * will have to convert in Isatis. You should rather use the new APIs indicated above.
 *
 * Examples of usage of the C API can be found here: \ref cexample.
 *
 * \section palettes_cscale Palettes & Color Scales API
 *
 * GTXserver version 2013.2 now handles Palettes & Color Scales. To check if server can handle reading
 * and writing to Palettes and Color Scales, use \ref GTXClientServerHasFeature.
 *
 * Examples of usage of the C API can be found here: \ref cexample.
 *
 * \section get_line_name Line Name Variable
 *
 * GTXserver version 2013.3 now returns the name of the Line Name variable for the Lines Header Points file inside its
 * GTXFileInfo.
 */

/*!
 * \page cexample
 *
 * \section Cexample Examples written in C
 *
 * Once compiled and linked, these examples can be run at the shell prompt. All
 * of them are clients for the GTXserver. Each client will either run a new server
 * or connect to an existing one depending on command line arguments and or choices
 * in the interface when possible (motif_client for example).
 *
 * \subsection text_list text_list.c example:
 * \ref sample_text_list dumps all Isatis studies, directories, files and
 * variables.
 *
 * This is a simple C program with no user interactions. 
 * 
 * \subsection text_client text_client.c example:
 * \ref sample_text_client is more difficult to understand as it is menu
 * driven. The user asks for a connection, selects a study, a directory... and
 * retrieves information on the variables of his choice. 
 *
 * \subsection motif_client motif_client.c example:
 * \ref sample_motif_client has a complete user interface but needs the Motif
 * libraries to be built. Here again, the user chooses what he wants to read
 * but in a friendlier way due to the Motif interface. The interface allows
 * to run a local GTXserver or to connect to a pre-existing one.
 *
 * \subsection vendor_data VendorData example:
 * This example will show you how to store specific data that are not handled
 * directly by Isatis. For example you can use \a Vendor \a Data to store
 * data between import and export phase to keep consistency. See
 * \link vendor_data_read.c vendor_data_read.c \endlink and
 * \link vendor_data_write.c vendor_data_write.c \endlink.
 *
 * \subsection write_files write_files.c example:
 * This example will show you how to write standard files (Grid/Points/Lines)
 * into Isatis file system.
 * See \link write_files.c write_files.c \endlink.
 *
 * \subsection write_fault write_fault.c example:
 * This example will show you how to write a grid and a fault associated to
 * this file into the Isatis file system.
 * See \link write_fault.c write_fault.c \endlink.
 *
 * \subsection read_polygon read_polygon.c example:
 * This example will show you how to read polygon systems from the Isatis file
 * system.
 * See \link read_polygon.c read_polygon.c \endlink.
 *
 * \subsection write_polygon write_polygon.c example:
 * This example will show you how to write a polygon system to the Isatis file
 * system.
 * See \link write_polygon.c write_polygon.c \endlink.
 * 
 * \subsection write_color_scale write_color_scale.c example:
 * This example shows the Isatis 2013 access to palettes and color scales.
 * See \link write_color_scale.c write_color_scale.c \endlink.
 *
 */

/*! \example motif_client.c */
/*! \example read_polygon.c */
/*! \example text_client.c */
/*! \example text_list.c */
/*! \example vendor_data_read.c */
/*! \example vendor_data_write.c */
/*! \example write_fault.c */
/*! \example write_files.c */
/*! \example write_polygon.c */
/*! \example write_color_scale.c */

/*!
 * \page GTXserver
 * 
 * \section GTXserver GTXserver
 *
 * This part describes the server. The user must remember that this is a
 * client/server application. A client (which can be modified by the user) will
 * talk (using sockets) with a server and through this server will get access to
 * the Isatis file system without running the Isatis application.
 *
 * The server can be mono-session or multi-session.

 * If started mono-session, the server waits for a client connection. The
 * client connects and obtain an exclusive communication channel with the
 * server. It does what it needs with the server and disconnects. At this
 * moment, the server exits.
 *
 * If started multi-session (-multisession flag), the server waits for a client
 * connection. When a client connects, the server forks (duplicates itself)
 * and open a new port. It gives the new port to the client. The client closes
 * connection with the master GTXserver and connects to its child. The master
 * server it still available for another connection during the transaction
 * between the client and the child server. When the client disconnects, the
 * child server stops but the master one remains there.
 *
 * To avoid making your application's user run a GTXserver each time you need
 * to read/write Isatis filesystem, you can use the C API function
 * GTXClientRunGTXServer. It will locate the installed Isatis and run the
 * appropriate GTXserver.
 *
 * But with this, you can only run local GTXservers. If one needs to read data
 * on another machine, this wouldn't work. So you should include in your
 * application, an option to run a local GTXserver or to use an already
 * running one given the host and port the server is running on. The user can
 * then run GTXserver each time it is needed on the "data" machine or use a
 * multi-session one to avoid this.
 *
 * Both mono and multi sessions have advantages and drawbacks:
 *
 * A mono session server will have to be restarted after each disconnect. It
 * can only answer one client at a time. If you need to run several clients,
 * you need to run several servers on several ports. However it is quite
 * simple to use and when ran by the client there is no problem with
 * filesystem access rights. The client starts a server with the same rights
 * as an Isatis run.
 *
 * A multi session server runs as a given user which may not be the one running
 * the client. In an heterogeneous environment like reading data with the
 * GTXserver on UNIX and a client on a PC, the mapping between users may not
 * be easy and would require administrator privileges. For now, when the server
 * forks, it is still running as the same user so it will have its rights to
 * read files from the database and will create file with its own user ID
 * whatever the client user ID is.
 *
 * Here is a list of the options available when running GTXserver:
 *
 * <TABLE>
 * <TR><TD>-help</TD><TD>Prints help on command usage</TD></TR>
 * <TR><TD>-debug</TD><TD>Prints client/server messages. Interesting to follow
 * the dialog between the server and its client.</TD></TR>
 * <TR><TD>-fork</TD><TD>Start the server in background</TD></TR>
 * <TR><TD>-port port_number</TD><TD>Port used to communicate with the client.
 * The default is the same for the server and the client. If for any reason
 * the default port number is not correct on your machine, you can change it
 * (you must use the same port number in the server and the client).</TD></TR>
 * <TR><TD>-multi[session]</TD><TD>Start the server as multi-session. The
 * server will create a new process for each connection</TD></TR>
 * </TABLE>
 */

/*!
 * \page text_list
 *
 * \section sample_text_list text_list.c example
 *
 * See \link text_list.c text_list.c \endlink source code.
 *
 * This example is the simpliest way to understand the communication between
 * GTXserver and a client. The program scrolls through the Isatis file system
 * to get information on all files and variables.
 * 
 * This example does not read any Isatis variable and you should refer to next
 * examples to know more about that point.
 * 
 * Here is a list of the options available when running text_list:
 * 
 * <TABLE>
 * <TR><TD>-help</TD><TD>Prints help on command usage</TD></TR>
 * <TR><TD>-debug</TD><TD>Prints client/server messages. Interesting to follow
 * the dialog between the server and its client.</TD></TR>
 * <TR><TD>-host hostname</TD><TD>Server's hostname. If not specified, the
 * default host is "localhost"..</TD></TR>
 * <TR><TD>-port port_number</TD><TD>Port used to communicate with the client.
 * The default is the same for the server and the client. If for any reason
 * the default port number is not correct on your machine, you can change it
 * (you must use the same port number in the server and the client).</TD></TR>
 * <TR><TD>-path data_path</TD><TD>Asks the server to use this data path to
 * access to the Isatis file system. The default is set to the \$GTX_DATA
 * environment variable if defined.<TD></TR>
 * <TR><TD COLSPAN=2>The default is to run a new server on a random port.<BR>
 * If -host or -port is specified, a server must have been started first on
 * the given host/port</TD></TR>
 * </TABLE>
 * 
 * You can try to run the client to see what happens. If you examine the
 * source code, you will find the main calls used to write a client
 * application. This source code is only composed of two functions: st_usage
 * and main. The first one only gives the usage of the program. 
 * 
 * During the arguments parsing, if an error occurs, the program uses the
 * function GTX_ERROR_FUNCTION(). The default behaviour of this function is to
 * simply call fprintf on stderr. This function will be overriden in the
 * motif_client example
 * 
 * The first thing is to initialize the library calling:
 * GTXClientInitialize(GTX_CLIENT_VERSION).
 *
 * With the option -debug, the program calls GTXClientDebug() with 1. Then all
 * the functions will print extra messages about the dialog between server and
 * client. This can be useful in a debug phase.
 * 
 * The default is then to run a new GTXserver on a random port. The function
 * will look for the server executable in the following places:
 * - the contents of GTX_SERVER environment variable
 * - the GTX_HOME environment variable (mostly used under UNIX to indicate
 * where Isatis is installed) or a registry key under Windows.
 * - through the PATH.
 * 
 * To connect to the server, the client calls GTXClientConnect(). The server
 * must already be running and waiting for connection. If no -port or -host
 * option has been used, it has been started just above, so it should be OK.
 * Otherwise, the server must already be running on the given host/port.
 * If not, the function will return 1 and send a message with:
 * GTX_ERROR_FUNCTION().
 *
 * If the server has been started with the -multisession parameter, the
 * server will still be available for other sessions during this session
 * lifetime. Moreover, when GTXClientDisconnect will be called, the server
 * will still be there...
 * 
 * As we are connected we can get the list of studies using
 * GTXClientGetStudyList(). This function returns the number of studies. Like
 * all other functions, if a problem occured, an error code is returned and a
 * message is sent using GTX_ERROR_FUNCTION().
 * 
 * We then set the name of the current study using GTXClientSetStudy() For
 * each study we can obtain the directory list with
 * GTXClientGetDirectoryList(). and we set the current directory using
 * GTXClientSetDirectory() for each directory of the list.
 * 
 * After getting the file list of each directory using GTXClientGetFileList(),
 * we must set as current each file with GTXClientSetFile(). 
 * 
 * At this point, we try to obtain file information using
 * GTXClientGetFileInfo(). We then dump the contents of the GTXFileInfo
 * structure returned by this function.
 * 
 * After that, we get the variable list with a call to
 * GTXClientGetVariableList() and define each variable as the current one with
 * GTXClientSetVariable().
 * 
 * After getting variable information through GTXClientGetVariableInfo(). We then
 * dump the contents of the GTXVariableInfo structure returned by this
 * function.
 * 
 * If the current variable is a macro variable, we get macro indices using
 * GTXClientGetMacroIndicesWithAlpha() and dump them. Note: Macro variables
 * appears in Isatis 3.2 and can have alphanumerical indices since Isatis 5.0.
 * 
 * The end of the program mainly consists in freeing memory allocated by
 * GTXserver functions.
 * 
 * Just before exiting, we must disconnect from the server using
 * GTXClientDisconnect(). If the server is mono session, it exits.
 * If the server is multi session, only the current session terminates and
 * GTXserver is still available for a further session.
 */

/*!
 * \page text_client
 *
 * \section sample_text_client text_client.c example
 *
 * See \link text_client.c text_client.c \endlink source code.
 *
 * This example is an interative GTXserver client. A set of actions is
 * available each corresponding to a call to the server.
 * 
 * At this point, you are supposed to already know program arguments, the way
 * to connect to and disconnect from the server and how to scroll through the
 * Isatis filesystem. For this purpose you can refer to text_list example.
 * 
 * A global variable called mode represents the current state of the program.
 * At the beginning this mode is set to INIT to say that we have no connection
 * established.
 * 
 * st_menu() gives the list of available functions and gets user answer. Each
 * answer is followed by a call to the corresponding function. The "RUN" action is
 * used to run a server (using GTXClientRunGTXserver) on a random port and if it
 * succeeds, connects to it. The "CONN" action is used to make the connection
 * to an already running server and get the study list.
 * The user can then use the "SET" action which sets an item as the current
 * study,directory,file or variable and gets the next level list(i.e if
 * setting a study, directory list is retrieved). The mode is then changed and
 * new actions may be available.
 * 
 * For example,if we have a file list, we can get file information. If we have
 * a variable list, and if there is a 1 bit float variable, we can set it as
 * the current selection (In this case further reads in this file will be
 * performed, excluding mask off samples). After setting a variable, we can
 * get variable information or read the variable...
 * 
 * If the variable you have set is a macro variable, you will need to give the
 * indice of interest before continuing. Getting variable information with
 * GTXClientGetVariableInfo() will give you the list of available indices. You
 * can set an indice using: GTXClientSetIndice().
 * 
 * To read a variable, you need to use the "STAT" action. If the variable is a
 * character variable, GTXClientReadCharVariable() is used to read the
 * variable. We also get the number of test values and the number of kept
 * samples if a selection is defined. We retrieve the same parameters while
 * reading float variables: GTXClientReadDoubleVariable(). After having read
 * the variable, some basic statistics are written.
 * 
 * The "DISC" action only disconnects the server, the "EXIT" action
 * disconnects and exits. Anyway, you have to remember that when you
 * disconnect, the server automatically goes down if it has been started
 * mono-session. You will not then be able to re-connect to the server
 * interactively, without running the server again first.
 */

/*!
 * \page motif_client
 *
 * \section sample_motif_client motif_client.c example
 *
 * See \link motif_client.c motif_client.c \endlink source code.
 *
 * This example is a GTXserver client with an Motif user interface. To be
 * compiled and linked, this program needs Motif includes and libraries.
 * Before reading this part, you should have read the two previous example
 * descriptions. This text only presents the main differences between
 * text_client example and this one.
 * 
 * After parsing arguments and initializing X Toolkit, the interface is
 * created in st_create_interf. It consists of two windows (forms). The first
 * presents parameters for connection. It allows to run a local GTXserver and
 * connect to it or connect to an already running server. In the second case,
 * you must indicate the machine and port the server is running on. The second
 * window/form masked at launch, contains the current list of elements
 * (studies,directories...) and zones where information appears. By the item
 * list, there is a second list for macro variable indices. After connection,
 * the two main forms are swapped, i.e: the first is popped down and the
 * second appears. The interface also contains a status zone.
 * 
 * Instead of the actions used in text_client, we will use motif callbacks in
 * the list to set current items and button callbacks for the other actions.
 * 
 * Just after the interface creation, GTX_ERROR_FUNCTION() is overriden to
 * redirect all errors in the status zone. To do so, a static function
 * st_status_function() is created. The string returned by the server is
 * always a single line string. This allows the string to be directly set in
 * the status zone.
 */

#ifndef _GTXClient_h
#define _GTXClient_h

/* Redefine names of C structures */
#ifndef DOXYGEN
#ifdef __USING_GTXSERVER_CPP_API__
# define GTXFault GTXFault_C
# define GTXFaultInfo GTXFaultInfo_C
# define GTXFaultSegment GTXFaultSegment_C
# define GTXFaultSystem GTXFaultSystem_C
# define GTXFileInfo GTXFileInfo_C
# define GTXPolygon GTXPolygon_C
# define GTXPolygonSystem GTXPolygonSystem_C
# define GTXVariableInfo GTXVariableInfo_C
# define GTXVendorData GTXVendorData_C
# define GTXWireframe GTXWireframe_C
# define GTXWireframeObject GTXWireframeObject_C
# define GTXWireframeSystem GTXWireframeSystem_C
# define GTXColorScale GTXColorScale_C
#else
# define GTXFault_C GTXFault
# define GTXFaultInfo_C GTXFaultInfo
# define GTXFaultSegment_C GTXFaultSegment
# define GTXFaultSystem_C GTXFaultSystem
# define GTXFileInfo_C GTXFileInfo
# define GTXPolygon_C GTXPolygon
# define GTXPolygonSystem_C GTXPolygonSystem
# define GTXVariableInfo_C GTXVariableInfo
# define GTXVendorData_C GTXVendorData
# define GTXWireframe_C GTXWireframe
# define GTXWireframeObject_C GTXWireframeObject
# define GTXWireframeSystem_C GTXWireframeSystem
# define GTXColorScale_C GTXColorScale
#endif
#endif /*DOXYGEN*/


#include "GTXOs.h"

/*!
 * \def GTX_CLIENT_VERSION
 * Version number of GTX client/server protocol
 * Major * 100000 + Minor * 1000 + Patch * 10 + Sub-Version
 */
#define GTX_CLIENT_VERSION 1300030

/*!
 * \enum GTX_FILE_TYPE
 * Define the different types of file that can be read from or written to the
 * Isatis file system.
 */
typedef enum
{
  GTX_FILE_TYPE_ANY = -1, /*!< Any kind of file (used only for queries) */
  GTX_FILE_TYPE_POINTS,   /*!< Points */
  GTX_FILE_TYPE_GRAVITY_LINES,    /*!< Lines  */
  GTX_FILE_TYPE_CORE_LINES,    /*!< Lines  */
  GTX_FILE_TYPE_GRID,     /*!< Grid   */
  GTX_FILE_TYPE_INVALID   /*!< Invalid type */
} GTX_FILE_TYPE;

/*!
 * \enum GTX_VAR_TYPE
 * Define the different types of variables that can be read from or written to the
 * Isatis file system.
 */
typedef enum
{
  GTX_VAR_TYPE_SELECTION = -3, /*!< Only selection variables (used only for queries) */
  GTX_VAR_TYPE_FLOAT_ALL = -2, /*!< Any float variable (used only for queries) */
  GTX_VAR_TYPE_ANY = -1, /*!< Any kind of variable (used only for queries) */
  GTX_VAR_TYPE_FLOAT,  /*!< Number variable */
  GTX_VAR_TYPE_CHAR,   /*!< String variable */
  GTX_VAR_TYPE_XG,     /*!< X coordinate variable */
  GTX_VAR_TYPE_YG,     /*!< Y coordinate variable */
  GTX_VAR_TYPE_ZG,     /*!< Z coordinate variable */
  GTX_VAR_TYPE_MACRO,  /*!< Macro variable (macro variables are always numbers */
  GTX_VAR_TYPE_INVALID /*!< Invalid type */
} GTX_VAR_TYPE;

/*!
 * \enum GTX_FORMAT_TYPE
 * Define the different types of printing formats.
 * With <B>a</B> the length of the format and <B>b</B> the number of digits.
 * \sa printf
 */
typedef enum
{
  GTX_FORMAT_TYPE_DECIMAL,  /*!< %%25.<B>b</B>lf */
  GTX_FORMAT_TYPE_INTEGER,  /*!< %%d */
  GTX_FORMAT_TYPE_EXPONENT, /*!< %<B>a</B>.<B>b</B>e */
  GTX_FORMAT_TYPE_FREE,     /*!< %<B>a</B>lg if a <= 13 & %<B>a</B>.<B>(a-7)</B>lg if a > 13 */
  GTX_FORMAT_TYPE_SHIFTED,  /*!< %f */
  GTX_FORMAT_TYPE_INVALID   /*!< Invalid type */
} GTX_FORMAT_TYPE;

/*!\brief Information on \a Fault \a System associated with a given file.
 */
typedef struct
{
  int faults_2d;       /*!< Faults are defined in 2D */
  int aux_var_used;    /*!< Auxiliary variables are used */
  int faults_number;   /*!< Total Number of faults */
  int segments_number; /*!< Total Number of segments */
  int min_fpriority;   /*!< Minimum fault priority */
  int max_fpriority;   /*!< Maximum fault priority */
  int auth_priority;   /*!< Authorized priority (minimum taken into account) */
} GTXFaultInfo;

/*!
 * \brief Information on a \a File.
 * \sa GTXClientGetFileInfo
 */
typedef struct
{
  GTX_FILE_TYPE type;      /*!< file type */
  char linked_file[50];    /*!< name of linked file (if non applicable the string is empty) */
  int faulted;             /*!< is this file faulted or not \arg \c 1 faulted \arg \c 0 not faulted */
  int polygon_file;        /*!< is this file a polygon file or not
                                \arg \c 1 is a polygon file
                                \arg \c 0 not a polygon file */
  int dimension;           /*!< dimension of the file \arg \c 2 for a 2D File
                                                      \arg \c 3 for a 3D File */
  gtx_long s_number;       /*!< give the number of samples in the file */
  int item_number;         /*!< give the number of lines in the file (if a \a Line file) */
  double X0;               /*!< X coordinate of the origin of the grid (if a \a Grid file) */
  double Y0;               /*!< Y coordinate of the origin of the grid (if a \a Grid file) */
  double Z0;               /*!< Z coordinate of the origin of the grid (if a \a Grid file) */
  double DX;               /*!< X mesh dimensions (if a \a Grid file) */
  double DY;               /*!< Y mesh dimensions (if a \a Grid file) */
  double DZ;               /*!< Z mesh dimensions (if a \a Grid file) */
  int NX;                  /*!< X number of grid nodes (if a \a Grid file) */
  int NY;                  /*!< Y number of grid nodes (if a \a Grid file) */
  int NZ;                  /*!< Z number of grid nodes (if a \a Grid file) */
  int rotation;            /*!< \arg \c 0 no grid rotation (if a \a Grid file)
                                \arg \c 1 grid rotation around the grid origin (if a \a Grid file) */
  double angle_z;          /*!< grid rotation angle around OZ (if a 3D file) */
  double angle_y;          /*!< grid rotation angle around OY (if a 3D file) */
  double angle_x;          /*!< grid rotation angle around OX */
  char sn_var_name[50];    /*!< \a Sample \a Number variable name */
  char ln_var_name[50];    /*!< \a Line \a Number variable Name (if a \a Line file) */
  char lname_var_name[50]; /*!< \a Line \a Name variable Name (if a \a Points File linked to a \a Lines file) */
  char rn_var_name[50];    /*!< \a Relative \a Number variable name (if a \a Grid file) */
  char xg_var_name[50];    /*!< \a X \a Gravity \a Center variable name */
  char yg_var_name[50];    /*!< \a Y \a Gravity \a Center variable name */
  char zg_var_name[50];    /*!< \a Z \a Gravity \a Center variable name (if a 3D File) */
  char xb_var_name[50];    /*!< \a X \a Core \a Begin variable name (if a \a Core lines file) */
  char yb_var_name[50];    /*!< \a Y \a Core \a Begin variable name (if a \a Core lines file)*/
  char zb_var_name[50];    /*!< \a Z \a Core \a Begin variable name (if a \a 3D Core lines file) */
  char xe_var_name[50];    /*!< \a X \a Core \a End variable name (if a \a Core lines file)*/
  char ye_var_name[50];    /*!< \a Y \a Core \a End variable name (if a \a Core lines file) */
  char ze_var_name[50];    /*!< \a Z \a Core \a End variable name (if a \a 3D Core lines file) */
  GTXFaultInfo fault_info; /*!< fault information (if the file is faulted) */
} GTXFileInfo;

/*!
 * \brief Information on a \a Variable.
 * \sa GTXClientGetVariableInfo
 */
typedef struct
{
  GTX_VAR_TYPE type;    /*!< variable type */
  int bit_length;       /*!< precision of the variable, can be 1, 2, 4, 8, 16,
                          32 or 64 bits. A selection variable is also known as
                          a 1 bit float variable */
  int alpha_length;     /*!< number of characters (only applicable for string
                          variables) */
  int implicit;         /*!< indicates that the variable is implicit in
                          Isatis file system */
  int i_number;         /*!< number of variables (indices) stored in a macro
                          variable */
  char unit[11];        /*!< unit string */
  char unit_symbol[11]; /*!< Additionnal Symbol (used for length) */
  int length_var;       /*!< 1 if the variable is a length variable. */
} GTXVariableInfo;

/*!
 * \brief Description of a \a Fault \a Segment
 */
typedef struct
{
  double X1;    /*!< X coordinate of the first point of the fault segment */
  double Y1;    /*!< Y coordinate of the first point of the fault segment */
  double Z1;    /*!< Z coordinate of the first point of the fault segment */
  double X2;    /*!< X coordinate of the second point of the fault segment */
  double Y2;    /*!< Y coordinate of the second point of the fault segment */
  double Z2;    /*!< Z coordinate of the second point of the fault segment */
  double X3;    /*!< X coordinate of the third point of the fault triangle (if the fault is a 3D fault) */
  double Y3;    /*!< Y coordinate of the third point of the fault triangle (if the fault is a 3D fault) */
  double Z3;    /*!< Z coordinate of the third point of the fault triangle (if the fault is a 3D fault) */
  double V1;    /*!< Auxilliary variable associated with the first point */
  double V2;    /*!< Auxilliary variable associated with the second point */
  double V3;    /*!< Auxilliary variable associated with the third point */
  int priority; /*!< Segment priority in the fault system (1 to 127) */
} GTXFaultSegment;

/*!
 * \brief Description of a \a Fault
 *
 * A \a Fault is made of several \a Fault \a Segments.
 * \sa GTXFaultSegment
 */
typedef struct
{
  char name[8];              /*!< Fault name */
  int  polygon;              /*!< Is the \a Fault polygonal (closed) ? \arg 1 = yes \arg 0 = no */
  int  n_segments;           /*!< Number of segments in the fault */
  GTXFaultSegment *segments; /*!< Segments array */
} GTXFault;

/*!
 * \brief Description of a \a Fault \a System
 *
 * A \a Fault \a System describes the faults associated with a \a File. It is made of
 * several \a Faults. All the faults are either 2D or 3D, the \a Fault \a System cannot
 * mix the dimensions.
 *
 * \sa GTXFault
 * \sa GTXClientGetFileInfo
 */
typedef struct
{
  int faults_2d;       /*!< Is the fault system defined in 2D or in 3D ? */
  int aux_var_used;    /*!< Are the auxiliary variables used ? */
  double aux_var_test; /*!< Test value for Auxiliary variables */
  int n_faults;        /*!< Number of faults in the fault system */
  GTXFault *faults;    /*!< Fault array */
} GTXFaultSystem;

/*!
 * \brief Description of a \a Polygon
 *
 * A \a Polygon
 */
typedef struct
{
  char name[16];       /*!< Polygon name */
  char group_name[16]; /*!< Polygon group name */
  double z_min;        /*!< 3D polygon minimum */
  double z_max;        /*!< 3D polygon maximum */
  int  n_vertices;     /*!< Number of vertices in the polygon */
  double *vertices;    /*!< X,Y coordinates of each vertices (interleaved) */
} GTXPolygon;

/*!
 * \brief Description of a \a Polygon \a System
 *
 * A \a Polygon \a System describes an Isatis \a Polygon \a File. It is made of
 * several \a Polygons. All the polygons are either 2D or 3D, the \a Polygon \a System cannot
 * mix the dimensions.
 *
 * \sa GTXPolygon
 */
typedef struct
{
  int polygons_2d;      /*!< Is the polygon system defined in 2D or in 3D ? */
  double ext_min[3];    /*!< Minimum extension X, Y & Z */
  double ext_max[3];    /*!< Maximum extension X, Y & Z */
  gtx_long n_polygons;  /*!< Number of polygons in the polygon system */
  GTXPolygon *polygons; /*!< Polygon array */
} GTXPolygonSystem;

/*!
 * \brief Description of a \a Wireframe \a Object
 */
typedef struct
{
  int n_vertices;    /*!< Number of vertices of this object */
  double *vertices;  /*!< coordinates of the vertices, size is 3*nvertices, order is X,Y,Z for
                          first vertex, then X,Y,Z for second vertex... */
  int n_triangles;   /*!< Number of triangles of this object */
  int *triangles;    /*!< triangles forming this object. size is 3*triangles, order is Pt 1,2,3
                          for first triangle, then Pt 1,2,3 for second triangle... */
} GTXWireframeObject;

/*!
 * \brief Description of a \a Wireframe
 *
 * A \a Wireframe is made of several \a Wireframe \a Objects.
 * \sa GTXWireframeObject
 */
typedef struct
{
  char name[50];               /*!< Wireframe name */
  int  color[3];               /*!< RGB color of this wireframe */
  int  n_objects;              /*!< Number of objects in the wireframe */
  GTXWireframeObject *objects; /*!< Objects array */
} GTXWireframe;

/*!
 * \brief Description of a \a Wireframe \a System
 *
 * A \a Wireframe \a System describes the wireframes associated with a \a File. It is made of
 * several \a Wireframes. All the wireframes are 3D.
 *
 * \sa GTXWireframe
 * \sa GTXClientGetFileInfo
 */
typedef struct
{
  int n_wireframes;         /*!< Number of wireframes in the wireframe system */
  GTXWireframe *wireframes; /*!< Wireframe array */
} GTXWireframeSystem;

/*!
 * \enum GTX_CSCALE_TYPE
 * Define the different types of color scales that can be read from or written to the
 * Isatis file system.
 */
typedef enum
{
  GTX_CSCALE_TYPE_AUTOMATIC,   /*!< Automatic */
  GTX_CSCALE_TYPE_ISOFREQ,     /*!< IsoFrequencies */
  GTX_CSCALE_TYPE_MANUAL,      /*!< Manual */
  GTX_CSCALE_TYPE_FACIES,      /*!< Facies */
  GTX_CSCALE_TYPE_INVALID      /*!< Invalid type */
} GTX_CSCALE_TYPE;

/*!
 * \brief Color Scale details.
 * \sa GTXClientReadColorScale
 */
typedef struct
{
  GTX_CSCALE_TYPE type;    /*!< color scale type */
  int readonly;            /*!< 0 for Read/Write Access, 1 for system/readonly color scales */
  char palette_name[50];   /*!< name of the used palette (for AUTO/ISOFREQ/MANUAL color scales) */
  int nclasses;            /*!< Number of classes        (for ISOFREQ/MANUAL/FACIES color scales) */
  int undef_col[3];        /*!< Undefined Color RGB      (for ALL color scales) */
  double* bounds;          /*!< Manual Bounds            (for MANUAL color scales) (has nclasses+1 values) */
  int* pal_ind;            /*!< Manual Palette Indices   (for MANUAL color scales) (has nclasses values from 0 to npalcol-1) */
  int lowval_col[3];       /*!< Lower Values Color RGB   (for MANUAL color scales) */
  int highval_col[3];      /*!< Upper Values Color RGB   (for MANUAL color scales) */
  int bnd_inc_low;         /*!< Lowest Bound Inclusion   (for MANUAL color scales) (can be 0=[, 1=], 2=expand) */
  int bnd_inc_inter;       /*!< Intermediate Bounds Incl.(for MANUAL color scales) (can be 0=[ [, 1=] ]) */
  int bnd_inc_high;        /*!< Highest Bounds Inclusion (for MANUAL color scales) (can be 0=[, 1=], 2=exp) */
  int other_col[3];        /*!< Other Values Color RGB   (for FACIES color scales) */
  int facies_int_nb;       /*!< Facies Interval Number   (for FACIES color scales) (nclasses values) */
  int facies_integers;     /*!< Facies Integer Mode      (for FACIES color scales) (0 for lower included,
			     upper excl, printed as double, 1 for both included, printed as integer */
  double* facies_int_min;  /*!< Facies Interval Minimum  (for FACIES color scales) (facies_int_nb values) */
  double* facies_int_max;  /*!< Facies Interval Maximum  (for FACIES color scales) (facies_int_nb values) */
  int* facies_class_ind;   /*!< Facies Interval Pal Index(for FACIES color scales) (facies_int_nb values, [0-ncl-1]) */
  int* facies_rgbs;        /*!< Facies RGBs              (for FACIES color scales) (order is RRRGGGBBB, with nclasses of each R,G and Bs) */
  char** facies_labels;    /*!< Facies Labels            (for FACIES color scales) (nclasses strings, each limited to 50 characters) */
  char undef_label[50];    /*!< Undefined Values Label   (for FACIES color scales) */
  char other_label[50];    /*!< Other Values Label       (for FACIES color scales) */
} GTXColorScale;

/*!
 * \var GTXVendorData
 * \brief Opaque \a Vendor \a Data structure
 *
 * This structure is used as an opaque data to manage \a Vendor \a Data,
 * \sa GTXClientVendorDataNew
 */
typedef struct _GTXVendorDataRec* GTXVendorData;

#ifdef __cplusplus
extern "C" {
#endif
/*!
 * \var void (*GTX_ERROR_FUNCTION)(const char *string)
 * \brief Error message function pointer
 * This function pointer is called by the API when an error occured to inform
 * the calling application.
 *
 * This function can be overriden by an application function. The default
 * function prints the error message to the standard error descriptor
 * prefixed by the string: "GTXClient ERROR:".
 */
GLOBAL void (*GTX_ERROR_FUNCTION)(const char *string);

/*! \enum GTXErrorCode
 * Define the different codes returned by GTXClient C APIs
 * /sa GTXClientGetLastError
 */
typedef enum {
  GTX_ERROR_NONE = 0,  /*!< OK, can be checked as a boolean */
  GTX_ERROR_PARAMETER, /*!< Invalid parameter */
  GTX_ERROR_NETWORK,   /*!< Network connection problem */
  GTX_ERROR_SERVER,    /*!< Server returned an error */
  GTX_ERROR_MEMORY     /*!< Memory Allocation problem */
} GTXErrorCode;

/*! \enum GTXFeatureType
 * Define the different new features that might not be available in some GTXserver.
 * /sa GTXClientServerHasFeature
 */
typedef enum
{
  GTX_FEATURE_PALETTES,       /*!< Support from server 2013.2+ */
  GTX_FEATURE_COLOR_SCALES,   /*!< Support from server 2013.2+ */
} GTXFeatureType;

/******************************************************************************/
/*                         GTXClient API functions                            */
/******************************************************************************/


/* @defgroup init Init/Debug functions
** @{
*/
GLOBAL int GTXClientInitialize(int linked_version);
GLOBAL void GTXClientDebug(int mode);
GLOBAL GTXErrorCode GTXClientGetLastError();
/* @}
*/

/* @defgroup connect Connection functions
** @{
*/
GLOBAL unsigned short GTXClientGetDefaultPort(void);
GLOBAL int GTXClientLocateGTXserver(char *path);
GLOBAL void GTXClientSetGTXserverPath(const char *path);
GLOBAL int GTXClientRunGTXserver(unsigned short *port);
GLOBAL int GTXClientConnect(const char *host, unsigned short port, const char *path);
GLOBAL int GTXClientIsConnected();
GLOBAL int GTXClientDisconnect(void);
GLOBAL int GTXClientGetServerVersion();
GLOBAL int GTXClientServerHasFeature(GTXFeatureType type, int generate_error);

/* @}
*/

/* Unit functions */
GLOBAL int GTXClientSetUnit(const char *unit);
GLOBAL double GTXClientUnitFactorFromSymbol(const char *symbol);
GLOBAL const char *GTXClientUnitSymbolFromFactor(double factor);
GLOBAL int GTXClientSetUnitMode(int mode);
GLOBAL int GTXClientSetCreationUnits(const char *x_unit, const char *y_unit, const char *z_unit);

/* Memory Management functions */
GLOBAL void *GTXClientFreePointer(void *pointer);
GLOBAL char **GTXClientFreeStringArray(gtx_long nlist, char **list);

/* Study functions */
GLOBAL int GTXClientGetStudyList(int *nb_study, char ***studies);
GLOBAL int GTXClientSetStudy(const char *study);
GLOBAL int GTXClientNewStudy(const char *study, const char *path);
GLOBAL int GTXClientDeleteStudy(void);

/* Directory functions */
GLOBAL int GTXClientGetDirectoryList(int *nb_dir, char ***directories);
GLOBAL int GTXClientDirectoryExists(const char *dir_name, int *exists);
GLOBAL int GTXClientSetDirectory(const char *dir);
GLOBAL int GTXClientNewDirectory(const char *name);
GLOBAL int GTXClientDeleteDirectory(void);

/* File functions */
GLOBAL int GTXClientGetFileList(int *nb_file, char ***files);
GLOBAL int GTXClientGetFileListByType(GTX_FILE_TYPE file_type, int file_dim, int *nb_file, char ***files);
GLOBAL int GTXClientFileExists(const char *file_name, int *exists);
GLOBAL int GTXClientSetFile(const char *file);
GLOBAL int GTXClientDeleteFile(void);
GLOBAL int GTXClientGetFileInfo(GTXFileInfo *file_info);
GLOBAL int GTXClientGetFileComment(char **comment);
GLOBAL int GTXClientSetFileComment(int append, const char *comment);
GLOBAL int GTXClientNewPointsFile(const char *name, gtx_long nsample, const double *x, const double *y, const double *z);
GLOBAL int GTXClientNewGridFile(const char *name, int dimension, double X0, double Y0, double Z0, double DX, double DY, double DZ, int NX, int NY, int NZ);
GLOBAL int GTXClientNewLinesFile(const char *name, int dimension);
GLOBAL int GTXClientPointsFileAppend(gtx_long nsample, const double *x, const double *y, const double *z);
GLOBAL int GTXClientLinesFileAddCoreLine(int nsample, double xbegin, double ybegin, double zbegin, const double *xend, const double *yend, const double *zend);
GLOBAL int GTXClientLinesFileAddGravityLine(int nsample, const double *xg, const double *yg, const double *zg);
GLOBAL int GTXClientLinesFileAddLineCoresOld(int nsample, double xbegin, double ybegin, double zbegin, const double *xend, const double *yend, const double *zend);
GLOBAL int GTXClientLinesFileCreateLink(const char *points_file);
GLOBAL int GTXClientGridModifyOrigin(double x0, double y0, double z0);
GLOBAL int GTXClientGridModifyMesh(double dx, double dy, double dz);
GLOBAL int GTXClientGridModifyRotation(double az, double ay, double ax);

/* Variable functions */
GLOBAL int GTXClientGetVariableList(int *nb_var, char ***vars);
GLOBAL int GTXClientGetVariableListByType(GTX_VAR_TYPE var_type, int *nb_var, char ***vars);
GLOBAL int GTXClientVariableExists(const char *var_name, int *exists);
GLOBAL int GTXClientSetVariable(const char *var);
GLOBAL int GTXClientDeleteVariable(void);
GLOBAL int GTXClientSetIndice(const int indice);
GLOBAL int GTXClientSetAlphaIndice(const char* alphaindice);
GLOBAL int GTXClientGetVariableInfo(GTXVariableInfo *var_info);
GLOBAL int GTXClientGetMacroIndices(int *nb_indices, int **indices);
GLOBAL int GTXClientGetMacroIndicesWithAlpha(int *nb_indices, int **indices, char ***alpha_indices);
GLOBAL int GTXClientAddMacroIndicesWithAlpha(const int nb_indice, const int* indices, const char * const*alpha_indices);
GLOBAL int GTXClientAddMacroIndices(const int nb_indice, const int* indices);
GLOBAL int GTXClientReadLineDoubleVariable(int line_index, double *test_val, int *nsample, double **data);
GLOBAL int GTXClientReadDoubleVariable(int compress, double *test_val, gtx_long *nb_val, double **data);
GLOBAL int GTXClientReadDoubleVariableSub(gtx_long first, gtx_long last, int compress, double *test_val, gtx_long *nb_val, double **data);
GLOBAL int GTXClientReadCharVariable(int compress, char **test_val, gtx_long *nb_val, char ***data);
GLOBAL int GTXClientReadCharVariableSub(gtx_long first, gtx_long last, int compress, char **test_val, gtx_long *nb_val, char ***data);
GLOBAL int GTXClientReadLineCharVariable(int line_index, char **test_val, int *nsample, char ***data);
GLOBAL int GTXClientReadSelectionVariable(gtx_long *nb_val, unsigned char **data);
GLOBAL int GTXClientReadSelectionVariableSub(gtx_long first, gtx_long last, gtx_long *nb_val, unsigned char **data);
GLOBAL int GTXClientNewDoubleVariable(const char *name, int bit_length, GTX_FORMAT_TYPE format_type, int format_length, int format_digits, const char *format_unit);
GLOBAL int GTXClientNewCharVariable(const char *name, int max_length);
GLOBAL int GTXClientSetVariableAsLineName(void);
GLOBAL int GTXClientWriteLineDoubleVariable(int line_index, double test_val, int nsample, const double *data);
GLOBAL int GTXClientWriteDoubleVariable(int flag_compress, double test_val, gtx_long nb_val, const double *dbl_data);
GLOBAL int GTXClientWriteDoubleVariableSub(gtx_long first, gtx_long last, int flag_compress, double test_val, gtx_long nb_val, const double *dbl_data);
GLOBAL int GTXClientWriteCharVariable(int flag_compress, const char *test_val, gtx_long nb_val, const char * const *char_data);
GLOBAL int GTXClientWriteCharVariableSub(gtx_long first, gtx_long last, int compress, const char *test_val, gtx_long nb_val, const char * const *data);
GLOBAL int GTXClientWriteLineCharVariable(int line_index, const char* test_val, int nsample, const char * const *char_data);
GLOBAL int GTXClientWriteSelectionVariable(const gtx_long nb_val, const unsigned char *data);
GLOBAL int GTXClientWriteSelectionVariableSub(const gtx_long first, const gtx_long last,const gtx_long nb_val, const unsigned char *data);
GLOBAL int GTXClientGetVariableComment(char **comment);
GLOBAL int GTXClientSetVariableComment(int append, const char *comment);
GLOBAL int GTXClientSetSelection(const char *sel);
GLOBAL int GTXClientSetVariableUnit(const char *factor, const char *symbol);

/* Fault functions */
GLOBAL int GTXClientReadFaults(int priority, GTXFaultSystem *fault_system);
GLOBAL int GTXClientWriteFaults(const GTXFaultSystem *fault_system);
GLOBAL void GTXClientFreeFaults(GTXFaultSystem *fault_system);
GLOBAL void GTXClientFaultsSegmentDuplicate(const GTXFaultSegment *fault_seg_in, GTXFaultSegment *fault_seg_out);
GLOBAL void GTXClientFaultDuplicate(const GTXFault *fault_in, GTXFault *fault_out);
GLOBAL GTXFaultSystem *GTXClientFaultSystemDuplicate(const GTXFaultSystem *fault_sys);

/* Polygon functions */
GLOBAL int GTXClientNewPolygonsFile(const char *name, const GTXPolygonSystem *psystem);
GLOBAL int GTXClientReadPolygons(GTXPolygonSystem *polygon_system);
GLOBAL void GTXClientFreePolygons(GTXPolygonSystem *polygon_system);
GLOBAL void GTXClientPolygonDuplicate(const GTXPolygon *polygon_in, GTXPolygon *polygon_out);
GLOBAL void GTXClientPolygonSystemDuplicate(const GTXPolygonSystem *polysys_in, GTXPolygonSystem *polysys_out);


/* Vendor Data functions */
GLOBAL GTXVendorData GTXClientVendorDataNew(const char *identificator, int version);
GLOBAL int GTXClientVendorDataAddAttributeInt(GTXVendorData vendor_data, const char *name, int value);
GLOBAL int GTXClientVendorDataAddAttributeDouble(GTXVendorData vendor_data, const char *name, double value);
GLOBAL int GTXClientVendorDataAddAttributeString(GTXVendorData vendor_data, const char *name, const char *value);
GLOBAL int GTXClientVendorDataAddAttributeIntArray(GTXVendorData vendor_data, const char *name, int nvalues, const int *values);
GLOBAL int GTXClientVendorDataAddAttributeDoubleArray(GTXVendorData vendor_data, const char *name, int nvalues, const double *values);
GLOBAL int GTXClientVendorDataAddAttributeStringArray(GTXVendorData vendor_data, const char *name, int nvalues, const char * const *values);
GLOBAL int GTXClientVendorDataWrite(int level, GTXVendorData vendor_data);
GLOBAL int GTXClientVendorDataGetVersion(const char *identificator, int level, int *version);
GLOBAL int GTXClientVendorDataRead(const char *identificator, int level, GTXVendorData *vendor_data, int *version);
GLOBAL int GTXClientVendorDataGetAttributeInt(GTXVendorData vendor_data, const char *name, int *value);
GLOBAL int GTXClientVendorDataGetAttributeDouble(GTXVendorData vendor_data, const char *name, double *value);
GLOBAL int GTXClientVendorDataGetAttributeString(GTXVendorData vendor_data, const char *name, char **value);
GLOBAL int GTXClientVendorDataGetAttributeIntArray(GTXVendorData vendor_data, const char *name, int *nvalues, int **values);
GLOBAL int GTXClientVendorDataGetAttributeDoubleArray(GTXVendorData vendor_data, const char *name, int *nvalues, double **values);
GLOBAL int GTXClientVendorDataGetAttributeStringArray(GTXVendorData vendor_data, const char *name, int *nvalues, char ***values);
GLOBAL GTXVendorData GTXClientVendorDataFree(GTXVendorData vendor_data);
GLOBAL int GTXClientVendorDataExists(const char *identificator, int level, int *exists);
GLOBAL GTXVendorData GTXClientVendorDataDuplicate(GTXVendorData vendor_data);

/* Wireframes functions */
GLOBAL int GTXClientReadWireframes(GTXWireframeSystem *wireframe_system);
GLOBAL int GTXClientNewWireframesFile(const char *filename, const GTXWireframeSystem *wireframe_system);
GLOBAL void GTXClientFreeWireframes(GTXWireframeSystem *wireframe_system);
GLOBAL void GTXClientWireframeObjectDuplicate(const GTXWireframeObject *object_in, GTXWireframeObject *object_out);
GLOBAL void GTXClientWireframeDuplicate(const GTXWireframe *wireframe_in, GTXWireframe *wireframe_out);
GLOBAL void GTXClientWireframeSystemDuplicate(const GTXWireframeSystem *wsys_in, GTXWireframeSystem *wsys_out);

/* Palettes functions */
GLOBAL int GTXClientPaletteGetList(int *nb_pal, char ***palettes);
GLOBAL int GTXClientPaletteExists(const char *cscale_name, int *exists);
GLOBAL int GTXClientPaletteWrite(const char *name, int ncolors, const int* rgbs);
GLOBAL int GTXClientPaletteRead(const char *name, int* ncolors, int** rgbs);
GLOBAL int GTXClientPaletteDelete(const char *name);

/* Color Scales functions */
GLOBAL int GTXClientColorScaleGetList(int *nb_cscales, char ***cscales);
GLOBAL int GTXClientColorScaleExists(const char *cscale_name, int *exists);
GLOBAL void GTXClientColorScaleInit(GTXColorScale *cscale);
GLOBAL void GTXClientColorScaleFree(GTXColorScale *cscale);
GLOBAL int GTXClientColorScaleNew(const char *name, const GTXColorScale* cscale);
GLOBAL int GTXClientColorScaleRead(const char *cscale_name, GTXColorScale *cscale);
GLOBAL int GTXClientColorScaleDelete(const char *name);

#ifdef __cplusplus
}
#endif

/* Stop Redefining names of C structures */
#ifdef __cplusplus
# undef GTXFault
# undef GTXFaultInfo
# undef GTXFaultSegment
# undef GTXFaultSystem
# undef GTXFileInfo
# undef GTXPolygon
# undef GTXPolygonSystem
# undef GTXVariableInfo
# undef GTXVendorData
# undef GTXWireframe
# undef GTXWireframeObject
# undef GTXWireframeSystem
# undef GTXColorScale
#endif

#endif /* _GTXClient_h */
