/*****************************************************************************
 
            Copyright (c)2009 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#ifndef __GTXClient_hpp__
#define __GTXClient_hpp__

/*!
 * \file GTXClient.hpp
 * \brief GTXclient C++ API
 */

#define __USING_GTXSERVER_CPP_API__

#ifdef DOXYGEN
#define SWIG_OR_DOXYGEN
#endif //DOXYGEN

#ifdef SWIG
#define GLOBAL_CLASS
#define SWIG_OR_DOXYGEN
#else //SWIG

#include <GTXStringArray.hpp>
#include <GTXByteArray.hpp>
#include <GTXFileInfo.hpp>
#include <GTXVariableInfo.hpp>
#include <GTXVendorData.hpp>
#include <GTXFaultSystem.hpp>
#include <GTXPolygonSystem.hpp>
#include <GTXWireframeSystem.hpp>

class GTXStringArray;
class GTXIntArray;
class GTXDoubleData;
class GTXCharData;
class GTXVariableFormat;
#endif //SWIG

/*!
 * \mainpage GTXClient C++ API
 *
 * \section what What are GTXserver and GTXClient ?
 *
 * You should first read the
 * <a href="../index.html" target="_blank">introduction</a> to know it.
 *
 * \section cpp What is the GTXClient C++ API ?
 *
 * GTXClient C++ is a C++ wrapper built on top of GTXClient C API.
 * It provides a better memory management (less calls needed to free) and error
 * reporting (they are reported with exceptions).
 *
 * The API has a main class \ref GTXClient and some other utility classes.
 *
 * \section other_lang What about Java and .NET ?
 *
 * Java and .NET APIs are built ontop of GTXClient C++ API. The API is quite
 * exactly the same than the C++ one and when things are a little different,
 * it should appear in the methods documentation.
 *
 * \page cppexample
 *
 * \section CPPexample Examples written in C++
 *
 * Once compiled and linked, these examples can be run at the shell prompt. All
 * of them are clients for the GTXserver. Each client will either run a new server
 * or connect to an existing one depending on command line arguments and or choices
 * in the interface when possible (\ref sample_browser).
 *
 * Here is a list of the options available when running those programs:
 * 
 * <TABLE>
 * <TR><TD>-help</TD><TD>Prints help on command usage</TD></TR>
 * <TR><TD>-debug</TD><TD>Prints client/server messages. Interesting to follow
 * the dialog between the server and its client.</TD></TR>
 * <TR><TD>-host hostname</TD><TD>Server's hostname. If not specified, the
 * default host is "localhost"..</TD></TR>
 * <TR><TD>-port port_number</TD><TD>Port used to communicate with the client.
 * The default is the same for the server and the client. If for any reason
 * the default port number is not correct on your machine, you can change it
 * (you must use the same port number in the server and the client).</TD></TR>
 * <TR><TD>-path data_path</TD><TD>Asks the server to use this data path to
 * access to the Isatis file system. The default is set to the \$GTX_DATA
 * environment variable if defined.<TD></TR>
 * <TR><TD COLSPAN=2>The default is to run a new server on a random port.<BR>
 * If -host or -port is specified, a server must have been started first on
 * the given host/port</TD></TR>
 * </TABLE>
 * 
 * You can try to run the client to see what happens. If you examine the
 * source code, you will find the main calls used to write a client
 * application. This source code is only composed of the main and some utility
 * functions can be found in utils.hpp (argument parsing and statistics methods).
 * 
 * The first thing is to create an instance of the GTXClient class to be able to
 * call its methods. Creating the instance initializes the library properly.
 *
 * With the option -debug, the program calls GTXClient::SetDebugMode() with true.
 * Then all the functions will print extra messages about the dialog between
 * server and client. This can be useful in a debug phase.
 * 
 * The default is then to run a new GTXserver on a random port. The function
 * will look for the server executable in the following places:
 * - the contents of GTX_SERVER environment variable
 * - the GTX_HOME environment variable (mostly used under UNIX to indicate
 * where Isatis is installed) or a registry key under Windows.
 * - through the PATH.
 * 
 * To connect to the server, the client calls GTXClient::Connect(). The server
 * must already be running and waiting for connection. If no -port or -host
 * option has been used, it has been started just above, so it should be OK.
 * Otherwise, the server must already be running on the given host/port.
 * If not, the function will throw an exception.
 *
 * If the server has been started with the -multisession parameter, the
 * server will still be available for other sessions during this session
 * lifetime. Moreover, when GTXClient::Disconnect will be called, the server
 * will still be there...
 *
 * \subsection exlinks Examples:
 * \ref sample_dump dumps all Isatis studies, directories, files and variables.
 * This is a simple C++ program with no user interactions. 
 *
 * \ref sample_browser Qt program allowing to browse the data tree and get information on the nodes.
 * Interactive Qt program to browse the Isatis file hierarchy.

 * \link fault.cpp \endlink attaches faults to a file and retrieves information on faults
 *
 * \link polygon.cpp \endlink creates a polygons file and retrieves information on polygons
 *
 * \link wireframes.cpp \endlink creates a wireframes file and retrieves information on wireframes
 *
 * \link vendor.cpp \endlink attaches some vendor data to a file and read it back
 *
 * \link write_files.cpp \endlink creates 2D & 3D Points, Lines and Grid files in a new study.
 *
 */

/*!
 * \page dump
 *
 * \section sample_dump dump.cpp example
 *
 * See \link dump.cpp dump.cpp \endlink source code.
 *
 * This example is the simpliest way to understand the communication between
 * GTXserver and a client. The program scrolls through the Isatis file system
 * to get information/statistics on all files and variables.
 * 
 * As we are connected we can get the list of studies using
 * GTXClient::GetStudyList(). We can either use the GTXStringArray class to
 * access each element with GTXStringArray::GetValue or retrieve the pointer
 * on the complete list with GTXStringArray::GetValues.
 * 
 * We then set the name of the current study using GTXClient::SetStudy() For
 * each study we can obtain the directory list with
 * GTXClient::GetDirectoryList(). And we set the current directory using
 * GTXClient::SetDirectory() for each directory of the list.
 * 
 * After getting the file list of each directory using GTXClient::GetFileList(),
 * we must set as current each file with GTXClient::SetFile().
 * 
 * At this point, we call a utility method from utils.hpp with obtains file
 * information using GTXClient::GetFileInfo(). It then dump the contents of the
 * GTXFileInfo class in a string and returns this string so that we can print
 * it from the main.
 * 
 * After that, we get the variable list with a call to
 * GTXClient::GetVariableList() and define each variable as the current one with
 * GTXClient::SetVariable().
 * 
 * After getting variable information through a utils.hpp method
 * (GTXClient::GetVariableInfo()), we dump the contents of the GTXVariableInfo
 * class returned by this method into a string that is returned to the main and
 * printed.
 * 
 * If the current variable is a macro variable, we get macro indices using
 * GTXClient::GetMacroIndices() and dump them. Note: Macro variables
 * appears in Isatis 3.2 and can have alphanumerical indices since Isatis 5.0.
 * 
 * Just before exiting, we must disconnect from the server using
 * GTXClient::Disconnect(). If the server is mono session, it exits.
 * If the server is multi session, only the current session terminates and
 * GTXserver is still available for a further session.
 */

/*!
 * \page browser
 *
 * \section sample_browser browser.cpp example
 *
 * See \link browser.cpp browser.cpp \endlink source code.
 *
 * This is interactive Qt (http://qt.nokia.com) browser interface allowing to browser
 * the data tree and get information on nodes. Works like the dump program (\ref sample_dump)
 * but with the user interacting to open/close tree nodes and get information those nodes.
 */

/*! \example dump.cpp */
/*! \example browser.cpp */
/*! \example fault.cpp */
/*! \example polygon.cpp */
/*! \example wireframes.cpp */
/*! \example vendor.cpp */
/*! \example write_files.cpp */

/*!
 * \brief Class handling connection and communication with GTXserver
 */
class GLOBAL_CLASS GTXClient
{
public:
  GTXClient();

  /* Debug methods */
  void SetDebugMode(bool debug);

  /* Connection methods */
  unsigned short GetDefaultPort(void);
  const char *LocateGTXserver();
  void SetGTXserverPath(const char *path);
  unsigned short RunGTXserver(unsigned short port);
  void Connect(const char *host, unsigned short port, const char *path);
  bool IsConnected();
  void Disconnect();
  int GetServerVersion();

  /* Study methods */
  GTXStringArray GetStudyList();
  void SetStudy(const char *study);
  void NewStudy(const char *study, const char *path);
  void DeleteStudy();

  /* Directory methods */
  GTXStringArray GetDirectoryList();
  bool DirectoryExists(const char *directory);
  void SetDirectory(const char *directory);
  void NewDirectory(const char *directory);
  void DeleteDirectory();

  /* File methods */
  GTXStringArray GetFileList();
  GTXStringArray GetFileList(GTXFileInfo::FileType file_type, int file_dim);
  bool FileExists(const char *file);
  void SetFile(const char *file);
  void DeleteFile();
  GTXFileInfo GetFileInfo();
  const char *GetFileComment();
  void SetFileComment(bool append, const char *comment);
  void NewPointsFile2D(const char *name, gtx_long nsample, const double x[], const double y[]);
  void NewPointsFile3D(const char *name, gtx_long nsample, const double x[], const double y[], const double z[]);
  void NewGridFile2D(const char *name, double X0, double Y0, double DX, double DY, int NX, int NY);
  void NewGridFile3D(const char *name, double X0, double Y0, double Z0, double DX, double DY, double DZ, int NX, int NY, int NZ);
  void NewLinesFile(const char *name, int dimension);
  void PointsFileAppend2D(gtx_long nsample, const double x[], const double y[]);
  void PointsFileAppend3D(gtx_long nsample, const double x[], const double y[], const double z[]);
  void LinesFileAddLineCores2DOld(int nsample, double xbegin, double ybegin, const double xend[], const double yend[]);
  void LinesFileAddLineCores3DOld(int nsample, double xbegin, double ybegin, double zbegin, const double xend[], const double yend[], const double zend[]);
  void LinesFileAddCoreLine2D(int nsample, double xbegin, double ybegin, const double xend[], const double yend[]);
  void LinesFileAddCoreLine3D(int nsample, double xbegin, double ybegin, double zbegin, const double xend[], const double yend[], const double zend[]);
  void LinesFileAddGravityLine2D(int nsample, const double xg[], const double yg[]);
  void LinesFileAddGravityLine3D(int nsample, const double xg[], const double yg[], const double zg[]);
  void LinesFileCreateLink(const char *points_file);
  void GridModifyOrigin(double x0, double y0, double z0);
  void GridModifyMesh(double dx, double dy, double dz);
  void GridModifyRotation(double az, double ay, double ax);

  /* Variable methods */
  GTXStringArray GetVariableList();
  GTXStringArray GetVariableList(GTXVariableInfo::VariableType var_type);
  bool VariableExists(const char *variable);
  void SetVariable(const char *variable);
  void DeleteVariable();
  void SetIndice(const int indice);
  void SetAlphaIndice(const char *alphaindice);
  GTXVariableInfo GetVariableInfo();
  GTXIntArray GetMacroIndices();
  GTXStringArray GetMacroAlphaIndices();
  void AddMacroIndicesWithAlpha( const GTXIntArray indices, const GTXStringArray alpha_indices);
  void AddMacroIndices( const GTXIntArray indices);

  GTXCharData ReadCharVariable(bool compress);
  GTXCharData ReadCharVariableSub(gtx_long first, gtx_long last, bool compress);
  GTXCharData ReadLineCharVariable(int line_index);
  GTXDoubleData ReadLineDoubleVariable(int line_index);
  GTXDoubleData ReadDoubleVariable(bool compress);
  GTXDoubleData ReadDoubleVariableSub(gtx_long first, gtx_long last, bool compress);
  GTXByteArray ReadSelectionVariable();
  GTXByteArray ReadSelectionVariableSub(gtx_long first, gtx_long last);
  void NewDoubleVariable(const char *name, int bit_length, const GTXVariableFormat &format);
  void NewCharVariable(const char *name, int max_length);
  void SetVariableAsLineName(void);
  void WriteDoubleVariable(bool compress, const GTXDoubleData &ddata);
  void WriteDoubleVariableSub(gtx_long first, gtx_long last, bool compress, const GTXDoubleData &ddata);
  void WriteLineDoubleVariable(int line_index, const GTXDoubleData &ddata);
  void WriteCharVariable(bool compress, const GTXCharData &cdata);
  void WriteCharVariableSub(gtx_long first, gtx_long last, bool compress, const GTXCharData &cdata);
  void WriteLineCharVariable(int line_index, const GTXCharData &cdata);
  void WriteSelectionVariableSub(const gtx_long first, const gtx_long last, const GTXByteArray data);
  void WriteSelectionVariable(const GTXByteArray data);
  const char *GetVariableComment();
  void SetVariableComment(bool append, const char *comment);
  void SetSelection(const char *sel);
  void SetVariableUnit(const char *factor, const char *symbol);

  /* Unit Methods */
  void SetUnit(const char *unit);
  double UnitFactorFromSymbol(const char *symbol);
  const char *UnitSymbolFromFactor(double factor);
  void SetUnitMode(int mode);
  void SetCreationUnits(const char *x_unit, const char *y_unit, const char *z_unit);

  /*Vendor Data Method*/
  void VendorDataWrite(int level, const GTXVendorData &vdata);
  GTXVendorData VendorDataRead(const char *identifier, int level);
  bool VendorDataExists(const char *identifier, int level);
  int VendorDataGetVersion(const char *identifier, int level);

  /*Fault System Method*/
  GTXFaultSystem ReadFaults(int priority);
  void WriteFaults(GTXFaultSystem fs);

  /*Polygon functions*/
  void NewPolygonsFile(const char *name, const GTXPolygonSystem *psystem);
  GTXPolygonSystem ReadPolygons();

  /*Wireframes functions*/
  void NewWireframesFile(const char *name, const GTXWireframeSystem &wsys);
  GTXWireframeSystem ReadWireframes();

private:
  static void _errorMessage(const char *msg);
  static void _cleanupErrors();
  static void _throwErrors();
  static char *_errors;

  friend class GTXVendorData;

};

#endif // __GTXClient_hpp__
