#include <GTXClient.hpp>
#include <GTXClient.h>
#include <GTXError.hpp>
#include <GTXStringArray.hpp>
#include <GTXIntArray.hpp>
#include <GTXDoubleData.hpp>
#include <GTXCharData.hpp>
#include <GTXVariableFormat.hpp>
#include <string.h>
#include <stdlib.h>

char *GTXClient::_errors = 0;

// Warning: this method is declared as C++ and assigned to C function pointers.
// This generate warnings by default on solaris (disabled with -erroff=wbadasg)
// This is because the standards tells that the functions may be called differently if C or C++.
// This does seem to be the case for solaris/linux/windows so I removed the warning.
// Otherwise we should make wrappers for those methods.

void GTXClient::_errorMessage(const char *msg)
{
  int len = (_errors == (char*)0)? 0: (int)strlen(_errors);
  int len2 = (int)strlen(msg);
  _errors = (char*)realloc(_errors, len+len2+1);
  (void)strcpy(_errors+len, msg);
}

void GTXClient::_cleanupErrors()
{
  if (_errors != (char*)0)
    free(_errors);
  _errors = (char*)0;
}

void GTXClient::_throwErrors()
{
  if (_errors != NULL)
    throw GTXError(_errors);
  else
  {
    switch (GTXClientGetLastError())
    {
      case GTX_ERROR_PARAMETER:
        throw GTXError("Invalid Parameter to GTXserver API.");
        break;
      case GTX_ERROR_NETWORK:
        throw GTXError("Network connection problem. GTXserver not connected?");
        break;
      case GTX_ERROR_SERVER:
        throw GTXError("GTXserver returned an error.");
        break;
      case GTX_ERROR_MEMORY:
        throw GTXError("GTXclient reported a memory allocation failure.");
        break;
      default:
        throw GTXError("GTXserver API. An unknown error occured.");
        break;
    }
  }
}

/*!
******************************************************************************
\brief Default constructor, initializes the client library
*****************************************************************************/
GTXClient::GTXClient()
{
  GTXClientInitialize(GTX_CLIENT_VERSION);
  GTX_ERROR_FUNCTION = GTXClient::_errorMessage;
}

/*!
******************************************************************************
\brief Set the Debug Mode (off by default)

\param debug new debug mode, false by default. When true, debug messages
are sent to the terminal.
*****************************************************************************/
void GTXClient::SetDebugMode(bool debug)
{
  GTXClientDebug(debug);
}

/*!
******************************************************************************
\brief Get the default port for GTXserver

When the GTXserver is ran without argument, it runs on a default port. This
default port is taken from the local /etc/services file using getservbyname
with the service name equal to "GTXserver". If not found in the services file,
it defaults to a hard-coded value.

\return port default GTXserver port

\par Remarks: The returned value may be used as an initial value for the port
in the user interface asking parameters for connection on an already running
server.
*****************************************************************************/
unsigned short GTXClient::GetDefaultPort(void)
{
  return (GTXClientGetDefaultPort());
}

/*!
****************************************************************************
This function tries to determine the location of GTXserver executable using:
- the value that has been set by \ref GTXClient::SetGTXserverPath()
- the value of the GTX_SERVER environment variable
- the executable located in Isatis installation (which is determined by
the value of GTX_HOME environment variable under UNIX and Windows or a
registry key under Windows),
- a GTXserver executable in one component of the PATH environment variable.
If all of those tests are unsuccessfull, this function will return 1.

\return GTXserver executable path or empty string

\par Remarks:
This function can be use to initialize an interface with the path of the
currently used GTXserver or for debugging purposes. You can also force this
function to find a given executable by using \ref GTXClient::SetGTXserverPath()
The returned string is a static string so it does have to be freed.
***************************************************************************/
const char *GTXClient::LocateGTXserver()
{
  static char path[1024];

  _cleanupErrors();
  if (GTXClientLocateGTXserver(path))
    (void)strcpy(path, "");
  return path;
}

/*!
****************************************************************************
\brief Set the GTXserver path to be used by \ref GTXClient::RunGTXserver()

\param path: full path of the GTXserver executable, should be less than 1024
characters or empty string to remove any previous specification of it.
***************************************************************************/
void GTXClient::SetGTXserverPath(const char *path)
{
  GTXClientSetGTXserverPath(path);
}

/*!
****************************************************************************
\brief Locate GTXserver executable and run it.

This function tries to locate the GTXserver executable using
\ref GTXClient::LocateGTXserver() and run it. If it is not found or if
in cannot run, this function will throw an exception.

\throw GTXError

\return port on which the server is now running

\param port if you have a preferred port to RUN GTXserver on. If 0 a free
port is taken and its value is returned.

\par Remarks:
GTXserver is ran in the background listening on the given port. If the port
is 0, GTXserver is ran on the first free port of the machine. This is the
preferred way to run GTXserver as it avoids trying to run a server on an
already used port. The new returned port should be used to connect to the server.
***************************************************************************/
unsigned short GTXClient::RunGTXserver(unsigned short port)
{
  _cleanupErrors();
  if (GTXClientRunGTXserver(&port))
    _throwErrors();
  return port;
}

/*!
******************************************************************************
\brief Open a session with a running GTXserver to read isatis file system

This function opens a session with a GTXserver running on the host \e host
that listens to the port \e port and starts to read the Isatis file system
located at \e path. If the server is a mono session one, the server will remain
attached to this client until disconnect then it will quit. If the server is
a multi session one, the server will still be available for other clients.

\param host hostname or IP address "aaa.bbb.ccc.ddd" of the machine where
the GTXserver is running on
\param port port number where the GTXserver is listening to. If 0, a call to
/ref GTXClient::GetDefaultPort is made to determine the port where GTXserver
may be running.
\param path pathname of the Isatis file system to be accessed (should be a
path entered by the user (GTX_DATA or GTX_INIT (see below))

\par Remarks:
A session must be closed using GTXClientDisconnect to free memory allocations.
When accessing Isatis < 4.1 data, the user must specify where its database is
stored. This place is known as GTX_DATA environment variable. You can get it
in Isatis from the Help->About menu.
Starting with Isatis 4.1, this variable is not useful anymore. Instead, an
hidden file is created to keep a list of studies which can be put anywhere on
disk. If accessing data linked to an Isatis >= 4.1, this variable should be
left empty (not NULL). However it is possible to override the default study
list that would then be taken by specifying the place where the study list
lies in. This directory is referred to as GTX_INIT in the Isatis Help -> About
Menu.

\sa GTXClient::Disconnect, GTXClient::GetDefaultPort
*****************************************************************************/
void GTXClient::Connect(const char *host, unsigned short port, const char *path)
{
  _cleanupErrors();
  if (GTXClientConnect(host, port, path))
    _throwErrors();
}

/*!
******************************************************************************
\brief Check is client is connected with the server.

Check the status of the connection with the GTXserver

\return connected
  \arg false if not connected
  \arg true if connected

\sa GTXClient::Connect, GTXClient::Disconnect
*****************************************************************************/
bool GTXClient::IsConnected()
{
  return (GTXClientIsConnected() != 0);
}

/*!
******************************************************************************
\brief Close the current session with the GTXserver.

This function closes the current session with the GTXserver.
The server will exit if in mono session mode, if in  multi-session  mode, it
will only close the current connection.

\throw GTXError

\par Remarks:
If there is no opened session the function will return 0.

\sa GTXClient::Connect
*****************************************************************************/
void GTXClient::Disconnect()
{
  _cleanupErrors();
  if (GTXClientDisconnect())
    _throwErrors();
}

/*!
******************************************************************************
\brief Retrieves Server Protocol Version

When the client is connected to the server, allows to know the server protocol
version. The version is encoded as followed:
Major * 1000000 + Minor * 1000 + Patch * 10 + Sub-Version
It may now correspond to the Isatis version but to the major latest change in
the protocol.

\return connected:
  \arg server version

\sa GTXClient::Connect
*****************************************************************************/
int GTXClient::GetServerVersion()
{
  return (GTXClientGetServerVersion());
}

/*!
******************************************************************************
\brief Set the current distance unit

Set the current distance unit.
\throw GTXError
\param unit Distance unit to set

\par Remarks:
This default Unit will be used when creating new variables. It will also be
used to scale length variables in this unit when reading and from this unit
when writing.
Authorized units are:
 \arg micr for \e Micron
 \arg mm for \e Millimeter
 \arg cm for \e Centimeter
 \arg dm for \e Decimeter
 \arg m for \e Meter
 \arg dam for \e Decameter
 \arg hm for \e Hectometer
 \arg km for \e Kilometer
 \arg in for \e Inch
 \arg ft for \e Foot
 \arg yd for \e Yard
 \arg mile for \e Mile
 \arg nmil for \e Nautical \e Mile
 \arg dakm for \e 10 \e Kilometers
 \arg hkm for \e 100 \e Kilometers
 \arg kkm for \e 1000 \e Kilometers
*****************************************************************************/
void GTXClient::SetUnit(const char *unit)
{
  _cleanupErrors();
  if (GTXClientSetUnit(unit))
    _throwErrors();
}

/*!
******************************************************************************
\brief Get the unit factor corresponding to a given unit symbol

This function returns the unit factor corresponding to a given unit symbol or
0. if not a known length symbol.
\param symbol unit symbol to be searched for
\return unit factor
*****************************************************************************/
double GTXClient::UnitFactorFromSymbol(const char *symbol)
{
  return (GTXClientUnitFactorFromSymbol(symbol));
}

/*!
******************************************************************************
\brief Get the unit factor corresponding to a given unit symbol

This function returns the unit symbol corresponding to a given unit symbol or
an empty string if not a known length factor.
\param factor unit factor to be searched for
\return unit symbol
\par Remarks: the returned string is a static string. It doesn't have to be
fred.
*****************************************************************************/
const char *GTXClient::UnitSymbolFromFactor(double factor)
{
  return (GTXClientUnitSymbolFromFactor(factor));
}

/*!
******************************************************************************
\brief Change the way length units are used

This functions allows to change the way units are used at the creation of new
length variables and reading of existing ones.
When mode is 0, all length variables are read/written using the current unit
(\\ref GTXClient::SetUnit).
When mode is 1, each length variable is read/written using its own unit. This
also affect File Information and File Creation that are also based on
coordinates variables which are the main length variables.
\throw GTXError
\param mode new unit mode

\par Remarks
Default mode if 0 which means that you read/write all information in the
current unit set with \\ref GTXClient::SetUnit.

Warning: setting mode to 1 and back to 0 resets creation units
(set with \ref GTXClient::SetCreationUnits) to the current unit.
*****************************************************************************/
void GTXClient::SetUnitMode(int mode)
{
  _cleanupErrors();
  if (GTXClientSetUnitMode(mode))
    _throwErrors();
}

/*!
******************************************************************************
\brief Change the default units for length variables creation

This functions allows to change the default units used for coordinate
variables creation when unit_mode (\ref GTXClient::SetUnitMode) is set to true.
After that, all coordinate variables creation take those units into
account. This has no effect when unit_mode is false.

\throw GTXError
\param x_unit new X coordinate unit
\param y_unit new Y coordinate unit
\param z_unit new Z coordinate unit

\par Remarks
Warning: setting mode to 1 and back to 0 resets creation units.
*****************************************************************************/
void GTXClient::SetCreationUnits(const char *x_unit, const char *y_unit, const char *z_unit)
{
  _cleanupErrors();
  if (GTXClientSetCreationUnits(x_unit, y_unit, z_unit))
    _throwErrors();
}

/*!
******************************************************************************
\brief Get study list in the Isatis file system

This function retrieve, from the GTXserver, the list of available
studies in the Isatis file system selected by the \ref GTXClient::Connect call
\throw GTXError
\return Array of study names
*****************************************************************************/
GTXStringArray GTXClient::GetStudyList()
{
  int nb_study;
  char **studies;
  _cleanupErrors();
  if (GTXClientGetStudyList(&nb_study, &studies))
    _throwErrors();
  GTXStringArray list = GTXStringArray(nb_study, studies);
  studies = GTXClientFreeStringArray(nb_study, studies);
  return list;
}

/*!
******************************************************************************
\brief Set the name of the current study

This function will set the name of the current study, all following calls
will be made using this study.
\throw GTXError
\param study Name of the study to be set
*****************************************************************************/
void GTXClient::SetStudy(const char *study)
{
  _cleanupErrors();
  if (GTXClientSetStudy(study))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new study in the Isatis file system

This function create a new study in the Isatis file system.
\throw GTXError
\param study Name of the study to be created
\param path Physical file PATH (only works with Isatis >= 4.1)
*****************************************************************************/
void GTXClient::NewStudy(const char *study, const char *path)
{
  _cleanupErrors();
  if (GTXClientNewStudy(study, path))
    _throwErrors();
}

/*!
******************************************************************************
\brief Delete the current study

This function will delete the current study. If successfull, the is no
current study set.

\throw GTXError
*****************************************************************************/
void GTXClient::DeleteStudy()
{
  _cleanupErrors();
  if (GTXClientDeleteStudy())
    _throwErrors();
}

/*!
******************************************************************************
\brief Get the list of directories in the current study

This function retrieve, from the GTXserver, the list of available
directories in the current study. 
\throw GTXError
\return Array of directory names
*****************************************************************************/
GTXStringArray GTXClient::GetDirectoryList()
{
  int nb_directory;
  char **directories;
  _cleanupErrors();
  if (GTXClientGetDirectoryList(&nb_directory, &directories))
    _throwErrors();
  GTXStringArray list = GTXStringArray(nb_directory, directories);
  directories = GTXClientFreeStringArray(nb_directory, directories);
  return list;
}

/*!
******************************************************************************
\brief Check if a given directory exists

This function asks the server if a given directory exists (or for old servers, it
retrieves the list of directories and check it by itself).
\throw GTXError
\param directory directory name to check
\return false if it does not exists, true if it exists
*****************************************************************************/
bool GTXClient::DirectoryExists(const char *directory)
{
  int exists;
  _cleanupErrors();
  if (GTXClientDirectoryExists(directory, &exists))
    _throwErrors();
  return exists != 0;
}

/*!
******************************************************************************
\brief Set the current directory

Set the current directory to name.
\throw GTXError
\param directory Name of the directory to be set as current
*****************************************************************************/
void GTXClient::SetDirectory(const char *directory)
{
  _cleanupErrors();
  if (GTXClientSetDirectory(directory))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new directory in the current study

This function creates a new directory in the current study.
\throw GTXError
\param directory Directory name to be created
*****************************************************************************/
void GTXClient::NewDirectory(const char *directory)
{
  _cleanupErrors();
  if (GTXClientNewDirectory(directory))
    _throwErrors();
}

/*!
******************************************************************************
\brief Delete the current directory

This function will delete the current directory. If successfull, the is no
current directory set.

\throw GTXError
*****************************************************************************/
void GTXClient::DeleteDirectory()
{
  _cleanupErrors();
  if (GTXClientDeleteDirectory())
    _throwErrors();
}

/*!
******************************************************************************
\brief Get the list of files in the current directory

This function retrieve, from the GTXserver, the list of available
files in the current directory. 
\throw GTXError
\return Array of file names
*****************************************************************************/
GTXStringArray GTXClient::GetFileList()
{
  int nb_file;
  char **files;
  _cleanupErrors();
  if (GTXClientGetFileList(&nb_file, &files))
    _throwErrors();
  GTXStringArray list = GTXStringArray(nb_file, files);
  files = GTXClientFreeStringArray(nb_file, files);
  return list;
}

/*!
******************************************************************************
\brief Get the list of files in the current directory given a type and
       dimension

This function retrieve, from the GTXserver, the list of available
files in the current directory searching for a given type and dimension.
\throw GTXError
\param file_type File type (\sa GTXFileInfo::FileType)
\param file_dim File dimension (2,3 or -1 if it does not matter)

\return Array of file names
*****************************************************************************/
GTXStringArray GTXClient::GetFileList(GTXFileInfo::FileType file_type,
                                     int file_dim)
{
  int nb_file;
  char **files;
  _cleanupErrors();
  if (GTXClientGetFileListByType((GTX_FILE_TYPE)file_type, file_dim, &nb_file, &files))
    _throwErrors();
  GTXStringArray list = GTXStringArray(nb_file, files);
  files = GTXClientFreeStringArray(nb_file, files);
  return list;
}

/*!
******************************************************************************
\brief Check if a given file exists

This function asks the server if a given file exists (or for old servers, it
retrieves the list of files and check it by itself).
\throw GTXError
\param file file name to check
\return false if it does not exists, true if it exists
*****************************************************************************/
bool GTXClient::FileExists(const char *file)
{
  int exists;
  _cleanupErrors();
  if (GTXClientFileExists(file, &exists))
    _throwErrors();
  return exists != 0;
}

/*!
******************************************************************************
\brief Set the current file in the current directory

Set the current file in the current directory.
\throw GTXError
\param file Name of the file to be set as current
*****************************************************************************/
void GTXClient::SetFile(const char *file)
{
  _cleanupErrors();
  if (GTXClientSetFile(file))
    _throwErrors();
}

/*!
******************************************************************************
\brief Delete the current file

This function will delete the current file. If successfull, the is no
current file set.

\throw GTXError
*****************************************************************************/
void GTXClient::DeleteFile()
{
  _cleanupErrors();
  if (GTXClientDeleteFile())
    _throwErrors();
}

/*!
******************************************************************************
\brief Get information about current file

This function retrieve, from the GTXserver, the information about
the current file. 
\throw GTXError
\retval file_info The information about current file

\par Remarks:
All coordinates are returned in the current unit (\ref GTXClient::SetUnit) or
using the corresponding coordinate variable unit if unit_mode is true
(\ref GTXClient::SetUnitMode).
*****************************************************************************/
GTXFileInfo GTXClient::GetFileInfo()
{
  GTXFileInfo info;
  _cleanupErrors();
  if (GTXClientGetFileInfo(&info._cinfo))
    _throwErrors();
  return info;
}

/*!
******************************************************************************
\brief Get the comment of the current file

Get the comment/description to the current file
\throw GTXError
\retval comment returned comment

\par Remarks:
Warning: errors concerning a non compatible server are silently ignored.

C++ only: comment must be freed using
\code
comment = GTXClientFreePointer(comment)
\endcode
*****************************************************************************/
const char *GTXClient::GetFileComment()
{
  char *comment;
  _cleanupErrors();
  if (GTXClientGetFileComment(&comment))
    _throwErrors();
  return comment;
}

/*!
******************************************************************************
\brief Set a comment for the current file

Set or append a comment/description to the current file
\throw GTXError
\param append
  \arg 0 to set a comment
  \arg 1 to append to an existing one
\param comment Comment to be set or append to the existing one

\par Remarks:
Warning: errors concerning a non compatible server are silently ignored.
*****************************************************************************/
void GTXClient::SetFileComment(bool append, const char *comment)
{
  _cleanupErrors();
  if (GTXClientSetFileComment(append, comment))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new 2D points file in the current directory

This function create a new 2D points file in the current directory.
\throw GTXError
\param name Name of the file to be created
\param nsample Number of samples to be created in this file
\param x Array of X coordinate of each sample
\param y Array of Y coordinate of each sample

\par Remarks:
The file will be created using the current unit (\ref GTXClient::SetUnit) or
the creation units (\ref GTXClient::SetCreationUnits) if unit_mode
is true (\ref GTXClient::SetUnitMode).

The X and Y values must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::NewPointsFile2D(const char *name, gtx_long nsample,
                                const double x[], const double y[])
{
  _cleanupErrors();
  if (GTXClientNewPointsFile(name, nsample, x, y, NULL))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new 3D points file in the current directory

This function create a new 3D points file in the current directory.
\throw GTXError
\param name Name of the file to be created
\param nsample Number of samples to be created in this file
\param x Array of X coordinate of each sample
\param y Array of Y coordinate of each sample
\param z Array of Z coordinate of each sample

\par Remarks:
The file will be created using the current unit (\ref GTXClient::SetUnit) or
the creation units (\ref GTXClient::SetCreationUnits) if unit_mode
is true (\ref GTXClient::SetUnitMode).

The X,Y and Z values must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::NewPointsFile3D(const char *name, gtx_long nsample,
                                const double x[], const double y[], const double z[])
{
  _cleanupErrors();
  if (GTXClientNewPointsFile(name, nsample, x, y, z))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new 3D grid file in the current directory

This function create a new 3D grid file in the current directory.
\throw GTXError
\param name Name of the grid file to be created
\param X0 X origin of the grid
\param Y0 Y origin of the grid
\param Z0 Z origin of the grid
\param DX X size of one cell of the grid
\param DY Y size of one cell of the grid
\param DZ Z size of one cell of the grid
\param NX X nodes number
\param NY Y nodes number
\param NZ Z nodes number

\par Remarks:
The file will be created using the current unit (\ref GTXClient::SetUnit) or
the creation units (\ref GTXClient::SetCreationUnits) if unit_mode is true
(\ref GTXClient::SetUnitMode).
The grid origin and mesh must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::NewGridFile3D(const char *name,
                              double X0, double Y0, double Z0,
                              double DX, double DY, double DZ,
                              int NX, int NY, int NZ)
{
  _cleanupErrors();
  if (GTXClientNewGridFile(name, 3, X0, Y0, Z0, DX, DY, DZ, NX, NY, NZ))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new 2D grid file in the current directory

This function create a new 2D grid file in the current directory.
\throw GTXError
\param name Name of the grid file to be created
\param X0 X origin of the grid
\param Y0 Y origin of the grid
\param DX X size of one cell of the grid
\param DY Y size of one cell of the grid
\param NX X nodes number
\param NY Y nodes number

\par Remarks:
The file will be created using the current unit (\ref GTXClient::SetUnit) or
the creation units (\ref GTXClient::SetCreationUnits) if unit_mode is true
(\ref GTXClient::SetUnitMode).
The grid origin and mesh must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::NewGridFile2D(const char *name,
                              double X0, double Y0,
                              double DX, double DY,
                              int NX, int NY)
{
  _cleanupErrors();
  if (GTXClientNewGridFile(name, 2, X0, Y0, 0., DX, DY, 1., NX, NY, 1))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new lines file in the current directory

This function create a new lines file in the current directory.
\throw GTXError
\param name Name of the new lines file
\param dimension File dimension 2 for 2D, 3 for 3D
*****************************************************************************/
void GTXClient::NewLinesFile(const char *name, int dimension)
{
  _cleanupErrors();
  if (GTXClientNewLinesFile(name, dimension))
    _throwErrors();
}

/*!
******************************************************************************
\brief Append samples to the current Points 2D File

This function appends samples to the current 2D Points File
\throw GTXError
\param nsample Number of samples to be appended in this file
\param x Array of X coordinate of each sample
\param y Array of Y coordinate of each sample

\par Remarks:
The X and Y values must be given in the current unit if unit_mode = false
or according to the corresponding units if unit_mode = true.

Creating a Points File by appending samples 1 by 1 is not optimum at all.
If you know the number of samples from the beginning, you should directly
create the file with the right number of points.
*****************************************************************************/
void GTXClient::PointsFileAppend2D(gtx_long nsample,
                                   const double x[], const double y[])
{
  _cleanupErrors();
  if (GTXClientPointsFileAppend(nsample, x, y, NULL))
    _throwErrors();
}

/*!
******************************************************************************
\brief Append samples to the current 3D Points File

This function appends samples to the current 3D Points File
\throw GTXError
\param nsample Number of samples to be appended in this file
\param x Array of X coordinate of each sample
\param y Array of Y coordinate of each sample
\param z Array of Z coordinate of each sample

\par Remarks:
The X,Y and Z values must be given in the current unit if unit_mode = false
or according to the corresponding units if unit_mode = true.

Creating a Points File by appending samples 1 by 1 is not optimum at all.
If you know the number of samples from the beginning, you should directly
create the file with the right number of points.
*****************************************************************************/
void GTXClient::PointsFileAppend3D(gtx_long nsample, const double x[], const double y[], const double z[])
{
  _cleanupErrors();
  if (GTXClientPointsFileAppend(nsample, x, y, z))
    _throwErrors();
}

/*!
******************************************************************************
\brief Add a new line in the current 2D lines file with core definition

This function add a new line in the current 2D lines file. Coordinates are
given with beginning and end of cores.
\throw GTXError
\param nsample Number of sample to be added to the new line
\param xbegin X coordinate of the beginning of the first core
\param ybegin Y coordinate of the beginning of the first core
\param xend Array of X coordinate of the end of cores
\param yend Array of Y coordinate of the end of cores

\par Remarks:
\e Lines file must have been set before appending a line. Samples will be
converted to gravity centers. One sample will be added at the beginning of
the line to enable a full reconstruction of the line.

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClientPointsFileAppend() and link the lines file
to the points file using GTXClientLinesFileCreateLink().

The X and Y values must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::LinesFileAddCoreLine2D(int nsample, double xbegin, double ybegin, const double xend[], const double yend[])
{
   _cleanupErrors();
   if (GTXClientLinesFileAddCoreLine(nsample, xbegin, ybegin, 0., xend, yend, NULL))
     _throwErrors();
}

/*!
******************************************************************************
\brief Add a new line in the current 2D lines file with core definition

This function add a new line in the current 2D lines file. Coordinates are
given with beginning and end of cores.
\throw GTXError
\param nsample Number of sample to be added to the new line
\param xbegin X coordinate of the beginning of the first core
\param ybegin Y coordinate of the beginning of the first core
\param xend Array of X coordinate of the end of cores
\param yend Array of Y coordinate of the end of cores

\par Remarks:
\e Lines file must have been set before appending a line. Samples will be
converted to gravity centers. One sample will be added at the beginning of
the line to enable a full reconstruction of the line.

You should consider using GTXClient::LinesFileAddCoreLine2D to create
real cores in Isatis instead of converting to gravity centers like this
function does. Warning: if you use the new function, you must also remove
the storage of an additional undefined value at the beginning of each
line (see client/CPP/write_files.cpp for an example).

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClient::PointsFileAppend() and link the lines file
to the points file using GTXClient::LinesFileCreateLink().

The X and Y values must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::LinesFileAddLineCores2DOld(int nsample, double xbegin, double ybegin, const double xend[], const double yend[])
{
   _cleanupErrors();
   if (GTXClientLinesFileAddLineCoresOld(nsample, xbegin, ybegin, 0., xend, yend, NULL))
     _throwErrors();
}

/*!
******************************************************************************
\brief Add a new line in the current 3D lines file with core definition

This function add a new line in the current 3D lines file. Coordinates are
given with beginning and end of cores.
\throw GTXError
\param nsample Number of sample to be added to the new line
\param xbegin X coordinate of the beginning of the first core
\param ybegin Y coordinate of the beginning of the first core
\param zbegin Z coordinate of the beginning of the first core
\param xend Array of X coordinate of the end of cores
\param yend Array of Y coordinate of the end of cores
\param zend Array of Z coordinate of the end of cores

\par Remarks:
\e Lines file must have been set before appending a line. Samples will be
converted to gravity centers. One sample will be added at the beginning of
the line to enable a full reconstruction of the line.

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClientPointsFileAppend() and link the lines file
to the points file using GTXClientLinesFileCreateLink().

The X,Y and Z values must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::LinesFileAddCoreLine3D(int nsample, double xbegin,
                                       double ybegin, double zbegin,
                                       const double xend[],
                                       const double yend[],
                                       const double zend[])
{
  
  _cleanupErrors();
  if (GTXClientLinesFileAddCoreLine(nsample, xbegin, ybegin, zbegin, xend, yend, zend))
    _throwErrors();
}

/*!
******************************************************************************
\brief Add a new line in the current 3D lines file with core definition

This function add a new line in the current 3D lines file. Coordinates are
given with beginning and end of cores.
\throw GTXError
\param nsample Number of sample to be added to the new line
\param xbegin X coordinate of the beginning of the first core
\param ybegin Y coordinate of the beginning of the first core
\param zbegin Z coordinate of the beginning of the first core
\param xend Array of X coordinate of the end of cores
\param yend Array of Y coordinate of the end of cores
\param zend Array of Z coordinate of the end of cores

\par Remarks:
\e Lines file must have been set before appending a line. Samples will be
converted to gravity centers. One sample will be added at the beginning of
the line to enable a full reconstruction of the line.

You should consider using GTXClient::LinesFileAddCoreLine3D to create
real cores in Isatis instead of converting to gravity centers like this
function does. Warning: if you use the new function, you must also remove
the storage of an additional undefined value at the beginning of each
line (see client/CPP/write_files.cpp for an example).

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClientPointsFileAppend() and link the lines file
to the points file using GTXClientLinesFileCreateLink().

The X,Y and Z values must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::LinesFileAddLineCores3DOld(int nsample, double xbegin,
                                           double ybegin, double zbegin,
                                           const double xend[],
                                           const double yend[],
                                           const double zend[])
{
  
  _cleanupErrors();
  if (GTXClientLinesFileAddLineCoresOld(nsample, xbegin, ybegin, zbegin, xend, yend, zend))
    _throwErrors();
}

/*!
******************************************************************************
\brief Add a new line in the current 2D lines file with gravity centers definition

This function add a new line in the current 2D lines file. Coordinates are
given with gravity centers.
\throw GTXError
\param nsample Number of sample to be added to the new line
\param xg Array of X coordinate of the centers of gravity
\param yg Array of Y coordinate of the centers of gravity

\par Remarks:
\e Lines file must have been set before appending a line. Samples will be
converted to gravity centers. If you want to use this line as cores in
Isatis, you must add a sample at the beginning of the line.

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClientPointsFileAppend() and link the lines file
to the points file using GTXClientLinesFileCreateLink().

The X and Y values must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::LinesFileAddGravityLine2D(int nsample, const double xg[], const double yg[])
{
  _cleanupErrors();
  if (GTXClientLinesFileAddGravityLine(nsample, xg, yg, NULL))
    _throwErrors();
}

/*!
******************************************************************************
\brief Add a new line in the current 3D lines file with gravity centers definition

This function add a new line in the current 3D lines file. Coordinates are
given with gravity centers.
\throw GTXError
\param nsample Number of sample to be added to the new line
\param xg Array of X coordinate of the centers of gravity
\param yg Array of Y coordinate of the centers of gravity
\param zg Array of Z coordinate of the centers of gravity

\par Remarks:
\e Lines file must have been set before appending a line. Samples will be
converted to gravity centers. If you want to use this line as cores in Isatis,
you must add a sample at the beginning of the line.

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClientPointsFileAppend() and link the lines file
to the points file using GTXClientLinesFileCreateLink().

The X,Y and Z values must be given in the current unit if unit_mode = false
or according to the corresponding creation unit if unit_mode = true.
*****************************************************************************/
void GTXClient::LinesFileAddGravityLine3D(int nsample,
                                          const double xg[],
                                          const double yg[],
                                          const double zg[])
{
  _cleanupErrors();
  if (GTXClientLinesFileAddGravityLine(nsample, xg, yg, zg))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a Link between current Lines file and its header Points file

This function creates a link between the current Lines file and its associated
Points File. Lines file must have been set before and the Points file must
have as many samples as the Lines file has separate lines.
\throw GTXError
\param points_file points file to link to
*****************************************************************************/
void GTXClient::LinesFileCreateLink(const char *points_file)
{
  _cleanupErrors();
  if (GTXClientLinesFileCreateLink(points_file))
    _throwErrors();
}

/*!
******************************************************************************
\brief Modify the origin of the current file (must be a grid)

This function modify the origin of the current file (must be a grid)
\throw GTXError
\param x0 X origin of the grid
\param y0 Y origin of the grid
\param z0 Z origin of the grid (unused if 2D)

\par Remarks:
The new origin must be given in the current unit (\ref GTXClient::SetUnit) if
unit_mode = false (\ref GTXClient::SetUnitMode). Elsewhere, you must enter the new
origin according to each coordinate variable unit.
*****************************************************************************/
void GTXClient::GridModifyOrigin(double x0, double y0, double z0)
{
  _cleanupErrors();
  if (GTXClientGridModifyOrigin(x0, y0, z0))
    _throwErrors();
}

/*!
******************************************************************************
\brief Modify the mesh of the current file (must be a grid)
 
This function modify the mesh of the current file (must be a grid)
\throw GTXError
\param dx X size of one cell of the grid
\param dy Y size of one cell of the grid
\param dz Z size of one cell of the grid (unused if 2D)

\par Remarks:
The new mesh must be given in the current unit (\ref GTXClient::SetUnit) if
unit_mode = false (\ref GTXClient::SetUnitMode). Elsewhere, you must enter the new
mesh according to each coordinate variable unit.
*****************************************************************************/
void GTXClient::GridModifyMesh(double dx, double dy, double dz)
{
  _cleanupErrors();
  if (GTXClientGridModifyMesh(dx, dy, dz))
    _throwErrors();
}

/*!
******************************************************************************
\brief Modify the rotation of the current file (must be a grid)
 
This function modify the rotation of the current file (must be a grid)
\throw GTXError
\param az rotation around Z axis
\param ay rotation around the Y'-axis (new Y axis in the system after the first rotation) (unused in 2D)
\param ax rotation around the X"-axis (new X axis in the systemd after the two first rotations) (unused in 2D)
*****************************************************************************/
void GTXClient::GridModifyRotation(double az, double ay, double ax)
{
  _cleanupErrors();
  if (GTXClientGridModifyRotation(az, ay, ax))
    _throwErrors();
}

/*!
******************************************************************************
\brief Get the list of variables in the current file

This function retrieve, from the GTXserver, the list of available
variables in the current file.
\throw GTXError
\return Array of variable names
*****************************************************************************/
GTXStringArray GTXClient::GetVariableList()
{
  int nb_variable;
  char **variables;
  _cleanupErrors();
  if (GTXClientGetVariableList(&nb_variable, &variables))
    _throwErrors();
  GTXStringArray list = GTXStringArray(nb_variable, variables);
  variables = GTXClientFreeStringArray(nb_variable, variables);
  return list;
}

/*!
******************************************************************************
\brief Get the list of variables in the current file

This function retrieve, from the GTXserver, the list of available
variables in the current file.
\throw GTXError
\return Array of variable names
*****************************************************************************/
GTXStringArray GTXClient::GetVariableList(GTXVariableInfo::VariableType var_type)
{
  int nb_variable;
  char **variables;
  _cleanupErrors();
  if (GTXClientGetVariableListByType((GTX_VAR_TYPE)var_type, &nb_variable, &variables))
    _throwErrors();
  GTXStringArray list = GTXStringArray(nb_variable, variables);
  variables = GTXClientFreeStringArray(nb_variable, variables);
  return list;
}

/*!
******************************************************************************
\brief Check if a given variable exists

This function asks the server if a given variable exists (or for old servers,
it retrieves the list of variables and check it by itself).
\throw GTXError
\param variable variable name to check
\return false if it does not exists, true if it exists
*****************************************************************************/
bool GTXClient::VariableExists(const char *variable)
{
  int exists;
  _cleanupErrors();
  if (GTXClientVariableExists(variable, &exists))
    _throwErrors();
  return exists != 0;
}

/*!
******************************************************************************
\brief Set the current variable in the current file

Set the current variable in the current file
\throw GTXError
\param variable Name of the variable to be set as curent
*****************************************************************************/
void GTXClient::SetVariable(const char *variable)
{
  _cleanupErrors();
  if (GTXClientSetVariable(variable))
    _throwErrors();
}

/*!
******************************************************************************
\brief Delete the current variable

This function will delete the current variable. If successfull, the is no
current variable directory.

\throw GTXError
*****************************************************************************/
void GTXClient::DeleteVariable()
{
  _cleanupErrors();
  if (GTXClientDeleteVariable())
    _throwErrors();
}

/*!
******************************************************************************
\brief Set the current indice for the current macro variable

Set the current indice for the current macro variable.
\throw GTXError
\param indice Macro indice to set

\par Remarks:
Current variable should have been set before.
*****************************************************************************/
void GTXClient::SetIndice(const int indice)
{
  _cleanupErrors();
  if (GTXClientSetIndice(indice))
    _throwErrors();
}

/*!
******************************************************************************
\brief Set the current alpha indice for the current macro variable

Set the current alpha indice for the current macro variable.
\throw GTXError
\param alphaindice Macro alpha indice to set

\par Remarks:
Current variable should have been set before.
*****************************************************************************/
void GTXClient::SetAlphaIndice(const char* alphaindice)
{
  _cleanupErrors();
  if (GTXClientSetAlphaIndice(alphaindice))
    _throwErrors();
}

/*!
******************************************************************************
\brief Get the current variable information

This function retrieve, from the GTXserver, the information about the
current variable. If used on a length variable, information will be returned
using the current default unit (\ref GTXClient::SetUnit) or in the variable unit
if current unit mode is true (\ref GTXClient::SetUnitMode).
\throw GTXError
\retval var_info The current variable information

\par Remarks:
On a length variable, by default, the returned unit is the current unit
(\ref GTXClient::SetUnit). If you want to get its real unit, you must set
unit_mode to true before (\ref GTXClient::SetUnitMode).
*****************************************************************************/
GTXVariableInfo GTXClient::GetVariableInfo()
{
  GTXVariableInfo info;
  _cleanupErrors();
  if (GTXClientGetVariableInfo(&info._cinfo))
    _throwErrors();
  return info;
}

/*!
******************************************************************************
\brief Get the list of available indices for the current macro variable

This function retrieve, from the GTXserver, the list of available
indices for the current macro variable.
\throw GTXError
\return Array of available indices for the current macro variable
\sa GTXClient::GetMacroAlphaIndices to get the list of alphanumerical indices
*****************************************************************************/
GTXIntArray GTXClient::GetMacroIndices()
{
  int nb_indices, *indices;

  _cleanupErrors();
  if (GTXClientGetMacroIndices(&nb_indices, &indices))
    _throwErrors();
  GTXIntArray array = GTXIntArray(nb_indices, indices);
  indices = (int*)GTXClientFreePointer((void*)indices);
  return array;
}

/*!
******************************************************************************
\brief Get the list of available macro alpha indices for the current macro variable

This function retrieve, from the GTXserver, the list of available
alphanumerical indices for the current macro variable
\throw GTXError
\return Array of available macro indices for the current macro variable (may
be empty if not a macro or if macro without alpha indices)
\sa GTXClient::GetMacroIndices to get the list of numerical indices
*****************************************************************************/
GTXStringArray GTXClient::GetMacroAlphaIndices()
{
  int nb_indices, *indices;
  char **alpha_indices;

  _cleanupErrors();
  if (GTXClientGetMacroIndicesWithAlpha(&nb_indices, &indices, &alpha_indices))
    _throwErrors();
  if (alpha_indices == NULL)
    nb_indices = 0;
  GTXStringArray list = GTXStringArray(nb_indices, alpha_indices);
  indices = (int*)GTXClientFreePointer((void*)indices);
  alpha_indices = GTXClientFreeStringArray(nb_indices, alpha_indices);
  return list;
}

/*!
******************************************************************************
\brief Read samples from the current variable

This function read all samples from the current variable. The variable
must be an alphanumerical variable.
\throw GTXError
\param compress Samples masked by the current selection are transformed
                into test values if 0, or do not appear in the output array
                if this flag is set to 1
\return Array of character variable samples
*****************************************************************************/
GTXCharData GTXClient::ReadCharVariable(bool compress)
{
  char *test_val, **data;
  gtx_long nb_val;

  _cleanupErrors();
  if (GTXClientReadCharVariable(compress, &test_val, &nb_val, &data))
    _throwErrors();
  GTXCharData ddata;
  ddata.StorePointer(data, nb_val);
  ddata.SetUndefinedValue(test_val);
  return ddata;
}

/*!
******************************************************************************
\brief Read continuous samples from the current variable

This function read contiguous samples from the current variable. The variable
must be an alphanumerical variable.
\throw GTXError
\param first Sample Number of the first item to read (from 1 to nsample)
\param last  Sample Number of the last item to read (from 1 to nsample)
\param compress Samples masked by the current selection are transformed
                into test values if 0, or do not appear in the output array
                if this flag is set to 1
\retval data Array of character variable samples
*****************************************************************************/
GTXCharData GTXClient::ReadCharVariableSub(gtx_long first, gtx_long last, bool compress)
{
  char *test_val, **data;
  gtx_long nb_val;

  _cleanupErrors();
  if (GTXClientReadCharVariableSub(first, last, compress, &test_val, &nb_val, &data))
    _throwErrors();
  GTXCharData ddata;
  ddata.StorePointer(data, nb_val);
  ddata.SetUndefinedValue(test_val);
  return ddata;
}

/*!
******************************************************************************
\brief Read samples of a line from the current alphanumerical variable

This function read all samples of a line from the current alphanumerical variable.
\throw GTXError
\param line_index The line index [1..nline]
\return data      Array of variable samples

*****************************************************************************/
GTXCharData GTXClient::ReadLineCharVariable(int line_index)
{
  char **data,*test_val;
  int nb_val;

  _cleanupErrors();
  if (GTXClientReadLineCharVariable(line_index,
                                    &test_val,
                                    &nb_val,
                                    &data))
    _throwErrors();
  GTXCharData cdata;
  cdata._undef_val = test_val;
  cdata.StorePointer(data, nb_val);
  return cdata;
}

/*!
******************************************************************************
\brief Read samples of a line from the current variable

This function read all samples of a line from the current variable. The variable
must be an numerical variable.
\throw GTXError
\param line_index The line index [1..nline]
\return data      Array of variable samples

\par Remarks:
If the variable is a length one, data is returned in the current unit
(\ref GTXClient::SetUnit) if unit_mode is false (\ref GTXClient::SetUnitMode).
If the variable is a length one and unit_mode is true, the variable unit is
used instead.
*****************************************************************************/
GTXDoubleData GTXClient::ReadLineDoubleVariable(int line_index)
{
  double *data,test_val;
  int nb_val;

  _cleanupErrors();
  if (GTXClientReadLineDoubleVariable(line_index,
                                      &test_val,
                                      &nb_val,
                                      &data))
    _throwErrors();
  GTXDoubleData ddata;
  ddata._undef_val = test_val;
  ddata.StorePointer(data, nb_val);
  return ddata;
}

/*!
******************************************************************************
\brief Read samples from the current variable

This function read all samples from the current variable. The variable
must be an numerical variable.
\throw GTXError
\param compress Samples masked by the current selection are transformed
                into test values if 0, or do not appear in the output array
                if this flag is set to 1
\return data Array of variable samples

\par Remarks:
If the variable is a length one, data is returned in the current unit
(\ref GTXClient::SetUnit) if unit_mode is false (\ref GTXClient::SetUnitMode).
If the variable is a length one and unit_mode is true, the variable unit is
used instead.
*****************************************************************************/
GTXDoubleData GTXClient::ReadDoubleVariable(bool compress)
{
  double test_val, *data;
  gtx_long nb_val;

  _cleanupErrors();
  if (GTXClientReadDoubleVariable(compress, &test_val, &nb_val, &data))
    _throwErrors();
  GTXDoubleData ddata;
  ddata.StorePointer(data, nb_val);
  ddata.SetUndefinedValue(test_val);
  return ddata;
}

/*!
******************************************************************************
\brief Read a subpart of the samples from the current variable

This function read a subpart of the samples from the current variable.
The variable must be an numerical variable.
\throw GTXError
\param first Sample Number of the first item to read (from 1 to nsample)
\param last  Sample Number of the last item to read (from 1 to nsample)
\param compress Samples masked by the current selection are transformed
                into test values if 0, or do not appear in the output array
                if this flag is set to 1
\return data Array of variable samples

\par Remarks:
Warning: This functions needs a server version >= 4.0.7 for 4.0.x series or
>= 4.1.1 for 4.1.x series.
*****************************************************************************/
GTXDoubleData GTXClient::ReadDoubleVariableSub(gtx_long first, gtx_long last, bool compress)
{
  double test_val, *data;
  gtx_long nb_val;

  _cleanupErrors();
  if (GTXClientReadDoubleVariableSub(first, last, compress, &test_val, &nb_val, &data))
    _throwErrors();
  GTXDoubleData ddata;
  ddata.StorePointer(data, nb_val);
  ddata.SetUndefinedValue(test_val);
  return ddata;
}

/*!
******************************************************************************
\brief Create a new double variable in the current file

This function creates a new double variable (also known as numerical variable)
in the current file
\throw GTXError
\param name Name of the variable to be created
\param bit_length Number of bits per sample, authorized values are
                  1, 2, 4, 8, 16, 32 and 64
\param format variable format

\par Remarks:
If you use NULL as the unit string, the variable will be a length variable
with the current default unit symbol and factor (\ref GTXClient::SetUnit).
If the format_unit is part of the known length units list, the variable will
be a length one with the specified unit and corresponding factor, elsewhere
a float variable will be created.
If you need to have a length variable with different unit symbol and factor,
use \ref GTXClient::SetVariableUnit
*****************************************************************************/
void GTXClient::NewDoubleVariable(const char *name, int bit_length,
                                  const GTXVariableFormat &format)
{
  _cleanupErrors();
  if (GTXClientNewDoubleVariable(name, bit_length,
                                 (GTX_FORMAT_TYPE)format._type,
                                 format._length, format._digits,
                                 format._unit))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new character variable in the current file

This function create a new character variable in the current file.
\throw GTXError
\param name Name of the variable to be created
\param max_length Maximum length of string per sample (without \\0)
*****************************************************************************/
void GTXClient::NewCharVariable(const char *name, int max_length)
{
  _cleanupErrors();
  if (GTXClientNewCharVariable(name, max_length))
    _throwErrors();
}

/*!
******************************************************************************
\brief Set the current variable as the Line Name variable

This function sets the current variable as the line name of the file in which
it appears. The file must be a Points file linked to Lines file and the
variable must be alphanumerical.

\throw GTXError
*****************************************************************************/
void GTXClient::SetVariableAsLineName(void)
{
  _cleanupErrors();
  if (GTXClientSetVariableAsLineName())
    _throwErrors();
}

/*!
******************************************************************************
\brief Write data to the current numerical variable
 
This function writes data to the current numerical variable
\throw GTXError
\param compress Samples masked by the selection are in the array but ignored
                if 0, or do not appear in the array if this flag is set to 1
\param ddata Array of numerical variable samples
 
\par Remarks:
All the numerical variables are stored in double precision arrays

If the variable is a length one, data must be in the current unit
(\ref GTXClient::SetUnit) if unit_mode is false (\ref GTXClient::SetUnitMode).
If the variable is a length one and unit_mode is true, it must be expressed in
the variable unit.
*****************************************************************************/
void GTXClient::WriteDoubleVariable(bool compress, const GTXDoubleData &ddata)
{
  _cleanupErrors();
  if (GTXClientWriteDoubleVariable(compress, ddata._undef_val,
                                   ddata._count, ddata._values))
    _throwErrors();
}

/*!
******************************************************************************
\brief Write line data to the current numerical variable
 
This function writes line data to the current numerical variable
\throw GTXError
\param line_index  The line index [1..nline]
\param ddata       Array of numerical variable samples
 
\par Remarks:
All the numerical variables are stored in double precision arrays
*****************************************************************************/
void GTXClient::WriteLineDoubleVariable(int line_index, const GTXDoubleData &ddata)
{
  _cleanupErrors();
  if (GTXClientWriteLineDoubleVariable(line_index,
                                       ddata._undef_val,
                                       (int)ddata._count,
                                       ddata._values))
    _throwErrors();
}

/*!
******************************************************************************
\brief Write continuous subpart of data to the current numerical variable
 
This function writes a subpart of data to the current numerical variable
\throw GTXError
\param first Sample Number of the first item to write (from 1 to nsample)
\param last  Sample Number of the last item to write (from 1 to nsample)
\param compress Samples masked by the selection are in the array but ignored
                if 0, or do not appear in the array if this flag is set to 1
\param ddata Array of numerical variable samples
 
\par Remarks:
All the numerical variables are stored in double precision arrays
Warning: This functions needs a server version >= 4.0.7 for 4.0.x series or
>= 4.1.1 for 4.1.x series.
*****************************************************************************/
void GTXClient::WriteDoubleVariableSub(gtx_long first, gtx_long last,
                                       bool compress,
                                       const GTXDoubleData &ddata)
{
  _cleanupErrors();
  if (GTXClientWriteDoubleVariableSub(first, last, compress, ddata._undef_val,
                                      ddata._count, ddata._values))
    _throwErrors();
}

/*!
******************************************************************************
\brief Write data to the current alphanumerical variable
 
This function writes data to the current alphanumerical variable
\throw GTXError
\param compress Samples masked by the selection are in the array but ignored
                if 0, or do not appear in the array if this flag is set to 1
\param cdata Array of alphanumerical variable samples
*****************************************************************************/
void GTXClient::WriteCharVariable(bool compress, const GTXCharData &cdata)
{
  _cleanupErrors();
  if (GTXClientWriteCharVariable(compress, cdata._undef_val,
                                 cdata._count, (const char **)cdata._values))
    _throwErrors();
}

/*!
******************************************************************************
\brief Write continuous subpart of data to the current alphanumerical variable
 
This function writes a subpart of data to the current alphanumerical variable
\throw GTXError
\param first Sample Number of the first item to write (from 1 to nsample)
\param last  Sample Number of the last item to write (from 1 to nsample)
\param compress Samples masked by the selection are in the array but ignored
                if 0, or do not appear in the array if this flag is set to 1
\param cdata Array of alphanumerical variable samples
\par Remarks:
Warning: This functions needs a server version >= 6.0.8
*****************************************************************************/
void GTXClient::WriteCharVariableSub(gtx_long first, gtx_long last,
                                     bool compress, const GTXCharData &cdata)
{
  _cleanupErrors();
  if (GTXClientWriteCharVariableSub(first, last, compress, cdata._undef_val,
                                    cdata._count, (const char **)cdata._values))
    _throwErrors();
}

/*!
******************************************************************************
\brief Write line data to the current alphanumerical variable
 
This function writes line data to the current alphanumerical variable
\throw GTXError
\param line_index  The line index [1..nline]
\param cdata       Array of alphanumerical variable samples
 
\par Remarks:
All the numerical variables are stored in double precision arrays
*****************************************************************************/
void GTXClient::WriteLineCharVariable(int line_index, const GTXCharData &cdata)
{
  _cleanupErrors();
  if (GTXClientWriteLineCharVariable(line_index,
                                     cdata._undef_val,
                                     (int)cdata._count,
                                     cdata._values))
    _throwErrors();
}

/*!
******************************************************************************
\brief Get the comment of the current variable

Get the comment/description to the current variable
\throw GTXError
\retval comment returned comment

\par Remarks:
Warning: errors concerning a non compatible server are silently ignored.

C++ only: comment must be freed using
\code
comment = GTXClientFreePointer(comment)
\endcode
*****************************************************************************/
const char *GTXClient::GetVariableComment()
{
  char *comment;
  _cleanupErrors();
  if (GTXClientGetVariableComment(&comment))
    _throwErrors();
  return comment;
}

/*!
******************************************************************************
\brief Set a comment for the current variable

Set or append a comment/description to the current variable
\throw GTXError
\param append
  \arg 0 to set a comment
  \arg 1 to append to an existing one
\param comment Comment to be set or append to the existing one

\par Remarks:
Warning: errors concerning a non compatible server are silently ignored.
*****************************************************************************/
void GTXClient::SetVariableComment(bool append, const char *comment)
{
  _cleanupErrors();
  if (GTXClientSetVariableComment(append, comment))
    _throwErrors();
}

/*!
******************************************************************************
\brief Set the current selection

Set the current selection for further reading in the current file.
\throw GTXError
\param sel Name of the selection to be set as curent, use NULL or empty
            to unset a selection
*****************************************************************************/
void GTXClient::SetSelection(const char *sel)
{
  _cleanupErrors();
  if (GTXClientSetSelection(sel))
    _throwErrors();
}

/*!
******************************************************************************
\brief Change the unit factor and symbol of a variable

This functions allows to change the unit attached to a variable. If the factor
unit is part of the known units list, it changes the variable to a length one.
Then we can specify a different symbol to display. If the factor, is not known
as a length one, the variable will be float (e.g, not length) and the symbol
will not be used.
\throw GTXError
\param factor new unit factor
\param symbol new unit symbol (used for length or NULL)

\sa GTXClient::SetUnit for the list of know units.
*****************************************************************************/
void GTXClient::SetVariableUnit(const char *factor, const char *symbol)
{
  _cleanupErrors();
  if (GTXClientSetVariableUnit(factor, symbol))
    _throwErrors();
}

/*!
******************************************************************************
\brief Write a vendor data.
 
This function writes a vendor data .
\throw GTXError
\param level Where to write the Vendor Data
  \arg 0 for study
  \arg 1 for directory
  \arg 2 for file
  \arg 3 for variable
\param vdata The Vendor Data to be written
*****************************************************************************/
void GTXClient::VendorDataWrite(int level, const GTXVendorData &vdata)
{
  _cleanupErrors();
  if (GTXClientVendorDataWrite( level,vdata._vdata))
    _throwErrors();
}
/*!
******************************************************************************
\brief Read a Vendor Data from Isatis file system
 
This function reads a Vendor Data from the Isatis file system
\throw GTXError
\return Returned Vendor Data 
\param identifier Identifier of the Vendor Data
\param level Where to read the Vendor Data
  \arg 0 for study
  \arg 1 for directory
  \arg 2 for file
  \arg 3 for variable
*****************************************************************************/
GTXVendorData GTXClient::VendorDataRead(const char *identifier, int level)
{
  GTXVendorData vdata("",0);
  _cleanupErrors();
  if (GTXClientVendorDataRead(identifier, level, &vdata._vdata, NULL))
    _throwErrors();

  return vdata;
}

/*!
******************************************************************************
\brief Checks if a Vendor Data exists
 
This function verifies the existence of a Vendor Data
\throw GTXError
\return exists false if it does not exists, true if it exists
\param identifier Identifier of the Vendor Data
\param level Where to read the Vendor Data
  \arg 0 for study
  \arg 1 for directory
  \arg 2 for file
  \arg 3 for variable
*****************************************************************************/
bool GTXClient::VendorDataExists(const char *identifier, int level)
{
  int exists;
  _cleanupErrors();
  if (GTXClientVendorDataExists(identifier, level, &exists))
    _throwErrors();

  if (exists != 0)
    return true;
  else
    return false;
}

/*!
******************************************************************************
\brief Get version string from a Vendor Data
 
This function reads the version string from a stored Vendor Data
\throw GTXError
\return Returned Vendor Data version
\param identifier Identifier of the Vendor Data
\param level Where to read the Vendor Data
  \arg 0 for study
  \arg 1 for directory
  \arg 2 for file
  \arg 3 for variable
*****************************************************************************/
int GTXClient::VendorDataGetVersion(const char *identifier, int level)
{
  int version;
  _cleanupErrors();
  if (GTXClientVendorDataGetVersion(identifier, level, &version))
    _throwErrors();
  return version;
}

/*!
******************************************************************************
\brief Get fault system associated to the current file

This function retrieve, from the GTXserver, the fault system
associated to the current file. Faults in the fault system can be masked
with the given priority.
\throw GTXError
\return Fault system associated to the current file
\param priority Authorized priority

\par Remarks:
Only segments which have a priority greater or equal to
\b priority are returned. Faults which have no such segments
are removed.

You may use a 0 \b priority to get all the faults with all
their segments. Use the value from GTXClientGetFileInfo to
get the faults as they were saved in Isatis Fault Manager.

All coordinates are returned in the current unit (\\ref GTXClient::SetUnit) or
using the corresponding coordinate variable unit if unit_mode is set to 1
(\\ref GTXClient::SetUnitMode).

Some memory is allocated inside passed fault_system. Memory must be freed
using GTXClientFreeFaults.
*****************************************************************************/
GTXFaultSystem GTXClient::ReadFaults(int priority)
{
  GTXFaultSystem fs;
  _cleanupErrors();
  if (GTXClientReadFaults(priority, fs._fsys))
    _throwErrors();

  return fs;
}
/*!
******************************************************************************
\brief Write fault system to the current file
\throw GTXError
This function store a fault system in the current file.

\param fs Fault system to be stored in the current file
*****************************************************************************/
void GTXClient::WriteFaults(GTXFaultSystem fs)
{
  _cleanupErrors();
  if ( GTXClientWriteFaults(fs._fsys))
    _throwErrors();
}

/*!
******************************************************************************
\brief Create a new polygon file in the current directory

This function creates a new polygons file in the current directory.
\throw GTXError
\param name File name of the polygons file
\param psystem Polygon system to be created

\par Remarks:
The file will be created using the current unit (\ref GTXClient::SetUnit) or
the creation units (\ref GTXClient::SetCreationUnits) if unit_mode is 1
(\ref GTXClient::SetUnitMode).

The X,Y and Z values must be given in the current unit if unit_mode = 0
or according to the corresponding creation unit if unit_mode = 1.
*****************************************************************************/
void GTXClient::NewPolygonsFile(const char *name, const GTXPolygonSystem *psystem)
{
  _cleanupErrors();
  if ( GTXClientNewPolygonsFile(name,&psystem->_polysys ))
  _throwErrors();
  
}
/*!
******************************************************************************
\brief Get current file as polygon system (if possible)

This function retrieve, from the GTXserver, the polygon system
associated to the current file.
\return Polygon system associated to the current file
\throw GTXError
\par Remarks:
All coordinates are returned in the current unit (\ref GTXClient::SetUnit) or
using the corresponding coordinate variable unit if unit_mode is set to 1
(\ref GTXClient::SetUnitMode).
*****************************************************************************/
GTXPolygonSystem GTXClient::ReadPolygons()
{
   GTXPolygonSystem ps;
  _cleanupErrors();
  if (GTXClientReadPolygons(&ps._polysys))
    _throwErrors();

  return ps; 
}

/*!
******************************************************************************
\brief Add a list of indices for the current macro variable

This function add, from the GTXserver, the list of available
indices for the current macro variable.
\throw GTXError
\param indices Array of numerical indices
\param alpha_indices Array of alpha indices

*****************************************************************************/
void GTXClient::AddMacroIndicesWithAlpha( const GTXIntArray indices, const GTXStringArray alpha_indices)
{
  _cleanupErrors();
  if (GTXClientAddMacroIndicesWithAlpha((int)indices.GetCount(), indices.GetValues(), alpha_indices.GetValues()))
    _throwErrors();

}
/*!
******************************************************************************
\brief Add a list of indices for the current macro variable

This function add, from the GTXserver, the list of available
indices for the current macro variable.
\throw GTXError
\param indices Array of numerical indices

*****************************************************************************/
void GTXClient::AddMacroIndices( const GTXIntArray indices)
{
  _cleanupErrors();
  if (GTXClientAddMacroIndices((int)indices.GetCount(), indices.GetValues()))
    _throwErrors();

}
/*!
******************************************************************************
\brief Write selection data (0,1) to the current variable (must be a selection)
 
This function writes data to the current numerical/selection variable
\throw GTXError
\param data Array of variable samples (0,1)
 
\par Remarks:
The current selection (which may have been set by GTXClientSetSelection) is
ignored while writing the current variable's (which must also be a selection
variable) contents.
The variable must be have been chosen beforehand by using GTXClientSetVariable.
*****************************************************************************/
void GTXClient::WriteSelectionVariable(const GTXByteArray data)
{
  _cleanupErrors();
  if (GTXClientWriteSelectionVariable(data.GetCount(), data.GetValues()))
    _throwErrors();
}

/*!
******************************************************************************
\brief Write continuous subpart of selection data (0,1) to the current
       variable (must be a selection)
 
This function writes data to the current numerical/selection variable
\throw GTXError
\param first Sample Number of the first item to write (from 1 to nsample)
\param last  Sample Number of the last item to write (from 1 to nsample)
\param data Array of variable samples (0,1)
 
\par Remarks:
The current selection (which may have been set by GTXClientSetSelection) is
ignored while writing the current variable's (which must also be a selection
variable) contents.
The variable must be have been chosen beforehand by using GTXClientSetVariable.
*****************************************************************************/
void GTXClient::WriteSelectionVariableSub(const gtx_long first, const gtx_long last,
                                       const GTXByteArray data)
{
  _cleanupErrors();
  if (GTXClientWriteSelectionVariableSub(first, last, data.GetCount(), data.GetValues()))
    _throwErrors();
}

/*!
******************************************************************************
\brief Read selection from the current variable

This function read all selection from the current variable. The variable
must be an numerical variable.
\throw GTXError
\return data Array of variable selection
*****************************************************************************/
GTXByteArray GTXClient::ReadSelectionVariable()
{
  gtx_long nb_val;
  unsigned char *data;
  _cleanupErrors();
  if (GTXClientReadSelectionVariable(&nb_val, &data))
    _throwErrors();
  GTXByteArray ret(nb_val, data);
  data =(unsigned char*) GTXClientFreePointer((void*)data);
  return ret;
}
/*!
******************************************************************************
\brief Read a subpart of the selection from the current variable

This function read a subpart of the selection from the current variable.
The variable must be an numerical variable.
\throw GTXError
\param first Selection Number of the first item to read (from 1 to nselection)
\param last  Selection Number of the last item to read (from 1 to nselection)
\return  Array of variable selection


Warning: This functions needs a server version >= 4.0.7 for 4.0.x series or
>= 4.1.1 for 4.1.x series.
*****************************************************************************/
GTXByteArray GTXClient::ReadSelectionVariableSub(gtx_long first, gtx_long last)
{
  gtx_long nb_val;
  unsigned char *data;
  _cleanupErrors();
  if (GTXClientReadSelectionVariableSub(first, last, &nb_val, &data))
    _throwErrors();
  GTXByteArray ret(nb_val, data);
  data =(unsigned char*) GTXClientFreePointer((void*)data);
  return ret;
}

/*!
******************************************************************************
\brief Create a new Wireframes file in the current directory

This function creates a new wireframes file in the current directory.
\throw GTXError
\param name File name of the wireframes file to be created
\param wsys Wireframe system to be created

\par Remarks:
The file will be created using the current unit (\ref GTXClient::SetUnit) or
the creation units (\ref GTXClient::SetCreationUnits) if unit_mode is 1
(\ref GTXClient::SetUnitMode).

The X,Y and Z values must be given in the current unit if unit_mode = 0
or according to the corresponding creation unit if unit_mode = 1.
*****************************************************************************/
void GTXClient::NewWireframesFile(const char *name, const GTXWireframeSystem &wsys)
{
  _cleanupErrors();
  if (GTXClientNewWireframesFile(name, &wsys._wiresys))
  _throwErrors();
}

/*!
******************************************************************************
\brief Get current file as wireframe system (if possible)

This function retrieve, from the GTXserver, the wireframe system
associated to the current file.
\return Wireframe system associated to the current file
\throw GTXError
\par Remarks:
All coordinates are returned in the current unit (\ref GTXClient::SetUnit) or
using the corresponding coordinate variable unit if unit_mode is set to 1
(\ref GTXClient::SetUnitMode).
*****************************************************************************/
GTXWireframeSystem GTXClient::ReadWireframes()
{
   GTXWireframeSystem ws;
  _cleanupErrors();
  if (GTXClientReadWireframes(&ws._wiresys))
    _throwErrors();

  return ws; 
}
