#include <GTXDoubleArray.hpp>
#include <GTXClientP.h>         // for sprintf...
#include <GTXError.hpp>
#include <new>

static void st_check_new(const void *ptr,
                         const size_t length)
{
  char message[100];
  if (ptr != NULL) return;

  // We do not want to use new, because we want to be able to use
  // StorePointer and GTXClientFreeStringArray
  (void)sprintf(message,
                "GTXDoubleArray: Cannot allocate memory (%0.2f MBytes)",
                length / 1024. / 1024.);
  throw GTXError(message);
}

/*!
****************************************************************************
\param count number of values
\param values array of values to be copied
***************************************************************************/
GTXDoubleArray::GTXDoubleArray(gtx_long count, double values[])
{
  _count = count;
  _values = (double*)malloc((_count) * (sizeof(double)));
  st_check_new(_values, _count * sizeof(double));
  for (gtx_long i = 0; i < _count; i++)
    _values[i] = values[i];

  _ref = (int*)malloc(sizeof(int));
  *_ref = 1;

}

GTXDoubleArray::GTXDoubleArray()
{
  _count = 0;
  _values = (double*)0;
  _ref = (int*)0;
}

GTXDoubleArray::GTXDoubleArray(const GTXDoubleArray &in_dbldata)
{
  _count = in_dbldata._count;
  _values = (double*)0;
  if (_count != 0)
  {
    _values = in_dbldata._values;
  }
  _ref = in_dbldata._ref;
  if(_ref)
    (*_ref)++;
}

GTXDoubleArray &GTXDoubleArray::operator=(const GTXDoubleArray &in_dbldata)
{
  Unref();
  _count = in_dbldata._count;
  _values = (double*)0;
  _values = in_dbldata._values;
  _ref = in_dbldata._ref;
  if(_ref)
    (*_ref)++;

  return *this;
}

GTXDoubleArray::~GTXDoubleArray()
{
  Unref();
}

gtx_long GTXDoubleArray::GetCount() const
{
  return (_count);
}

void GTXDoubleArray::Resize(gtx_long count)
{
  double *vals2;
  gtx_long min_cnt = (count < _count)? count: _count;

  vals2 = (double*)malloc(sizeof(double)*count);
  st_check_new(vals2, count * sizeof(double));
  for (gtx_long i = 0; i < min_cnt; i++)
    vals2[i] = _values[i];
  for (gtx_long i = min_cnt; i < count; i++)
    vals2[i] = 0;

  Unref();

  _values = vals2;
  _count = count;
  _ref = (int*)malloc(sizeof(int));
  *_ref = 1;
}


double GTXDoubleArray::GetValue(gtx_long rank) const
{
  return (rank >= 0 && rank < _count? _values[rank]: -1);
}

void GTXDoubleArray::SetValue(gtx_long rank, double value)
{
  if (rank >= 0 && rank < _count)
    _values[rank] = value;
}

void GTXDoubleArray::SetValues(double values[])
{
  for (int i = 0; i < _count; i++)
    _values[i] = values[i];
}

const double *GTXDoubleArray::GetValues() const
{
  return _values;
}

void GTXDoubleArray::StorePointer(double *values, gtx_long count)
{
  Unref();
  _values = values;
  _count = count;
  _ref = (int*)malloc(sizeof(int));
  *_ref = 1;
}

void GTXDoubleArray::Unref()
{
  if (_ref && --(*_ref)==0){
    _values = (double*)GTXClientFreePointer(_values);
    _ref = (int*)GTXClientFreePointer(_ref);
  }
}

