#include <GTXStringArray.hpp>
#include <GTXClientP.h>         // for strdup and NULL
#include <GTXError.hpp>
#include <new>

static void st_check_new(const void *ptr,
                         const size_t length)
{
  char message[100];
  if (ptr != NULL) return;

  // We do not want to use new, because we want to be able to use
  // StorePointer and GTXClientFreeStringArray
  (void)sprintf(message,
                "GTXStringArray: Cannot allocate memory (%0.2f MBytes)",
                length / 1024. / 1024.);
  throw GTXError(message);
}

GTXStringArray::GTXStringArray(gtx_long count, const char*const *list)
{
  _count = count;
  _values = (char**)malloc(_count * sizeof(char*));
  st_check_new(_values, _count * sizeof(char*));
  for (gtx_long i = 0; i < _count; i++)
    _values[i] = (char*)strdup(list[i]);

  _ref = (int*)malloc(sizeof(int));
  *_ref = 1;
}

GTXStringArray::GTXStringArray()
{
  _count = 0;
  _values = NULL;
  _ref = (int*)0;
}

GTXStringArray::GTXStringArray(const GTXStringArray &in_chrdata)
{
  _count = in_chrdata._count;
  _values = NULL;
  if (_count != 0)
    _values = in_chrdata._values;
  _ref = in_chrdata._ref;
  if(_ref)
    (*_ref)++;
}

GTXStringArray &GTXStringArray::operator=(const GTXStringArray &in_chrdata)
{
  Unref();
  _count = in_chrdata._count;
  _values = NULL;

  _values = in_chrdata._values;
  _ref = in_chrdata._ref;
  if(_ref)
    (*_ref)++;

  
  return (*this);
}

GTXStringArray::~GTXStringArray()
{
  Unref();
  _count = 0;
}

gtx_long GTXStringArray::GetCount() const
{
  return (_count);
}

void GTXStringArray::Resize(gtx_long count)
{
  char **vals2;

  gtx_long min_cnt = (count < _count)? count: _count;

  vals2 = (char **)malloc(count * sizeof(char*));
  st_check_new(vals2, count * sizeof(char*));
  // duplicate first values
  for (gtx_long i = 0; i < min_cnt; i++)
    vals2[i] = (char*)strdup(_values[i]);
  // fill in new values
  for (gtx_long i = min_cnt; i < count; i++)
    vals2[i] = (char*)strdup("");

  Unref();

  _values = vals2;
  _count = count;
  _ref = (int*)malloc(sizeof(int));
  *_ref = 1;
}



const char *GTXStringArray::GetValue(gtx_long rank) const
{
  return (rank >= 0 && rank < _count? _values[rank]: "");
}

void GTXStringArray::SetValue(gtx_long rank, const char *value)
{
  if (rank >= 0 && rank < _count)
  {
    if (_values[rank])
      free(_values[rank]);
    _values[rank] = (char*)strdup(value);
  }
}

void GTXStringArray::SetValues(const char * const *values)
{
  for (gtx_long i = 0; i < _count; i++)
  {
    free(_values[i]);
    _values[i] = (char*)strdup(values[i]);
  }
}

const char **GTXStringArray::GetValues() const
{
  return (const char **)_values;
}

void GTXStringArray::StorePointer(char **values, gtx_long count)
{
  Unref();
  _values = values;
  _count = count;
  _ref = (int*)malloc(sizeof(int));
  *_ref = 1;
}

void GTXStringArray::Unref()
{
  if (_ref && --(*_ref)==0){
    _values = (char **)GTXClientFreeStringArray(_count, _values);
    _ref = (int*)GTXClientFreePointer(_ref);
  }
}

