#include <GTXVendorData.hpp>
#include <GTXFaultInfo.hpp>
#include <GTXClient.hpp>
#include <GTXClientP.h>

/*!
******************************************************************************
\brief Vendor Data Constructor
 
Constructs a vendor data from a string an a version number
\throw GTXError
\param identifier vendor data identifier
\param version vendor data version
*****************************************************************************/
GTXVendorData::GTXVendorData(const char *identifier, int version)
{
  _vdata =  GTXClientVendorDataNew(identifier , version);
}

/*!
******************************************************************************
\brief Vendor Data Copy Constructor
 
Constructs a vendor data by copying an existing one
\throw GTXError
\param vdata source vendor data
*****************************************************************************/
GTXVendorData::GTXVendorData(const GTXVendorData &vdata)
{
  _vdata = GTXClientVendorDataDuplicate(vdata._vdata );
}

/*!
******************************************************************************
\brief Vendor Data Copy = operator
 
Copies a vendor data to another one
\throw GTXError
\return copied vendor data
\param vdata source vendor data
*****************************************************************************/
GTXVendorData GTXVendorData::operator=(const GTXVendorData &vdata)
{
  _vdata = GTXClientVendorDataFree(_vdata);
  _vdata = GTXClientVendorDataDuplicate(vdata._vdata );
  return *this;
}

/*!
******************************************************************************
\brief Vendor Data Destructor
 
Deletes a Vendor Data contents
*****************************************************************************/
GTXVendorData::~GTXVendorData()
{
  _vdata = GTXClientVendorDataFree(_vdata);
}

/*!
******************************************************************************
\brief Add an integer attribute to a Vendor Data
 
This function adds an integer attribute to a Vendor Data
\throw GTXError
\param name Attribute name
\param value Value of the attribute
*****************************************************************************/
void GTXVendorData::AddAttributeInt(const char *name, int value)
{
  GTXClient::_cleanupErrors();
  if(GTXClientVendorDataAddAttributeInt(_vdata, name, value))
    GTXClient::_throwErrors();
}

/*!
******************************************************************************
\brief Add a double attribute to a Vendor Data
 
This function adds a double attribute to a Vendor Data
\throw GTXError
\param name Attribute name
\param value Value of the attribute
*****************************************************************************/
void GTXVendorData::AddAttributeDouble(const char *name, double value)
{
  GTXClient::_cleanupErrors();
  if(GTXClientVendorDataAddAttributeDouble(_vdata, name, value))
    GTXClient::_throwErrors();
}
/*!
******************************************************************************
\brief Add a string attribute to a Vendor Data

This function adds a string attribute to a Vendor Data
\throw GTXError
\param name Attribute name
\param value Value of the attribute
*****************************************************************************/
void GTXVendorData::AddAttributeString(const char *name, const char *value)
{
  GTXClient::_cleanupErrors();
  if(GTXClientVendorDataAddAttributeString(_vdata, name, value))
    GTXClient::_throwErrors();
}
/*!
******************************************************************************
\brief Add an integer array attribute to a Vendor Data
 
This function adds an integer array attribute to a Vendor Data
\throw GTXError
\param name Attribute name
\param values Array of integer
*****************************************************************************/
void GTXVendorData::AddAttributeIntArray(const char *name, const GTXIntArray &values)
{
  GTXClient::_cleanupErrors();
  if(GTXClientVendorDataAddAttributeIntArray(_vdata, name, (int)values.GetCount(), values.GetValues()))
    GTXClient::_throwErrors();
}
/*!
******************************************************************************
\brief Add a double array attribute to a Vendor Data
 
This function adds a double array attribute to a Vendor Data
\throw GTXError
\param name Attribute name
\param values Array of double
*****************************************************************************/
void GTXVendorData::AddAttributeDoubleArray(const char *name, const GTXDoubleArray &values)
{
  GTXClient::_cleanupErrors();
  if(GTXClientVendorDataAddAttributeDoubleArray(_vdata, name, (int)values.GetCount(), values.GetValues()))
    GTXClient::_throwErrors();
}
/*!
******************************************************************************
\brief Add a string array attribute to a Vendor Data
 
This function adds a double array attribute to a Vendor Data
\throw GTXError
\param name Attribute name
\param values Array of strings
*****************************************************************************/
void GTXVendorData::AddAttributeStringArray(const char *name, const GTXStringArray &values)
{
  GTXClient::_cleanupErrors();
  if(GTXClientVendorDataAddAttributeStringArray(_vdata,  name, (int)values.GetCount(), values.GetValues()))
    GTXClient::_throwErrors();
}
  
/*!
******************************************************************************
\brief Get value from an integer attribute of a Vendor Data
 
This function reads the value from an integer attribute of a Vendor Data
\return Returned integer value
\throw GTXError
\param name Name of the attribute to read
*****************************************************************************/
int GTXVendorData::GetAttributeInt(const char *name) const
{
  GTXClient::_cleanupErrors();
  int tmp;
  if(GTXClientVendorDataGetAttributeInt(_vdata, name, &tmp))
    GTXClient::_throwErrors();
  return tmp;
}
/*!
******************************************************************************
\brief Get value from a double attribute of a Vendor Data
 
This function reads the value from a double attribute of a Vendor Data
\return Returned double value
\throw GTXError
\param name Name of the attribute to read
*****************************************************************************/
double GTXVendorData::GetAttributeDouble(const char *name) const
{
  GTXClient::_cleanupErrors();
  double tmp;
  if(GTXClientVendorDataGetAttributeDouble(_vdata, name, &tmp))
    GTXClient::_throwErrors();
  return tmp;

}
/*!
******************************************************************************
\brief Get value from a string attribute of a Vendor Data
 
This function reads the value from a string attribute of a Vendor Data
\return Returned string value
\throw GTXError
\param name Name of the attribute to read
\par Remarks: returned value is a string from the vendor data contents, so
it does not have to (and should not) be freed.
*****************************************************************************/
const char* GTXVendorData::GetAttributeString(const char *name) const
{
  GTXVendorDataAttrRec *attr;

  GTXClient::_cleanupErrors();
  // We need a pointer on the memory value here because an allocation
  // here would need freeing and the other API don't need this.
  attr = _gtx_vendor_data_search_attribute(_vdata, name,
                                           _GTX_VENDOR_DATA_TYPE_STRING);
  if (attr == NULL)
    GTXClient::_throwErrors();

  return attr->values.val_string;
}
/*!
******************************************************************************
\brief Get value from an integer array attribute of a Vendor Data
 
This function reads the value from an integer array attribute of a Vendor Data
\return Returned integer values
\throw GTXError
\param name Name of the attribute to read
*****************************************************************************/
GTXIntArray GTXVendorData::GetAttributeIntArray(const char *name) const
{
  GTXClient::_cleanupErrors();
  int *tmp;
  int nval;
  if(GTXClientVendorDataGetAttributeIntArray(_vdata, name, &nval, &tmp))
    GTXClient::_throwErrors();
  GTXIntArray array = GTXIntArray(nval, tmp);
  tmp = (int*)GTXClientFreePointer(tmp);
  return array;
}
/*!
******************************************************************************
\brief Get value from a double array attribute of a Vendor Data
 
This function reads the value from a double array attribute of a Vendor Data
\return Returned doubles values
\throw GTXError
\param name Name of the attribute to read
*****************************************************************************/
GTXDoubleArray GTXVendorData::GetAttributeDoubleArray(const char *name) const
{
  GTXClient::_cleanupErrors();
  double* tmp;
  int nval;
  if (GTXClientVendorDataGetAttributeDoubleArray(_vdata, name, &nval, &tmp))
    GTXClient::_throwErrors();
  GTXDoubleArray darray(nval, tmp);
  tmp = (double*)GTXClientFreePointer(tmp);
  return darray;

}
/*!
******************************************************************************
\brief Get value from a string array attribute of a Vendor Data
 
This function reads the value from a string array attribute of a Vendor Data
\return Returned strings values
\throw GTXError
\param name Name of the attribute to read
*****************************************************************************/
GTXStringArray GTXVendorData::GetAttributeStringArray(const char *name) const
{
  GTXClient::_cleanupErrors();
  char ** tmp;
  int nval;
  if (GTXClientVendorDataGetAttributeStringArray(_vdata, name, &nval, &tmp))
    GTXClient::_throwErrors();
  GTXStringArray list = GTXStringArray(nval, tmp);
  tmp = GTXClientFreeStringArray(nval, tmp);
  return list;
}


/*!
******************************************************************************
\brief Get version string from a Vendor Data
 
This function reads the version string of a Vendor Data
\return Returned Vendor Data version

*****************************************************************************/
int GTXVendorData::GetVersion() const
{
  return _vdata->version;
}

/*!
******************************************************************************
\brief Get identifier string from a Vendor Data
 
This function reads the identifier string of a Vendor Data

\return Returned Vendor Data identifier
*****************************************************************************/
const char *GTXVendorData::GetIdentifier() const
{
  return _vdata->identificator;
}


