#include <GTXWireframe.hpp>
#include <GTXClientP.h>
#include <GTXClient.hpp>
#include <GTXError.hpp>

/*!
******************************************************************************
\brief Empty Constructor
*****************************************************************************/
GTXWireframe::GTXWireframe()
{
  _wireframe.name[0] = '\0';
  _wireframe.color[0] = 0;
  _wireframe.color[1] = 0;
  _wireframe.color[2] = 0;
  _wireframe.n_objects = 0;
  _wireframe.objects = NULL;
}
/*!
******************************************************************************
\brief Copy Constructor
*****************************************************************************/
GTXWireframe::GTXWireframe(const GTXWireframe &wireframe)
{
  GTXClientWireframeDuplicate(&wireframe._wireframe, &_wireframe);
}
/*!
******************************************************************************
\brief Equal Operator
*****************************************************************************/
GTXWireframe GTXWireframe::operator=(const GTXWireframe &wireframe)
{
  if (_wireframe.objects != NULL)
    free(_wireframe.objects);
  GTXClientWireframeDuplicate(&wireframe._wireframe, &_wireframe);
  return *this;
}
/*!
******************************************************************************
\brief Destructor
*****************************************************************************/
GTXWireframe::~GTXWireframe()
{
  if (_wireframe.objects)
    free(_wireframe.objects);
}

/*!
******************************************************************************
\brief Get the name attribute of a Wireframe
 
This function returns the name of a Wireframe
\return Returned name
*****************************************************************************/
const char *GTXWireframe::GetName() const
{
  return _wireframe.name;
}

/*!
******************************************************************************
\brief Get the Red component of a Wireframe color
 
This function returns the red component of a Wireframe
\return Returned red component [0-255]
*****************************************************************************/
int GTXWireframe::GetColorR() const
{
  return _wireframe.color[0];
}

/*!
******************************************************************************
\brief Get the green component of a Wireframe color
 
This function returns the green component of a Wireframe
\return Returned green component [0-255]
*****************************************************************************/
int GTXWireframe::GetColorG() const
{
  return _wireframe.color[1];
}

/*!
******************************************************************************
\brief Get the blue component of a Wireframe color
 
This function returns the blue component of a Wireframe
\return Returned blue component [0-255]
*****************************************************************************/
int GTXWireframe::GetColorB() const
{
  return _wireframe.color[2];
}

/*!
******************************************************************************
\brief Get the number of Objects of a Wireframe
 
This function returns thenumber of objects of a Wireframe
\return Returned number of objects
*****************************************************************************/
int GTXWireframe::GetObjectsNumber() const
{
  return _wireframe.n_objects;
}

/*!
******************************************************************************
\brief Get the n'th Object of a Wireframe
 
This function returns the n'th Object of a Wireframe
\return Returned Object
\param rank The Rank of the Object [between 0 and n_object-1]
*****************************************************************************/
GTXWireframeObject GTXWireframe::GetWireframeObject(int rank) const
{
  if (rank < 0 || rank > _wireframe.n_objects-1)
    throw GTXError("GTXWireframe::GetWireframeObject :Rank should be between 0 and Objects Number-1");
  GTXWireframeObject tmp;
  GTXClientWireframeObjectDuplicate(&_wireframe.objects[rank], &tmp._object); 
  return tmp;
}

/*!
******************************************************************************
\brief Set the name of a Wireframe 
 
This function sets the name of a Wireframe (49 characters at max)
\param name Wireframe's name
*****************************************************************************/
void GTXWireframe::SetName(const char *name) 
{
  (void)strncpy(_wireframe.name, name, 50);
  _wireframe.name[49] = '\0';
}

/*!
******************************************************************************
\brief Set color of a Wireframe 
 
This function changes the color of a Wireframe given its RGB components
\param r red component of the color [0-255]
\param g green component of the color [0-255]
\param b blue component of the color [0-255]
*****************************************************************************/
void GTXWireframe::SetColorRGB(int r, int g, int b)
{
  _wireframe.color[0] = r;
  _wireframe.color[1] = g;
  _wireframe.color[2] = b;
}

/*!
******************************************************************************
\brief add a WireframeObject to a Wireframe 
 
This function adds a Wireframe Object to a Wireframe 
\param object The wireframe object to be added
*****************************************************************************/
void GTXWireframe::AddObject(const GTXWireframeObject &object) 
{
  _wireframe.objects = (GTXWireframeObject_C *)
    realloc(_wireframe.objects, sizeof(GTXWireframeObject_C)*(_wireframe.n_objects+1));
  GTXClientWireframeObjectDuplicate(&object._object,
                                    &_wireframe.objects[_wireframe.n_objects]);
  _wireframe.n_objects++;
}
