#include <GTXWireframeObject.hpp>
#include <GTXClientP.h>
#include <GTXClient.hpp>
#include <GTXError.hpp>

/*!
******************************************************************************
\brief Empty Constructor
*****************************************************************************/
GTXWireframeObject::GTXWireframeObject()
{
  _object.n_vertices  = 0;
  _object.vertices    = NULL;
  _object.n_triangles = 0;
  _object.triangles   = NULL;
}

/*!
******************************************************************************
\brief Copy Constructor
*****************************************************************************/
GTXWireframeObject::GTXWireframeObject(const GTXWireframeObject &object)
{
  GTXClientWireframeObjectDuplicate(&object._object, &_object);
}

/*!
******************************************************************************
\brief Equal Operator
*****************************************************************************/
GTXWireframeObject GTXWireframeObject::operator=(const GTXWireframeObject &object)
{
  GTXClientWireframeObjectDuplicate(&object._object, &_object);
  return *this;
}

/*!
******************************************************************************
\brief Destructor
*****************************************************************************/
GTXWireframeObject::~GTXWireframeObject()
{
  if (_object.vertices)
    free(_object.vertices);
  if (_object.triangles)
    free(_object.triangles);
}

/*!
******************************************************************************
\brief Get the number of vertices of a Wireframe Object
 
This function returns the number of vertices of a Wireframe Object
\return number of vertices
*****************************************************************************/
int GTXWireframeObject::GetVerticesNumber()const
{
  return _object.n_vertices;
}

/*!
******************************************************************************
\brief Get X value of a Wireframe Object vertex
 
This function gets the X value from the i'th vertex of a Wireframe Object
\return Returned double value (expressed in current unit or current X unit)
*****************************************************************************/
double GTXWireframeObject::GetVertexX(int rank)const
{
  if (_object.vertices == NULL || rank < 0 || rank > _object.n_vertices-1)
    throw GTXError("GTXWireframeObject::GetVertexX : "
                   "Rank should be between 0 and Vertices Number-1");
  return _object.vertices[3*rank];
}

/*!
******************************************************************************
\brief Get Y value of a Wireframe Object vertex
 
This function gets the Y value from the i'th vertex of a Wireframe Object
\return Returned double value (expressed in current unit or current Y unit)
*****************************************************************************/
double GTXWireframeObject::GetVertexY(int rank)const
{
  if (_object.vertices == NULL || rank < 0 || rank > _object.n_vertices-1)
    throw GTXError("GTXWireframeObject::GetVertexY : "
                   "Rank should be between 0 and Vertices Number-1");
  return _object.vertices[3*rank+1];
}

/*!
******************************************************************************
\brief Get Z value of a Wireframe Object vertex
 
This function gets the Z value from the i'th vertex of a Wireframe Object
\return Returned double value (expressed in current unit or current Z unit)
*****************************************************************************/
double GTXWireframeObject::GetVertexZ(int rank)const
{
  if (_object.vertices == NULL || rank < 0 || rank > _object.n_vertices-1)
    throw GTXError("GTXWireframeObject::GetVertexZ : "
                   "Rank should be between 0 and Vertices Number-1");
  return _object.vertices[3*rank+2];
}

/*!
******************************************************************************
\brief Get the number of triangles of a Wireframe Object
 
This function returns the number of triangles of a Wireframe Object
\return number of triangles
*****************************************************************************/
int GTXWireframeObject::GetTrianglesNumber()const
{
  return _object.n_triangles;
}

/*!
******************************************************************************
\brief Get first point index of a Wireframe Object triangle
 
This function gets the first point index for the i'th triangle of a Wireframe Object
\return first point index (between 0 and n_vertices-1)
*****************************************************************************/
int GTXWireframeObject::GetTrianglePt1(int rank)const
{
  if (_object.triangles == NULL || rank < 0 || rank > _object.n_triangles-1)
    throw GTXError("GTXWireframeObject::GetTrianglePt1 : "
                   "Rank should be between 0 and Triangles Number-1");
  return _object.triangles[3*rank];
}

/*!
******************************************************************************
\brief Get second point index of a Wireframe Object triangle
 
This function gets the second point index for the i'th triangle of a Wireframe Object
\return second point index (between 0 and n_vertices-1)
*****************************************************************************/
int GTXWireframeObject::GetTrianglePt2(int rank)const
{
  if (_object.triangles == NULL || rank < 0 || rank > _object.n_triangles-1)
    throw GTXError("GTXWireframeObject::GetTrianglePt2 : "
                   "Rank should be between 0 and Triangles Number-1");
  return _object.triangles[3*rank+1];
}

/*!
******************************************************************************
\brief Get third point index of a Wireframe Object triangle
 
This function gets the third point index for the i'th triangle of a Wireframe Object
\return third point index (between 0 and n_vertices-1)
*****************************************************************************/
int GTXWireframeObject::GetTrianglePt3(int rank)const
{
  if (_object.triangles == NULL || rank < 0 || rank > _object.n_triangles-1)
    throw GTXError("GTXWireframeObject::GetTrianglePt3 : "
                   "Rank should be between 0 and Triangles Number-1");
  return _object.triangles[3*rank+2];
}

/*!
******************************************************************************
\brief Add a vertex to a Wireframe Object
 
This function appends a vertex to the vertices of a Wireframe Object
\par Remarks:
coordinates are expressed in the current unit or in the current
X,Y,Z units depending on the Unit Mode (\sa GTXClient::SetUnitMode)
*****************************************************************************/
void GTXWireframeObject::AddVertex(double x, double y, double z)
{
  _object.vertices = (double *)
    realloc(_object.vertices, sizeof(double)*3*(_object.n_vertices+1));
  if (_object.vertices == NULL)
    throw GTXError("GTXWireframeObject::AddVertex : Cannot reallocate vertices array");
  _object.vertices[3*_object.n_vertices]   = x;
  _object.vertices[3*_object.n_vertices+1] = y;
  _object.vertices[3*_object.n_vertices+2] = z;
  _object.n_vertices++;
}

/*!
******************************************************************************
\brief Add a triangle to a Wireframe Object
 
This function appends a triangle to the shape of a Wireframe Object
*****************************************************************************/
void GTXWireframeObject::AddTriangle(int pt1, int pt2, int pt3)
{
  if (!_object.vertices || pt1 < 0 || pt1 >= _object.n_vertices ||
      pt2 < 0 || pt2 >= _object.n_vertices ||
      pt3 < 0 || pt3 >= _object.n_vertices)
    throw GTXError("GTXWireframeObject::AddTriangle : "
                   "Triangle Points should have indices between 0 and Vertices Number-1");
  _object.triangles = (int *)
    realloc(_object.triangles, sizeof(int)*3*(_object.n_triangles+1));
  if (_object.triangles == NULL)
    throw GTXError("GTXWireframeObject::AddTriangle : Cannot reallocate vertices array");
  _object.triangles[3*_object.n_triangles]   = pt1;
  _object.triangles[3*_object.n_triangles+1] = pt2;
  _object.triangles[3*_object.n_triangles+2] = pt3;
  _object.n_triangles++;
}
