/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include "version.h"
#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_connect.c 17697 2010-12-15 09:43:13Z foucher $";

#if !defined(winnt)
#include <unistd.h> /* close() on solaris */
#else
#include <winsock.h>
#endif
#include <string.h> /* strcpy() on solaris */

int _gtx_client64 = 0;
int _gtx_server64 = 0;

int _gtx_client_linked_version = 0;
int _gtx_client_server_version = 0;

int _gtx_client_n_supported_requests = 0;
char **_gtx_client_supported_requests = NULL;

/*!
******************************************************************************
\brief Initialize the client library

This function checks consistency between the version of the library you linked
with and the current one. It also inititializes tracing. You must call it
before you first use GTXClientConnect.

\return error code:
  \arg 0 if Ok
  \arg 1 cannot initialize the library

\param linked_version is GTX_CLIENT_VERSION (used to determine the version
       of GTXclient you linked with if using dynamic linking).

\par Remarks:
Typical usage is: if (\ref GTXClientInitialize (GTX_CLIENT_VERSION))
If the library is already initialized, nothing is done and 0 is returned.
*****************************************************************************/
int GTXClientInitialize(int linked_version)
{
  GTXErrorCode error;

  _gtx_trace_init();
  GTX_TRACE_FUNC_START("GTXClientInitialize",1);
  GTX_TRACE(1,("(%d)", linked_version));

#ifdef GTX_EXE_64
  _gtx_client64 = 1;
#endif

  error = GTX_ERROR_NONE;
  if (_gtx_client_linked_version != 0)
    goto label_end;

  GTX_TRACE(1,("Full library version: %s (%s)", __REV_COM_VERSION_STRING__, __REV_VERSION_STRING__));
  GTX_TRACE(1,("Protocol Version: %d", GTX_CLIENT_VERSION));
  if (linked_version > GTX_CLIENT_VERSION)
  {
    _gtx_error("Bad GTXClient library (version %d > %d)",
               linked_version, GTX_CLIENT_VERSION);
    error = GTX_ERROR_PARAMETER;
    goto label_end;
  }
  _gtx_client_linked_version = linked_version;

  error = GTX_ERROR_NONE;
label_end:
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/* Sends HELLO to the server expecting to retrieve WELCOME */
/* It Retrieves the server version and manages error code */
static int st_send_hello(const char *path, GTXErrorCode *error)
{
  const char *s[1];
  char answer[10];
  int version;

  *error = GTX_ERROR_NETWORK;
  if (_gtx_start_packet("CHELLO")) goto label_end;
  s[0] = path;       /* send PATH because we may have an old server */
  if (!_gtx_write_string(_gtx_server_socket, s, 10000)) goto label_end;
  version = GTX_CLIENT_VERSION;
  if (!_gtx_write_int(_gtx_server_socket, &version)) goto label_end;
  if (_gtx_send_and_wait(answer)) goto label_end;

  *error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SWELCOME")) goto label_end;
  *error = GTX_ERROR_NETWORK;
  if (!_gtx_read_int(_gtx_server_socket,
                     &_gtx_client_server_version)) goto label_end;

  *error = GTX_ERROR_NONE;
 label_end:
  return (*error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Open a session with a running GTXserver to read isatis file system

This function opens a session with a GTXserver running on the host \e host
that listens to the port \e port and starts to read the Isatis file system
located at \e path. If the server is a mono session one, the server will remain
attached to this client until disconnect then it will quit. If the server is
a multi session one, the server will still be available for other clients.

\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)

\param [in] host hostname or IP address "aaa.bbb.ccc.ddd" of the machine where
the GTXserver is running on
\param [in] port port number where the GTXserver is listening to. If 0, a call to
GTXClientGetDefaultPort is made to determine the port where GTXserver may be
running.
\param [in] path pathname of the Isatis file system to be accessed (should be a
path entered by the user (GTX_DATA or GTX_INIT (see below))

\par Remarks:
A session must be closed using GTXClientDisconnect to free memory allocations.
When accessing Isatis < 4.1 data, the user must specify where its database is
stored. This place is known as GTX_DATA environment variable. You can get it
in Isatis from the Help->About menu.
Starting with Isatis 4.1, this variable is not useful anymore. Instead, an
hidden file is created to keep a list of studies which can be put anywhere on
disk. If accessing data linked to an Isatis >= 4.1, this variable should be
left empty (not NULL). However it is possible to override the default study
list that would then be taken by specifying the place where the study list
lies in. This directory is referred to as GTX_INIT in the Isatis Help -> About
Menu.

\sa GTXClientDisconnect, GTXClientGetDefaultPort
*****************************************************************************/
int GTXClientConnect(const char *host,
                     unsigned short port,
                     const char *path)
{
  unsigned short new_port;
  int i, winsock_ok;
  char answer[10];
  const char *s[1];
  GTXErrorCode error;

  winsock_ok = 0;
  error = GTX_ERROR_PARAMETER;

  /* Check if the library has been initialized */
  _gtx_trace_init();
  GTX_TRACE_FUNC_START("GTXClientConnect",1);
  GTX_TRACE(1, ("(%s,%d,%s)",host,port,path));
  CHECK_INIT(goto label_end;);

  error = GTX_ERROR_NETWORK;
  if (_gtx_winsock_init()) goto label_end;
  winsock_ok = 1;

  /* Determine Default port used by GTXserver */
  if (port == 0)
    port = GTXClientGetDefaultPort();

  if (_gtx_init_client(host, port)) goto label_end;

  /* Send HELLO to the server expecting to retrieve WELCOME */
  /* It Retrieves the server version and manages error code */
  if (st_send_hello(path, &error)) goto label_end;

  /* Version 100: connect OK */
  if (_gtx_client_server_version == 100)
  {
    error = GTX_ERROR_NONE;
    goto label_end;
  }

  /* With new servers, try to open a session */
  if (_gtx_start_packet("CNEWSESS")) goto label_end;
  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (!strcmp(answer, "SNEWPORT"))
  {
    (void)_gtx_check_answer(answer, answer);
    error = GTX_ERROR_NETWORK;
    if (!_gtx_read_int(_gtx_server_socket, &i)) goto label_end;
    new_port = (unsigned short)i;

    /* Close last and open new connection on server */
    _gtx_close_socket(&_gtx_server_socket);
    if (_gtx_connect(host, new_port, &_gtx_server_socket))
      goto label_end;

    /* Sends HELLO again to the new server which may have been forked
    ** and may not know our version (indeed on Windows only because its not
    ** a real fork but a new executed process. */
    if (st_send_hello(path, &error)) goto label_end;
  }
  else if (!strcmp(answer, "SNOSESS"))
    (void)_gtx_check_answer(answer, answer);
  else
    if (_gtx_check_answer(answer, "SNEWPORT")) goto label_end;

  /* Finally send CGTX_DATA */
  error = GTX_ERROR_NETWORK;
  s[0] = path;
  if (_gtx_start_packet("CGTXDATA")) goto label_end;
  if (!_gtx_write_string(_gtx_server_socket, s, 10000)) goto label_end;
  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SOK")) goto label_end;

  /* New version number style with new API to know the list of features */
  if (_gtx_client_server_version > 400000)
  {
    error = GTX_ERROR_NETWORK;
    if (_gtx_start_packet("CSUPREQ")) goto label_end;
    if (_gtx_send_and_wait(answer)) goto label_end;

    error = GTX_ERROR_SERVER;
    if (_gtx_check_answer(answer, "SSUPREQ")) goto label_end;
    if (!_gtx_read_int(_gtx_server_socket,
                       &_gtx_client_n_supported_requests)) goto label_end;
    _gtx_client_supported_requests =
      malloc(sizeof(char*)*_gtx_client_n_supported_requests);
    if (_gtx_client_supported_requests == NULL)
      goto label_end;
    for (i = 0; i < _gtx_client_n_supported_requests; i++)
      _gtx_client_supported_requests[i] = NULL;
    for (i = 0; i < _gtx_client_n_supported_requests; i++)
      if (!_gtx_read_string(_gtx_server_socket,
                            &_gtx_client_supported_requests[i], 10000))
        goto label_end;
  }

  /* Check architectures 32/64 bits */
  _gtx_server64 = 0;
  if (_gtx_is_supported("CARCH", 0))
  {
    error = GTX_ERROR_NETWORK;
    if (_gtx_start_packet("CARCH")) goto label_end;
    if (!_gtx_write_int(_gtx_server_socket, &_gtx_client64)) goto label_end;
    if (_gtx_send_and_wait(answer)) goto label_end;

    error = GTX_ERROR_SERVER;
    if (_gtx_check_answer(answer, "SARCH")) goto label_end;
    if (!_gtx_read_int(_gtx_server_socket, &_gtx_server64)) goto label_end;
  }

  error = GTX_ERROR_NONE;

label_end:
  if (error && winsock_ok)
  {
    _gtx_cleanup_socket();
    if (_gtx_server_socket != _GTX_SOCKET_ERROR)
    {
      if (_gtx_start_packet("CBYE BYE")) goto label_end;
      (void)_gtx_send_and_wait(answer);
      _gtx_close_socket(&_gtx_server_socket);
    }
  }
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Check is client is connected with the server.

Check the status of the connection with the GTXserver

\return connected:
  \arg 0 if not connected
  \arg 1 if connected

\sa GTXClientConnect, GTXClientDisconnect
*****************************************************************************/
int GTXClientIsConnected()
{
  return (_gtx_server_socket != _GTX_SOCKET_ERROR);
}

/*!
******************************************************************************
\brief Retrieves Server Protocol Version

When the client is connected to the server, allows to know the server protocol
version. The version is encoded as followed:
Major * 1000000 + Minor * 1000 + Patch * 10 + Sub-Version
It may now correspond to the Isatis version but to the major latest change in
the protocol.

\return connected:
  \arg server version

\sa GTXClientConnect
*****************************************************************************/
int GTXClientGetServerVersion()
{
  return (_gtx_client_server_version);
}
