/*****************************************************************************
 
            Copyright (c)2013 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_new_dir.c 11482 2008-01-04 20:37:41Z foucher $";

/*!
******************************************************************************
\brief Get the list of colors scales in the current study

This function retrieve, from the GTXserver, the list of available
color scales in the current study. 
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\retval nb_cscales Number of color scales in the current study
\retval cscales Array of color scale names

\par Remarks:
Color Scales must be freed using
\code
cscales = GTXClientFreeStringArray(nb_cscales, cscales);
\endcode

To check if server supports reading and writing color scales, use
\ref GTXClientServerHasFeature.
*****************************************************************************/
int GTXClientColorScaleGetList(int *nb_cscales, char ***cscales)
{
  GTXErrorCode error;
  char answer[10];
  int i;

  GTX_TRACE_FUNC_START("GTXClientColorScaleGetList",1);
  GTX_TRACE(1, ("(%p,%p)",nb_cscales,cscales));

  *nb_cscales = 0;
  *cscales = NULL;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_is_supported("CSCALST", 1))
    goto label_end;
  if( _gtx_start_packet("CSCALST") ) goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SSCALST") ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if( !_gtx_read_int(_gtx_server_socket, nb_cscales) ) goto label_end;

  error = GTX_ERROR_MEMORY;
  *cscales = (char**)malloc(sizeof(char*) * *nb_cscales);
  if( *cscales == NULL ) goto label_end;

  error = GTX_ERROR_NETWORK;
  for(i=0; i<*nb_cscales; i++) (*cscales)[i] = NULL;
  for(i=0; i<*nb_cscales; i++)
    if( !_gtx_read_string(_gtx_server_socket, &((*cscales)[i]), 10000) )
      goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if( error )
  {
    *cscales = GTXClientFreeStringArray(*nb_cscales, *cscales);
    *nb_cscales = 0;
    _gtx_cleanup_socket();
  }
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Check if a given color scale exists

This function asks the server if a given color scale exists.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param cscale_name color scale name to check
\retval exists 0 if it does not exists, 1 if it exists
*****************************************************************************/
int GTXClientColorScaleExists(const char *cscale_name, int *exists)
{
  GTXErrorCode error;
  char answer[10];

  GTX_TRACE_FUNC_START("GTXClientColorScaleExists",1);
  GTX_TRACE(1, ("(%s,%p)", cscale_name, exists));
  *exists = 0;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_is_supported("CSCAXIST", 1))
    goto label_end;
  if (_gtx_start_packet("CSCAXIST")) goto label_end;

  if (!_gtx_write_string(_gtx_server_socket, &cscale_name, 50))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SEXIST")) goto label_end;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_read_int(_gtx_server_socket, exists)) goto label_end;

  error = GTX_ERROR_NONE;
label_end:
  if (error)
    _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Inits a color scale structure

This function initializes a color scale structure to avoid problems with non-set
structure members afterwards (like at free-time).

\param cscale Color Scale to be initialized
*****************************************************************************/
void GTXClientColorScaleInit(GTXColorScale* cscale)
{
  cscale->type = GTX_CSCALE_TYPE_AUTOMATIC;
  cscale->readonly = 0;
  cscale->palette_name[0] = 0;
  cscale->nclasses = 0;
  cscale->bounds = NULL;
  cscale->pal_ind = NULL;
  cscale->facies_int_nb = 0;
  cscale->facies_integers = 0;
  cscale->facies_int_min = NULL;
  cscale->facies_int_max = NULL;
  cscale->facies_class_ind = NULL;
  cscale->undef_col[0] = cscale->undef_col[1] = cscale->undef_col[2] = -1;
  cscale->lowval_col[0] = cscale->lowval_col[1] = cscale->lowval_col[2] = -1;
  cscale->highval_col[0] = cscale->highval_col[1] = cscale->highval_col[2] = -1;
  cscale->bnd_inc_low = 2;
  cscale->bnd_inc_inter = 0;
  cscale->bnd_inc_high = 2;
  cscale->other_col[0] = cscale->other_col[1] = cscale->other_col[2] = -1;
  cscale->facies_rgbs = NULL;
  cscale->facies_labels = NULL;
  cscale->undef_label[0] = 0;
  cscale->other_label[0] = 0;
}

/*!
******************************************************************************
\brief Frees a color scale structure

This function frees a color scale structure. To avoid

\param cscale Color Scale to be freed
*****************************************************************************/
void GTXClientColorScaleFree(GTXColorScale* cscale)
{
  cscale->bounds           = GTXClientFreePointer(cscale->bounds);
  cscale->pal_ind          = GTXClientFreePointer(cscale->pal_ind);
  cscale->facies_int_min   = GTXClientFreePointer(cscale->facies_int_min);
  cscale->facies_int_max   = GTXClientFreePointer(cscale->facies_int_max);
  cscale->facies_class_ind = GTXClientFreePointer(cscale->facies_class_ind);
  cscale->facies_rgbs      = GTXClientFreePointer(cscale->facies_rgbs);
  cscale->facies_labels    = GTXClientFreeStringArray(cscale->nclasses,
                                                      cscale->facies_labels);
}

/*!
******************************************************************************
\brief Create a new color scale in the current study

This function creates a new color scale in the current study.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param name Color Scale name to be created
\param cscale Color Scale contents to store
*****************************************************************************/
int GTXClientColorScaleNew(const char *name, const GTXColorScale* cscale)
{
  GTXErrorCode error;
  const char *s1[1],*s2[1],*s3[1],*s4[1];
  char answer[10];
  int icl;
  int has_manual_bounds, has_facies_intervals;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientColorScaleNew",1);
  GTX_TRACE(1, ("(%s,%p)", name,cscale));
  if( _gtx_start_packet("CSCANEW") ) goto label_end;

  s1[0] = name;
  s2[0] = cscale->palette_name;
  s3[0] = cscale->undef_label;
  s4[0] = cscale->other_label;

  has_manual_bounds = (cscale->bounds != NULL && cscale->pal_ind != NULL);
  has_facies_intervals = (cscale->facies_int_min != NULL && cscale->facies_int_max != NULL &&
                          cscale->facies_class_ind != NULL && cscale->facies_rgbs != NULL &&
                          cscale->facies_labels != NULL);

  if (!_gtx_write_string(_gtx_server_socket, s1, 10000) ||
      !_gtx_write_int(_gtx_server_socket, (int*)&cscale->type) ||
      !_gtx_write_bool(_gtx_server_socket, &cscale->readonly) ||
      !_gtx_write_string(_gtx_server_socket, s2, 10000) ||
      !_gtx_write_int(_gtx_server_socket, &cscale->nclasses) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->undef_col, 3,
                      sizeof(int), (_gtx_proc_rw_t)_gtx_write_int) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->lowval_col, 3,
                      sizeof(int), (_gtx_proc_rw_t)_gtx_write_int) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->highval_col, 3,
                      sizeof(int), (_gtx_proc_rw_t)_gtx_write_int) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->other_col, 3,
                      sizeof(int), (_gtx_proc_rw_t)_gtx_write_int) ||
      !_gtx_write_int(_gtx_server_socket, &cscale->bnd_inc_low) ||
      !_gtx_write_int(_gtx_server_socket, &cscale->bnd_inc_inter) ||
      !_gtx_write_int(_gtx_server_socket, &cscale->bnd_inc_high) ||
      !_gtx_write_string(_gtx_server_socket, s3, 10000) ||
      !_gtx_write_string(_gtx_server_socket, s4, 10000) ||
      !_gtx_write_int(_gtx_server_socket, &cscale->facies_int_nb) ||
      !_gtx_write_bool(_gtx_server_socket, &cscale->facies_integers) ||
      !_gtx_write_bool(_gtx_server_socket, &has_manual_bounds) ||
      !_gtx_write_bool(_gtx_server_socket, &has_facies_intervals))
    goto label_end;

  if (has_manual_bounds &&
      (!_gtx_rw_vector(_gtx_server_socket, (char*)cscale->bounds, cscale->nclasses+1,
                       sizeof(double), (_gtx_proc_rw_t)_gtx_write_double) ||
       !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->pal_ind, cscale->nclasses,
                       sizeof(int), (_gtx_proc_rw_t)_gtx_write_int)))
    goto label_end;

  if (has_facies_intervals)
  {
    if (!_gtx_rw_vector(_gtx_server_socket, (char*)cscale->facies_int_min, cscale->facies_int_nb,
                        sizeof(double), (_gtx_proc_rw_t)_gtx_write_double) ||
        !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->facies_int_max, cscale->facies_int_nb,
                        sizeof(double), (_gtx_proc_rw_t)_gtx_write_double) ||
        !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->facies_class_ind, cscale->facies_int_nb,
                        sizeof(int), (_gtx_proc_rw_t)_gtx_write_int) ||
        !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->facies_rgbs, cscale->nclasses*3,
                        sizeof(int), (_gtx_proc_rw_t)_gtx_write_int))
      goto label_end;

    for (icl = 0; icl < cscale->nclasses; icl++)
    {
      s1[0] = cscale->facies_labels[icl];
      if (!_gtx_write_string(_gtx_server_socket, s1, 10000))
        goto label_end;
    }
  }

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SDONE") ) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Read a Color Scale from the Isatis database

This function retrieve, from the GTXserver, the contents of the color scale whose
name is given as argument.

\return error code
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)

\param cscale_name Name of the Color Scale to be read (must be initialized)
\retval cscale Read color scale contents

\par Remarks:
output cscale must have been initialized using GTXClientColorScaleInit and
must be freed using GTXClientColorScaleFree.
*****************************************************************************/
int GTXClientColorScaleRead(const char *cscale_name, GTXColorScale *cscale)
{
  GTXErrorCode error;
  char *s2[1],*s3[1],*s4[1];
  char answer[10];
  int icl,has_manual_bounds, has_facies_intervals;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientColorScaleRead",1);
  GTX_TRACE(1, ("(%p,%p)", cscale_name,cscale));
  if (_gtx_start_packet("CSCAREAD")) goto label_end;
  if (!_gtx_write_string(_gtx_server_socket, &cscale_name, 50))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SSCALE")) goto label_end;

  /* Reading part */
  s2[0] = cscale->palette_name;
  s3[0] = cscale->undef_label;
  s4[0] = cscale->other_label;

  if (!_gtx_read_int(_gtx_server_socket, (int*)&cscale->type) ||
      !_gtx_read_bool(_gtx_server_socket, &cscale->readonly) ||
      !_gtx_read_string(_gtx_server_socket, s2, 10000) ||
      !_gtx_read_int(_gtx_server_socket, &cscale->nclasses) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->undef_col, 3,
                      sizeof(int), (_gtx_proc_rw_t)_gtx_read_int) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->lowval_col, 3,
                      sizeof(int), (_gtx_proc_rw_t)_gtx_read_int) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->highval_col, 3,
                      sizeof(int), (_gtx_proc_rw_t)_gtx_read_int) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->other_col, 3,
                      sizeof(int), (_gtx_proc_rw_t)_gtx_read_int) ||
      !_gtx_read_int(_gtx_server_socket, &cscale->bnd_inc_low) ||
      !_gtx_read_int(_gtx_server_socket, &cscale->bnd_inc_inter) ||
      !_gtx_read_int(_gtx_server_socket, &cscale->bnd_inc_high) ||
      !_gtx_read_string(_gtx_server_socket, s3, 10000) ||
      !_gtx_read_string(_gtx_server_socket, s4, 10000) ||
      !_gtx_read_int(_gtx_server_socket, &cscale->facies_int_nb) ||
      !_gtx_read_bool(_gtx_server_socket, &cscale->facies_integers) ||
      !_gtx_read_bool(_gtx_server_socket, &has_manual_bounds) ||
      !_gtx_read_bool(_gtx_server_socket, &has_facies_intervals))
    goto label_end;

  if (has_manual_bounds && cscale->nclasses > 0)
  {
    cscale->bounds = malloc(sizeof(double)*(cscale->nclasses+1));
    cscale->pal_ind = malloc(sizeof(int)*(cscale->nclasses));
    if (!_gtx_rw_vector(_gtx_server_socket, (char*)cscale->bounds, cscale->nclasses+1,
                        sizeof(double), (_gtx_proc_rw_t)_gtx_read_double) ||
        !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->pal_ind, cscale->nclasses,
                        sizeof(int), (_gtx_proc_rw_t)_gtx_read_int))
      goto label_end;
  }

  if (has_facies_intervals && cscale->nclasses > 0 && cscale->facies_int_nb > 0)
  {
    cscale->facies_int_min = malloc(sizeof(double)*cscale->facies_int_nb);
    cscale->facies_int_max = malloc(sizeof(double)*cscale->facies_int_nb);
    cscale->facies_class_ind = malloc(sizeof(int)*cscale->facies_int_nb);
    if (!_gtx_rw_vector(_gtx_server_socket, (char*)cscale->facies_int_min, cscale->facies_int_nb,
                        sizeof(double), (_gtx_proc_rw_t)_gtx_read_double) ||
        !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->facies_int_max, cscale->facies_int_nb,
                        sizeof(double), (_gtx_proc_rw_t)_gtx_read_double) ||
        !_gtx_rw_vector(_gtx_server_socket, (char*)cscale->facies_class_ind, cscale->facies_int_nb,
                        sizeof(int), (_gtx_proc_rw_t)_gtx_read_int))
      goto label_end;

    cscale->facies_rgbs = malloc(sizeof(int)*cscale->nclasses*3);
    if (!_gtx_rw_vector(_gtx_server_socket, (char*)cscale->facies_rgbs, cscale->nclasses*3,
                        sizeof(int), (_gtx_proc_rw_t)_gtx_read_int))
      goto label_end;

    /* Allocate and initialize labels to NULL */
    cscale->facies_labels = malloc(sizeof(char*)*cscale->nclasses);
    for (icl = 0; icl < cscale->nclasses; icl++)
      cscale->facies_labels[icl] = NULL;
    for (icl = 0; icl < cscale->nclasses; icl++)
      if (!_gtx_read_string(_gtx_server_socket, &(cscale->facies_labels[icl]), 10000))
        goto label_end;
  }

  error = GTX_ERROR_NETWORK;

  error = GTX_ERROR_NONE;

label_end:
  if (error)
  {
    cscale->type = GTX_VAR_TYPE_INVALID;
    _gtx_cleanup_socket();
  }
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Deletes a color scale in the current study

This function deletes a color scale in the current study.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)

\param name Color Scale name to be deleted
*****************************************************************************/
int GTXClientColorScaleDelete(const char *name)
{
  GTXErrorCode error;
  const char *s[1];
  char answer[10];

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientColorScaleDelete",1);
  GTX_TRACE(1, ("(%s,%p)", name));
  if( _gtx_start_packet("CSCADEL") ) goto label_end;

  s[0] = name;
  if (!_gtx_write_string(_gtx_server_socket, s, 10000))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

