/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_get_faults.c 11546 2008-01-11 18:08:19Z foucher $";

#include <string.h> /* strcpy() on solaris */

/*!
******************************************************************************
\brief Get fault system associated to the current file

This function retrieve, from the GTXserver, the fault system
associated to the current file. Faults in the fault system can be masked
with the given priority.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param priority Authorized priority
\retval fault_system Fault system associated to the current file

\par Remarks:
Only segments which have a priority greater or equal to
\b priority are returned. Faults which have no such segments
are removed.

You may use a 0 \b priority to get all the faults with all
their segments. Use the value from GTXClientGetFileInfo to
get the faults as they were saved in Isatis Fault Manager.

All coordinates are returned in the current unit (\ref GTXClientSetUnit) or
using the corresponding coordinate variable unit if unit_mode is set to 1
(\ref GTXClientSetUnitMode).

Some memory is allocated inside passed fault_system. Memory must be freed
using GTXClientFreeFaults.
*****************************************************************************/
int GTXClientReadFaults(int priority, GTXFaultSystem *fault_system)
{
  char *s[1];
  int i, j;
  GTXErrorCode error;
  char answer[10];
  GTXFault *fault;
  GTXFaultSegment *segment;

  GTX_TRACE_FUNC_START("GTXClientReadFaults",1);
  GTX_TRACE(1, ("(%d,%p)", priority, fault_system));

  fault_system->faults = NULL;
  fault_system->n_faults = 0;

  error = GTX_ERROR_NETWORK;
  if( _gtx_start_packet("CFAULTS") ) goto label_end;
  if( !_gtx_write_int(_gtx_server_socket, &priority) ) goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SFAULTS") ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if( !_gtx_read_bool(_gtx_server_socket, &fault_system->faults_2d) ||
      !_gtx_read_bool(_gtx_server_socket, &fault_system->aux_var_used) ||
      !_gtx_read_double(_gtx_server_socket, &fault_system->aux_var_test) ||
      !_gtx_read_int(_gtx_server_socket, &fault_system->n_faults) )
    goto label_end;

  error = GTX_ERROR_MEMORY;
  fault_system->faults = (GTXFault*)malloc(sizeof(GTXFault) * fault_system->n_faults);
  if(fault_system->faults == NULL) goto label_end;
  for (i = 0; i < fault_system->n_faults; i++)
    fault_system->faults[i].segments = NULL;

  error = GTX_ERROR_NETWORK;
  fault = fault_system->faults;
  for(i=0; i<fault_system->n_faults; i++, fault++)
  {
    s[0] = fault->name;
    if( !_gtx_read_string(_gtx_server_socket, s, 8) ||
        !_gtx_read_bool(_gtx_server_socket, &fault->polygon) ||
        !_gtx_read_int(_gtx_server_socket, &fault->n_segments) )
      goto label_end;

    error = GTX_ERROR_MEMORY;
    fault->segments = (GTXFaultSegment*)malloc(sizeof(GTXFaultSegment) * fault->n_segments);
    if( fault->segments == NULL ) goto label_end;

    error = GTX_ERROR_NETWORK;
    segment = fault->segments;
    for(j=0; j<fault->n_segments; j++, segment++)
      if( !_gtx_read_double(_gtx_server_socket, &segment->X1) ||
          !_gtx_read_double(_gtx_server_socket, &segment->Y1) ||
          !_gtx_read_double(_gtx_server_socket, &segment->Z1) ||
          !_gtx_read_double(_gtx_server_socket, &segment->X2) ||
          !_gtx_read_double(_gtx_server_socket, &segment->Y2) ||
          !_gtx_read_double(_gtx_server_socket, &segment->Z2) ||
          !_gtx_read_double(_gtx_server_socket, &segment->X3) ||
          !_gtx_read_double(_gtx_server_socket, &segment->Y3) ||
          !_gtx_read_double(_gtx_server_socket, &segment->Z3) ||
          !_gtx_read_double(_gtx_server_socket, &segment->V1) ||
          !_gtx_read_double(_gtx_server_socket, &segment->V2) ||
          !_gtx_read_double(_gtx_server_socket, &segment->V3) ||
          !_gtx_read_int(_gtx_server_socket, &segment->priority) )
        goto label_end;
  } /* end of loop on faults */

  error = GTX_ERROR_NONE;

label_end:
  if (error)
  {
    GTXClientFreeFaults(fault_system);
    _gtx_cleanup_socket();
  }
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Frees memory allocated in the fault system

This function frees the memory allocated by a previous call to
GTXClientReadFaults
\param fault_system Fault system to be freed

\par Remarks:
This function is meant to only free memory allocated by GTXClientReadFaults.
Only call this function AFTER a call to GTXClientReadFaults.
Elsewhere memory in the fault system is not initialized and an attempt to
free it could cause problems.
Indeed this function frees fault_system->faults so if you may call this
function before read, first initialize faults to NULL.
*****************************************************************************/
void GTXClientFreeFaults(GTXFaultSystem *fault_system)
{
  int i;
  GTX_TRACE_FUNC_START("GTXClientFreeFaults",1);
  GTX_TRACE(1, ("(%p)", fault_system));
  if (fault_system->faults)
  {
    for (i=0; i<fault_system->n_faults; i++)
    {
      if (fault_system->faults[i].segments)
        free(fault_system->faults[i].segments);
    }
    free(fault_system->faults);
    fault_system->faults = NULL;
  }
  fault_system->n_faults = 0;
  GTX_TRACE_FUNC_END(NULL,NULL);
}

/*!
******************************************************************************
\brief Copies a given Fault Segment
 
This function copies a Fault Segment
\param fault_seg_in original Fault Segment
\retval fault_seg_out pointer on destination Fault Segment
*****************************************************************************/
void GTXClientFaultsSegmentDuplicate(const GTXFaultSegment *fault_seg_in,
                                     GTXFaultSegment *fault_seg_out)
{
  fault_seg_out->X1 = fault_seg_in->X1;
  fault_seg_out->Y1 = fault_seg_in->Y1;
  fault_seg_out->Z1 = fault_seg_in->Z1;
  fault_seg_out->X2 = fault_seg_in->X2;
  fault_seg_out->Y2 = fault_seg_in->Y2;
  fault_seg_out->Z2 = fault_seg_in->Z2;
  fault_seg_out->X3 = fault_seg_in->X3;
  fault_seg_out->Y3 = fault_seg_in->Y3;
  fault_seg_out->Z3 = fault_seg_in->Z3;
  fault_seg_out->V1 = fault_seg_in->V1;
  fault_seg_out->V2 = fault_seg_in->V2;
  fault_seg_out->V3 = fault_seg_in->V3;
  fault_seg_out->priority = fault_seg_in->priority;
}

/*!
******************************************************************************
\brief Copies a given Fault
 
This function copies a Fault
\param fault_in original Fault
\retval fault_out pointer on destination Fault
*****************************************************************************/
void GTXClientFaultDuplicate(const GTXFault *fault_in,
                             GTXFault *fault_out)
{
  int i;

  strcpy(fault_out->name,fault_in->name);
  fault_out->polygon = fault_in->polygon;
  fault_out->n_segments = fault_in->n_segments;
  fault_out->segments = malloc(fault_in->n_segments * sizeof(GTXFaultSegment));
  
  for( i = 0; i<fault_out->n_segments; i++)
    GTXClientFaultsSegmentDuplicate(&fault_in->segments[i], &fault_out->segments[i] );


}

/*!
******************************************************************************
\brief Copies a given Fault System
 
This function copies a Fault System
\return new copy of the Fault System
\param fault_sys original Fault System
*****************************************************************************/
GTXFaultSystem *GTXClientFaultSystemDuplicate(const GTXFaultSystem *fault_sys)
{
  int i;

  GTXFaultSystem *_fault_sys;
  _fault_sys = malloc(sizeof(GTXFaultSystem));

  _fault_sys->faults_2d = fault_sys->faults_2d;
  _fault_sys->aux_var_used = fault_sys->aux_var_used;
  _fault_sys->aux_var_test = fault_sys->aux_var_test;
  _fault_sys->n_faults = fault_sys->n_faults;
  _fault_sys->faults = malloc(fault_sys->n_faults * sizeof(GTXFault));

  for(i = 0; i<fault_sys->n_faults; i++)
    GTXClientFaultDuplicate(&fault_sys->faults[i], &_fault_sys->faults[i]);
  
  return _fault_sys;
}


