/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_get_file_list.c 26586 2016-10-21 15:08:13Z foucher $";

#include <string.h> /* strcpy() on solaris */

/****************************************************************************
**
** FUNCTION: st_get_file_list
**
** PURPOSE:  File List of the current directory
**
** RETURNS:  GTXserver error code...
**
** IN_ARGS:  file_type: GTX_FILE_TYPE
** IN_ARGS:  file_dim : 2,3 or -1 if ignored
**
** OUT_ARGS: nb_file  : returned number of files
** OUT_ARGS: files    : file list (must be freed)
**
*****************************************************************************/
static GTXErrorCode st_get_file_list(GTX_FILE_TYPE file_type,
                                     int file_dim,
                                     int *nb_file,
                                     char ***files)
{
  int i;
  GTXErrorCode error;
  char answer[10];

  *nb_file = 0;
  *files = NULL;

  error = GTX_ERROR_NETWORK;
  if (file_type == GTX_FILE_TYPE_ANY && file_dim == -1)
  {
    if (_gtx_start_packet("CFILELST")) goto label_end;
  }
  else
  {
    if (_gtx_start_packet("CFILETLST")) goto label_end;
    if (!_gtx_write_int(_gtx_server_socket, (int*)&file_type)) goto label_end;
    if (!_gtx_write_int(_gtx_server_socket, &file_dim)) goto label_end;
  }

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer,"SFILES") ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if( !_gtx_read_int(_gtx_server_socket, nb_file) ) goto label_end;

  error = GTX_ERROR_MEMORY;
  *files = (char**)malloc(sizeof(char*) * *nb_file);
  if( *files == NULL ) goto label_end;

  error = GTX_ERROR_NETWORK;
  for(i=0; i<*nb_file; i++) (*files)[i] = NULL;
  for(i=0; i<*nb_file; i++)
    if( !_gtx_read_string(_gtx_server_socket, &((*files)[i]), 10000)) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if( error )
  {
    *files = GTXClientFreeStringArray(*nb_file, *files);
    *nb_file = 0;
    _gtx_cleanup_socket();
  }

  _gtx_client_last_error = error;
  return error;
}

/*!
******************************************************************************
\brief Get the list of files in the current directory

This function retrieve, from the GTXserver, the list of available
files in the current directory. 
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\retval nb_file Number of files in the current directory
\retval files Array of file names

\par Remarks:
Files must be freed using
\code
files = GTXClientFreeStringArray(nb_files, files);
\endcode
*****************************************************************************/
int GTXClientGetFileList(int *nb_file, char ***files)
{
  GTXErrorCode error;
  GTX_TRACE_FUNC_START("GTXClientGetFileList",1);
  GTX_TRACE(1, ("(%p,%p)", nb_file, files));
  error = st_get_file_list(GTX_FILE_TYPE_ANY,-1,nb_file,files);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get the list of files in the current directory given a type and
       dimension

This function retrieve, from the GTXserver, the list of available
files in the current directory searching for a given type and dimension.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param file_type File type (\sa GTX_FILE_TYPE)
\param file_dim File dimension (2,3 or -1 if it does not matter)
\retval nb_file Number of files in the current directory
\retval files Array of file names

\par Remarks:
Files must be freed using
\code
files = GTXClientFreeStringArray(nb_files, files);
\endcode
*****************************************************************************/
int GTXClientGetFileListByType(GTX_FILE_TYPE file_type, int file_dim,
                               int *nb_file, char ***files)
{
  GTXErrorCode error;
  GTX_TRACE_FUNC_START("GTXClientGetFileList",1);
  GTX_TRACE(1, ("(%d,%d,%p,%p)",
              file_type, file_dim, nb_file, files));
  if ((file_type != GTX_FILE_TYPE_ANY || file_dim != -1) &&
      !_gtx_is_supported("CFILETLST",1))
  {
    error = GTX_ERROR_SERVER;
    goto label_end;
  }
  error = st_get_file_list(file_type,file_dim,nb_file,files);
label_end:
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Check if a given file exists

This function asks the server if a given file exists (or for old servers, it
retrieves the list of files and check it by itself).
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param file_name file name to check
\retval exists 0 if it does not exists, 1 if it exists
*****************************************************************************/
int GTXClientFileExists(const char *file_name, int *exists)
{
  GTXErrorCode error;
  char answer[10];

  GTX_TRACE_FUNC_START("GTXClientFileExists",1);
  GTX_TRACE(1, ("(%s,%p)", file_name, exists));
  *exists = 0;
  if (!_gtx_is_supported("CFILEXIST",0))
  {
    char **files = NULL;
    int i,nb_file;
    error = st_get_file_list(GTX_FILE_TYPE_ANY,-1,&nb_file,&files);
    if (!error)
    {
      for (i = 0; i < nb_file && !*exists; i++)
        if (!strcmp(files[i], file_name))
          *exists = 1;
      files = GTXClientFreeStringArray(nb_file, files);
    }
    goto label_end;
  }

  error = GTX_ERROR_NETWORK;
  if (_gtx_start_packet("CFILEXIST")) goto label_end;

  if (!_gtx_write_string(_gtx_server_socket, &file_name, 10000))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SEXIST")) goto label_end;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_read_int(_gtx_server_socket, exists)) goto label_end;

  error = GTX_ERROR_NONE;
label_end:
  if (error)
    _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}
