/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_get_var_list.c 26586 2016-10-21 15:08:13Z foucher $";

#include <string.h> /* strcpy() on solaris */

/****************************************************************************
**
** FUNCTION: st_get_var_list
**
** PURPOSE:  Variable List of the current directory
**
** RETURNS:  GTXserver error code...
**
** IN_ARGS:  var_type: Variable type
**
** OUT_ARGS: nb_var  : returned number of vars
** OUT_ARGS: vars    : variable list (must be freed)
**
*****************************************************************************/
static GTXErrorCode st_get_var_list(GTX_VAR_TYPE var_type,
                                    int *nb_var,
                                    char ***vars)
{
  int i;
  GTXErrorCode error;
  char answer[10];

  *nb_var = 0;
  *vars = NULL;

  error = GTX_ERROR_NETWORK;
  if (var_type == GTX_VAR_TYPE_ANY)
  {
    if (_gtx_start_packet("CVARLST")) goto label_end;
  }
  else
  {
    if (_gtx_start_packet("CVARTLST")) goto label_end;
    if (!_gtx_write_int(_gtx_server_socket, (int*)&var_type)) goto label_end;
  }

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SVARS") ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if( !_gtx_read_int(_gtx_server_socket, nb_var) ) goto label_end;

  error = GTX_ERROR_MEMORY;
  *vars = (char**)malloc(sizeof(char*) * *nb_var);
  if( *vars == NULL ) goto label_end;

  error = GTX_ERROR_NETWORK;
  for(i=0; i<*nb_var; i++) (*vars)[i] = NULL;
  for(i=0; i<*nb_var; i++)
    if( !_gtx_read_string(_gtx_server_socket, &((*vars)[i]), 10000) ) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if( error )
  {
    *vars = GTXClientFreeStringArray(*nb_var, *vars);
    *nb_var = 0;
    _gtx_cleanup_socket();
  }

  _gtx_client_last_error = error;
  return (GTXErrorCode)(error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get the list of variables in the current file

This function retrieve, from the GTXserver, the list of available
variables in the current file.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\retval nb_var Number of variables in the current file
\retval vars Array of variable names

\par Remarks:
Variables must be freed using:
\code
*vars = GTXClientFreeStringArray(*nb_var, *vars)
\endcode
*****************************************************************************/
int GTXClientGetVariableList(int *nb_var, char ***vars)
{
  GTXErrorCode error;
  GTX_TRACE_FUNC_START("GTXClientGetVariableList",1);
  GTX_TRACE(1, ("(%p,%p)", nb_var, vars));
  error = st_get_var_list(GTX_VAR_TYPE_ANY,nb_var,vars);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (int)(error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get the list of variables in the current file given a variable type

This function retrieve, from the GTXserver, the list of available
variables in the current file for a given type.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param var_type File type
  \arg GTX_VAR_TYPE_ANY will get the list of all variables
  \arg GTX_VAR_TYPE_SELECTION will get the list of selection variables
    but info on those variables will then
    return Float 1 bit.
  \arg GTX_VAR_TYPE_FLOAT_ALL will get the list of all float variables
    including selections and implicit but excluding macro variables.
  \arg GTX_VAR_TYPE_FLOAT will get the list of all float variables
    including selections but excluding implicit and macro variables.
  \arg GTX_VAR_TYPE_CHAR will get the list of all alphanumerical variables
  \arg GTX_VAR_TYPE_MACRO will get the list of all macro variables
  other variable types are forbidden
          
\retval nb_var Number of variables in the current file
\retval vars Array of variable names

\par Remarks:
Variables must be freed using:
\code
*vars = GTXClientFreeStringArray(*nb_var, *vars)
\endcode
*****************************************************************************/
int GTXClientGetVariableListByType(GTX_VAR_TYPE var_type,
                                   int *nb_var, char ***vars)
{
  GTXErrorCode error;
  GTX_TRACE_FUNC_START("GTXClientGetVariableListByType",1);
  GTX_TRACE(1, ("(%d,%p,%p)", var_type, nb_var, vars));
  if (var_type != GTX_VAR_TYPE_ANY &&
      !_gtx_is_supported("CVARTLST",1))
  {
    error = GTX_ERROR_SERVER;
    goto label_end;
  }

  if (var_type == GTX_VAR_TYPE_XG ||
      var_type == GTX_VAR_TYPE_YG ||
      var_type == GTX_VAR_TYPE_ZG)
  {
    _gtx_error("Coordinates variable names are within File Info");
    error = GTX_ERROR_PARAMETER;
    goto label_end;
  }
  if (var_type != GTX_VAR_TYPE_SELECTION &&
      var_type != GTX_VAR_TYPE_FLOAT_ALL &&
      var_type != GTX_VAR_TYPE_FLOAT &&
      var_type != GTX_VAR_TYPE_ANY &&
      var_type != GTX_VAR_TYPE_CHAR &&
      var_type != GTX_VAR_TYPE_MACRO)
  {
    _gtx_error("var_type (%d) is not allowed as selection type",var_type);
    error = GTX_ERROR_PARAMETER;
    goto label_end;
  }
  error = st_get_var_list(var_type,nb_var,vars);
label_end:
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (int)(error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Check if a given variable exists

This function asks the server if a given variable exists (or for old servers,
it retrieves the list of variables and check it by itself).
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param var_name variable name to check
\retval exists 0 if it does not exists, 1 if it exists
*****************************************************************************/
int GTXClientVariableExists(const char *var_name, int *exists)
{
  GTXErrorCode error;
  char answer[10];

  GTX_TRACE_FUNC_START("GTXClientVariableExists",1);
  GTX_TRACE(1, ("(%s,%p)", var_name, exists));
  *exists = 0;
  if (!_gtx_is_supported("CVARXIST",0))
  {
    char **vars = NULL;
    int i,nb_var;
    error = st_get_var_list(GTX_VAR_TYPE_ANY,&nb_var,&vars);
    if (!error)
    {
      for (i = 0; i < nb_var && !*exists; i++)
        if (!strcmp(vars[i], var_name))
          *exists = 1;
      vars = GTXClientFreeStringArray(nb_var, vars);
    }
    goto label_end;
  }

  error = GTX_ERROR_NETWORK;
  if (_gtx_start_packet("CVARXIST")) goto label_end;

  if (!_gtx_write_string(_gtx_server_socket, &var_name, 10000))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SEXIST")) goto label_end;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_read_int(_gtx_server_socket, exists)) goto label_end;

  error = GTX_ERROR_NONE;
label_end:
  if (error)
    _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (int)(error != GTX_ERROR_NONE);
}
