/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_new_lines_file.c 18576 2011-05-16 09:33:31Z meric $";

/*!
******************************************************************************
\brief Create a new lines file in the current directory

This function create a new lines file in the current directory.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param name Name of the new lines file
\param dimension File dimension 2 for 2D, 3 for 3D

\par Remarks:
The file will be created using the current unit (\ref GTXClientSetUnit) or
the creation units (\ref GTXClientSetCreationUnits) if unit_mode is 1
(\ref GTXClientSetUnitMode).
*****************************************************************************/
int GTXClientNewLinesFile(const char *name, int dimension)
{
  GTXErrorCode error;
  char answer[10];

  error = GTX_ERROR_PARAMETER;
  GTX_TRACE_FUNC_START("GTXClientNewLinesFile",1);
  GTX_TRACE(1, ("(%s,%d)", name,dimension));
  if( dimension < 2 || dimension > 3 )
  {
    _gtx_error("GTXClientNewLinesFile(): invalid dimension.");
    goto label_end;
  }

  error = GTX_ERROR_NETWORK;
  if( _gtx_start_packet("CLFILENEW") ) goto label_end;

  if( !_gtx_write_string(_gtx_server_socket, &name, 10000) ||
      !_gtx_write_int(_gtx_server_socket, &dimension) ) goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SDONE") ) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Add a new line in the current lines file with core definition

This function add a new line in the current lines file. Coordinates are
given with beginning and end of cores.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param nsample Number of sample to be added to the new line
\param xbeg X coordinate of the beginning of the first core
\param ybeg Y coordinate of the beginning of the first core
\param zbeg Z coordinate of the beginning of the first core
\param xend Array of X coordinate of the end of cores
\param yend Array of Y coordinate of the end of cores
\param zend Array of Z coordinate of the end of cores (or NULL)

\par Remarks:
\e zend must be NULL for adding a line to a 2D lines file. Lines file
must have been set before appending a line. Samples will be converted
to gravity centers. One sample will be added at the beginning of the line
to enable a full reconstruction of the line.

You should consider using GTXClientLinesFileAddCoreLine to create
real cores in Isatis instead of converting to gravity centers like this
function does. Warning: if you use the new function, you must also remove
the storage of an additional undefined value at the beginning of each
line (see client/C/write_files.c for an example).

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClientPointsFileAppend() and link the lines file
to the points file using GTXClientLinesFileCreateLink().

The X,Y and Z values must be given in the current unit if unit_mode = 0
or according to the corresponding creation unit if unit_mode = 1.
*****************************************************************************/
int GTXClientLinesFileAddLineCoresOld(int nsample,
                                      double xbeg, double ybeg, double zbeg,
                                      const double *xend, const double *yend,
                                      const double *zend)
{
  GTXErrorCode error;
  int i, size;
  int new_nsample, dimension;
  char answer[10];
  double *xarray, *yarray, *zarray;

  xarray = yarray = zarray = NULL;

  error = GTX_ERROR_PARAMETER;
  GTX_TRACE_FUNC_START("GTXClientLinesFileAddLineCoresOld",1);
  GTX_TRACE(1, ("(%d,%g,%g,%g,%p,%p,%p)",
                nsample,xbeg,ybeg,zbeg,xend,yend,zend));
  if( xend == NULL || yend == NULL )
  {
    _gtx_error("GTXClientLinesFileAddLineCores(): xend and yend can't be NULL.");
    goto label_end;
  }
  dimension = (zend == NULL) ? 2 : 3;

  error = GTX_ERROR_MEMORY;
  new_nsample = nsample + 1;
  size = sizeof(double) * (nsample + 1);
  xarray = (double*)malloc(size);
  if( xarray == NULL ) goto label_end;
  yarray = (double*)malloc(size);
  if( yarray == NULL ) goto label_end;
  if( dimension == 3 )
  {
    zarray = (double*)malloc(size);
    if( zarray == NULL ) goto label_end;
  }

  /* calculate gravity centers */
  xarray[0] = (3 * xbeg - xend[0]) / 2.;
  xarray[1] = (xend[0] + xbeg) / 2.;
  yarray[0] = (3 * ybeg - yend[0]) / 2.;
  yarray[1] = (yend[0] + ybeg) / 2.;
  if( dimension == 3 )
  {
    zarray[0] = (3 * zbeg - zend[0]) / 2.;
    zarray[1] = (zend[0] + zbeg) / 2.;
  }

  for(i=1; i<nsample; i++)
  {
    xarray[i+1] = (xend[i] + xend[i-1]) / 2.;
    yarray[i+1] = (yend[i] + yend[i-1]) / 2.;
    if( dimension == 3 ) zarray[i+1] = (zend[i] + zend[i-1]) / 2.;
  }
  error = GTX_ERROR_NETWORK;
  if( _gtx_start_packet("CLFILEAPP") ) goto label_end;

  if( !_gtx_write_int(_gtx_server_socket, &new_nsample) ||
      !_gtx_write_int(_gtx_server_socket, &dimension) ) goto label_end;


  if (!_gtx_rw_vector(_gtx_server_socket, (char*)xarray, new_nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)yarray, new_nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double))
    goto label_end;

  if (dimension == 3 &&
      !_gtx_rw_vector(_gtx_server_socket, (char*)zarray, new_nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double))
    goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SDONE") ) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if( xarray != NULL ) free(xarray);
  if( yarray != NULL ) free(yarray);
  if( zarray != NULL ) free(zarray);

  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Add a new line in the current lines file with core definition

This function add a new line in the current lines file. Coordinates are
given with beginning and end of cores.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param nsample Number of sample to be added to the new line
\param xbeg X coordinate of the beginning of the first core
\param ybeg Y coordinate of the beginning of the first core
\param zbeg Z coordinate of the beginning of the first core
\param xend Array of X coordinate of the end of cores
\param yend Array of Y coordinate of the end of cores
\param zend Array of Z coordinate of the end of cores (or NULL)

\par Remarks:
\e zend must be NULL for adding a line to a 2D lines file. Lines file
must have been set before appending a line. Samples will be converted
to gravity centers. One sample will be added at the beginning of the line
to enable a full reconstruction of the line.

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClientPointsFileAppend() and link the lines file
to the points file using GTXClientLinesFileCreateLink().

The X,Y and Z values must be given in the current unit if unit_mode = 0
or according to the corresponding creation unit if unit_mode = 1.

Replaces GTXClientLinesFileAddLineCores.
Warning: if you used the old GTXClientLinesFileAddLineCores function, you must
remove the storage of an additional undefined value at the beginning of each
line (see client/C/write_files.c for an example).
*****************************************************************************/
int GTXClientLinesFileAddCoreLine(int nsample,
                                  double xbeg, double ybeg, double zbeg,
                                  const double *xend, const double *yend,
                                  const double *zend)
{
  GTXErrorCode error;
  char answer[10]; 
  int dimension;

  error = GTX_ERROR_PARAMETER;
  GTX_TRACE_FUNC_START("GTXClientLinesFileAddCoreLine",1);
  GTX_TRACE(1, ("(%d,%g,%g,%g,%p,%p,%p)",
                nsample,xbeg,ybeg,zbeg,xend,yend,zend));
  if( xend == NULL || yend == NULL )
  {
    _gtx_error("GTXClientLinesFileAddLineCores(): xend and yend can't be NULL.");
    goto label_end;
  }
  dimension = (zend == NULL) ? 2 : 3;
  error = GTX_ERROR_NETWORK;
  if (!_gtx_is_supported("CLFILEAPC", 0))
  {
    error = (GTXErrorCode)GTXClientLinesFileAddLineCoresOld(nsample, xbeg, ybeg, zbeg, xend, yend, zend);
    goto label_end;
  }

  if( _gtx_start_packet("CLFILEAPC") ) goto label_end;

  if( !_gtx_write_int(_gtx_server_socket, &nsample) ||
      !_gtx_write_int(_gtx_server_socket, &dimension) ) goto label_end;

  if( !_gtx_write_double(_gtx_server_socket, &xbeg))
    goto label_end;
  if( !_gtx_write_double(_gtx_server_socket, &ybeg))
    goto label_end;

  if (dimension == 3)
    if( !_gtx_write_double(_gtx_server_socket, &zbeg))
      goto label_end;


  if (!_gtx_rw_vector(_gtx_server_socket, (char*)xend, nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)yend, nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double))
    goto label_end;

  if (dimension == 3 &&
      !_gtx_rw_vector(_gtx_server_socket, (char*)zend, nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double))
    goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SDONE") ) goto label_end;

  error = GTX_ERROR_NONE;

label_end:

  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Add a new line in the current lines file with gravity centers definition

This function add a new line in the current lines file. Coordinates are
given with gravity centers.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param nsample Number of sample to be added to the new line
\param xg Array of X coordinate of the centers of gravity
\param yg Array of Y coordinate of the centers of gravity
\param zg Array of Z coordinate of the centers of gravity (or NULL)

\par Remarks:
\e zg must be NULL for adding a line to a 2D lines file. Lines file
must have been set before appending a line. Samples will be converted
to gravity centers. If you want to use this line as cores in Isatis, you must
add a sample at the beginning of the line.

When appending samples to a lines file linked to a points file, the link is
deleted. If you want to restore it, you must append one sample per added line
to the points file using GTXClientPointsFileAppend() and link the lines file
to the points file using GTXClientLinesFileCreateLink().

The X,Y and Z values must be given in the current unit if unit_mode = 0
or according to the corresponding creation unit if unit_mode = 1.

replaces GTXClientLinesFileAddLineCenters.
*****************************************************************************/
int GTXClientLinesFileAddGravityLine(int nsample,
                                     const double *xg, const double *yg,
                                     const double *zg)
{
  GTXErrorCode error;
  int dimension;
  char answer[10];

  dimension = (zg == NULL) ? 2 : 3;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientLinesFileAddGravityLine",1);
  GTX_TRACE(1, ("(%d,%p,%p,%p)", nsample,xg,yg,zg));
  if( _gtx_start_packet("CLFILEAPP") ) goto label_end;

  if( !_gtx_write_int(_gtx_server_socket, &nsample) ||
      !_gtx_write_int(_gtx_server_socket, &dimension) )
    goto label_end;

  if (!_gtx_rw_vector(_gtx_server_socket, (char*)xg, nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)yg, nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double))
    goto label_end;

  if (dimension == 3 &&
      !_gtx_rw_vector(_gtx_server_socket, (char*)zg, nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double))
    goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SDONE") ) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Create a Link between current Lines file and its header Points file

This function creates a link between the current Lines file and its associated
Points File. Lines file must have been set before and the Points file must
have as many samples as the Lines file has separate lines.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param points_file points file to link to
*****************************************************************************/
int GTXClientLinesFileCreateLink(const char *points_file)
{
  GTXErrorCode error;
  char answer[10];

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientLinesFileCreateLink",1);
  GTX_TRACE(1, ("(%s)",points_file));
  if (_gtx_start_packet("CLFILELNK")) goto label_end;

  if (!_gtx_write_string(_gtx_server_socket, &points_file, 10000))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}
