/*****************************************************************************
 
            Copyright (c)2013 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_new_dir.c 11482 2008-01-04 20:37:41Z foucher $";

/*!
******************************************************************************
\brief Get the list of palettes in the current study

This function retrieve, from the GTXserver, the list of available
palettes in the current study.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\retval nb_pal Number of palettes in the current study
\retval palettes Array of palette names

\par Remarks:
Palettes must be freed using
\code
palettes = GTXClientFreeStringArray(nb_pal, palettes);
\endcode

Warning: System palettes are not listed here and can be read/written.

To check if server supports reading and writing palettes, use
\ref GTXClientServerHasFeature.
*****************************************************************************/
int GTXClientPaletteGetList(int *nb_pal, char ***palettes)
{
  GTXErrorCode error;
  char answer[10];
  int i;

  GTX_TRACE_FUNC_START("GTXClientPaletteGetList",1);
  GTX_TRACE(1, ("(%p,%p)",nb_pal,palettes));

  *nb_pal = 0;
  *palettes = NULL;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_is_supported("CPALLST", 1))
    goto label_end;
  if( _gtx_start_packet("CPALLST") ) goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SPALLST") ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if( !_gtx_read_int(_gtx_server_socket, nb_pal) ) goto label_end;

  error = GTX_ERROR_MEMORY;
  *palettes = (char**)malloc(sizeof(char*) * *nb_pal);
  if (*palettes == NULL) goto label_end;

  error = GTX_ERROR_NETWORK;
  for (i=0; i<*nb_pal; i++) (*palettes)[i] = NULL;
  for (i=0; i<*nb_pal; i++)
    if (!_gtx_read_string(_gtx_server_socket, &((*palettes)[i]), 10000))
      goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if( error )
  {
    *palettes = GTXClientFreeStringArray(*nb_pal, *palettes);
    *nb_pal = 0;
    _gtx_cleanup_socket();
  }
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Check if a given palette exists

This function asks the server if a given palette exists in the Isatis database.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)

\param pal_name palette name to check
\retval exists 0 if it does not exists, 1 if it exists

\par Remarks:
Warning: System palettes cannot be checked for existence.
*****************************************************************************/
int GTXClientPaletteExists(const char *pal_name, int *exists)
{
  GTXErrorCode error;
  char answer[10];

  GTX_TRACE_FUNC_START("GTXClientPaletteExists",1);
  GTX_TRACE(1, ("(%s,%p)", pal_name, exists));
  *exists = 0;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_is_supported("CPALXIST", 1))
    goto label_end;
  if (_gtx_start_packet("CPALXIST")) goto label_end;

  if (!_gtx_write_string(_gtx_server_socket, &pal_name, 50))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SEXIST")) goto label_end;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_read_int(_gtx_server_socket, exists)) goto label_end;

  error = GTX_ERROR_NONE;
label_end:
  if (error)
    _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Create/Updates a palette scale in the current study

This function creates or overwrite a palette in the current study.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)

\param pal_name Name of the Palette to be created/written
\param ncol Number of Colors
\param rgbs RGBs of the palette colors. Each value must [0-255]. Values are
ordered RRRGGGBBB with as many R,G and Bs as colors.

\par Remarks:
Palette can already exist and will be updated.
Warning: System palettes cannot be overwritten.
*****************************************************************************/
int GTXClientPaletteWrite(const char *pal_name, int ncol, const int* rgbs)
{
  GTXErrorCode error;
  const char *s[1];
  char answer[10];

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientPaletteWrite",1);
  GTX_TRACE(1, ("(%s,%d,%p)", pal_name,ncol,rgbs));
  if (!_gtx_is_supported("CPALWRIT", 1))
    goto label_end;
  if( _gtx_start_packet("CPALWRIT") ) goto label_end;

  s[0] = pal_name;
  if (!_gtx_write_string(_gtx_server_socket, s, 10000) ||
      !_gtx_write_int(_gtx_server_socket, &ncol) ||
      !_gtx_rw_vector(_gtx_server_socket, (char*)rgbs, 3*ncol,
		      sizeof(int), (_gtx_proc_rw_t)_gtx_write_int))
    goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SDONE") ) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Read a Palette from the Isatis database

This function retrieve, from the GTXserver, the contents of the palette whose
name is given as argument.

\return error code
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)

\param pal_name Name of the Palette to be read
\retval ncol Number of Colors in the palette
\retval rgbs Color RGBs of the palettes (order RRRRGGGGBBB with as many R as colors)

\par Remarks:
Color RGBs must be freed using
\code
rgbs = GTXClientFreePointer(rgbs);
\endcode

Warning: System palettes cannot be read.
*****************************************************************************/
int GTXClientPaletteRead(const char *pal_name, int* ncol, int **rgbs)
{
  GTXErrorCode error;
  const char *s[1];
  char answer[10];

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientPaletteRead",1);
  GTX_TRACE(1, ("(%s,%p,%p)", pal_name, ncol, rgbs));
  if (!_gtx_is_supported("CPALREAD", 1))
    goto label_end;
  if (_gtx_start_packet("CPALREAD")) goto label_end;

  s[0] = pal_name;
  if (!_gtx_write_string(_gtx_server_socket, s, 10000))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SPAL")) goto label_end;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_read_int(_gtx_server_socket, ncol))
    goto label_end;
  if (*ncol > 0)
  {
    *rgbs = malloc(sizeof(int)*(*ncol)*3);
    if (!_gtx_rw_vector(_gtx_server_socket, (char*)*rgbs, (*ncol)*3,
			sizeof(int), (_gtx_proc_rw_t)_gtx_read_int))
      goto label_end;
  }

  error = GTX_ERROR_NONE;

label_end:
  if (error)
    _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Deletes a palette in the current study

This function deletes a palette in the current study.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)

\param name Palette name to be deleted

\par Remarks:
Warning: System palettes cannot be deleted.
*****************************************************************************/
int GTXClientPaletteDelete(const char *name)
{
  GTXErrorCode error;
  const char *s[1];
  char answer[10];

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientPaletteDelete",1);
  GTX_TRACE(1, ("(%s)", name));
  if (!_gtx_is_supported("CPALDEL", 1))
    goto label_end;
  if( _gtx_start_packet("CPALDEL") ) goto label_end;

  s[0] = name;
  if (!_gtx_write_string(_gtx_server_socket, s, 10000))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}
