/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_run_server.c 26493 2016-09-16 10:32:24Z foucher $";

#ifdef winnt
#include <io.h> /* access on nt */
#include <time.h> /* time on nt */
#else
#include <unistd.h> /* access on solaris */
#include <sys/wait.h>           /* waitpid */
#endif
#ifdef linux
#include <time.h> /* time on linux */
#endif
#include <sys/types.h>
#include <sys/stat.h>

#include <stdlib.h> /* getenv on solaris */

static char gtx_server_variable[1024] = { '\0' };

/* Checks if a file exists */
int st_file_exists(char *filename)
{
  GTX_TRACE(2, ("Checking if file exists: %s", filename));
#if defined(winnt)
  if (!access(filename, 02))
#else
  if (!access(filename, R_OK))
#endif
    return(1);
  else
    return(0);
}

#ifdef winnt
/**************************************************************************
**
** FUNCTION:    st_hkey_get
**
** DESCRIPTION: Get a Registry Key or NULL if not found
**
** RETURNS:     Pointer on a static string or NULL
**
** IN_ARGS:     char *key:    Key name
** IN_ARGS:     char *subkey: Value Name
**
** REMARKS:     Memory has NOT to be freed
**
***************************************************************************/
char *st_hkey_get(char *key,
                  char *subkey)
{
  static char result[1024];
  HKEY hkey;
  LONG size = 1024;
  DWORD type;
  
  if (RegOpenKey(HKEY_LOCAL_MACHINE, key, &hkey) != ERROR_SUCCESS)
  {
    /* Opening it with no particular flag failed, try it forcing alternate 32/64 bit access */
    if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, key, 0,
#ifdef winnt64
                     KEY_READ | KEY_WOW64_32KEY, /* force 32 bit reg access with 64 bit exe */
#else
                     KEY_READ | KEY_WOW64_64KEY, /* force 64 bit reg access under 32 bit exe */
#endif
                     &hkey) != ERROR_SUCCESS)
      return NULL;
  }

  if (RegQueryValueEx(hkey, subkey, NULL,
                      &type, result, &size) != ERROR_SUCCESS)
  {
    RegCloseKey(hkey);
    return NULL;
  }

  RegCloseKey(hkey);
  return result;
}
#endif

/**************************************************************************
**
** FUNCTION:    search_in_path
**
** DESCRIPTION: Search the PATH for a given exe
**
** RETURNS:     0 if OK, 1 on error
**
** IN_ARGS:     char *exe_name: Executable name (with .exe if needed)
** IN_ARGS:     char slash:     slash character
**
** OUT_ARGS:    char *full_path: returned path
**
** REMARKS:     Warning: full_path is changed even if exe is not found
**
***************************************************************************/
int search_in_path(char *exe_name,
                   char slash,
                   char *full_path)
{
  char *char_ptr, *char_ptr2, *char_end;
  struct stat stat_buf;
  int len, found;
#ifdef winnt
  char sep = ';';
#else
  char sep = ':';
  uid_t my_uid = getuid();
  gid_t my_gid = getgid();
#endif

  GTX_TRACE(4, ("search_in_path(%s,%c,%p)",exe_name,slash,full_path));
  if ((char_ptr = getenv("PATH")) == NULL)
    return(1);
  GTX_TRACE(5, ("PATH=%s",char_ptr));

  /* search in it */
  found = 0;
  char_end = char_ptr + strlen(char_ptr);
  for (char_ptr2 = char_ptr;
       char_ptr2 < char_end && !found;
       char_ptr = char_ptr2)
  {
    while (char_ptr2 < char_end && *char_ptr2 != sep) char_ptr2++;
    len = (int)(char_ptr2-char_ptr);
    char_ptr2++;              /* point on the next character */
    if (len < 1) break;

    (void)strncpy(full_path,char_ptr,len);
    (void)sprintf(&full_path[len],"%c%s", slash, exe_name);

    GTX_TRACE(5, ("checking if exists %s", full_path));
    if (stat(full_path, &stat_buf) != 0) continue;

    /* File must be a regular file (not a directory...) */
    GTX_TRACE(5, ("checking if regular (mode=%o)", stat_buf.st_mode));
    if ((stat_buf.st_mode & S_IFMT) != S_IFREG) continue;

    /* File must be a executable */
#if !defined(winnt)
    found  = (stat_buf.st_mode | S_IXUSR) && stat_buf.st_uid == my_uid;
    found |= (stat_buf.st_mode | S_IXGRP) && stat_buf.st_gid == my_gid;
    found |= (stat_buf.st_mode | S_IXOTH);
#else
    found  = (stat_buf.st_mode | _S_IFREG) && (stat_buf.st_mode | _S_IEXEC);
#endif

    /* File found */
    if (found)
    {
      GTX_TRACE(2, ("Found GTXserver in PATH: %s", full_path));
      return(0);
    }
  }
  GTX_TRACE(2, ("...not found."));
  return(1);
}

#ifdef winnt
/****************************************************************************
**
** FUNCTION: st_get_exe_version
**
** PURPOSE:  Get executable version (using Windows API)
**
** IN_ARGS:  path: executable path
**
** REMARKS:  Just traces it...
**
*****************************************************************************/
static void st_get_exe_version(char *path)
{
  UINT BufLen;
  DWORD dwLen;
  void *lpData = NULL;
  VS_FIXEDFILEINFO *pFileInfo;

  dwLen = GetFileVersionInfoSize(path, lpData);
  if (dwLen != 0)
  {
    lpData = malloc(dwLen);
    if (lpData != NULL)
    {
      if (GetFileVersionInfo(path, 0, dwLen, lpData) &&
          VerQueryValue (lpData, "\\", &pFileInfo, &BufLen))
      {
        GTX_TRACE(2, ("Executable version=%d.%d.%d.%d",
                      HIWORD(pFileInfo->dwFileVersionMS),
                      LOWORD(pFileInfo->dwFileVersionMS),
                      HIWORD(pFileInfo->dwFileVersionLS),
                      LOWORD(pFileInfo->dwFileVersionLS)));
      }
      free(lpData);
    }
  }
}
#endif

/*!
****************************************************************************
\brief Locate GTXserver executable

This function tries to determine the location of GTXserver executable using:
- the value that has been set by \ref GTXClientSetGTXserverPath()
- the value of the GTX_SERVER environment variable
- the executable located in Isatis installation (which is determined by
the value of GTX_HOME environment variable under UNIX and Windows or a
registry key under Windows),
- a GTXserver executable in one component of the PATH environment variable.
If all of those tests are unsuccessfull, this function will return 1.

\return error code:
  \arg 0 if OK
  \arg 1 if the server was not found

\retval path return path of GTXserver executable (must be at least 1024 chars)

\par Remarks:
This function can be use to initialize an interface with the path of the
currently used GTXserver or for debugging purposes. You can also force this
function to find a given executable by using \ref GTXClientSetGTXserverPath()
***************************************************************************/
int GTXClientLocateGTXserver(char *path)
{
  GTXErrorCode error;
  char *env;
  char exe_name[20];
  char slash;
  int success;

  success = 0;
  error = GTX_ERROR_PARAMETER;
  GTX_TRACE_FUNC_START("GTXClientLocateGTXserver",1);
  GTX_TRACE(1, ("(%p)",path));
  CHECK_INIT(goto label_end;);

  path[0] = '\0';

  error = GTX_ERROR_PARAMETER;
  /* First use the value set by GTXClientSetGTXserverPath if any */
  GTX_TRACE(2, ("Checking value set by GTXClientSetGTXserverPath()"));
  if (strcmp(gtx_server_variable, ""))
  {
    env = path;
    (void)strcpy(path, gtx_server_variable);
    if (st_file_exists(path))
      success = 1;
  }

  /* First try to use GTX_SERVER environment variable (GTXSERVER deprecated) */
  if (!success)
  {
    GTX_TRACE(2, ("Checking environment variable GTX_SERVER"));
    env = getenv("GTXSERVER");
    if (env != NULL)
    {
      _gtx_error("Use GTX_SERVER environment variable instead of GTXSERVER");
      goto label_end;
    }
    else
      env = getenv("GTX_SERVER");
    if (env != NULL)
    {
      GTX_TRACE(2, ("GTX_SERVER=%s", env));
      (void)strcpy(path, env);
      if (st_file_exists(path))
        success = 1;
    }
  }

  /* Failed: search if GTX_HOME is defined */
  if (!success)
  {
    /* GTX_HOME defined -> add bin/GTXserver */
    GTX_TRACE(2, ("Checking environment variable GTX_HOME"));
    env = getenv("GTX_HOME");
#if defined(winnt)
    (void)strcpy(exe_name, "GTXserver.exe");
    slash = '\\';
    if (env == NULL || !strcmp(env, ""))
    {
      GTX_TRACE(2, ("Reading Reg Key HKLM\\Sft\\GV\\Isatis\\3.2\\GTX_HOME"));
      env = st_hkey_get("Software\\Geovariances\\Isatis\\3.2", "GTX_HOME");
    }
#else
    (void)strcpy(exe_name, "GTXserver");
    slash = '/';
#endif

    /* Search in bin/GTX_ARCH first */
    if (env != NULL)
    {
      GTX_TRACE(2, ("GTX_HOME=%s", env));
      (void)sprintf(path, "%s%cbin%c%s%c%s", env, slash, slash,
                    GTX_ARCH, slash, exe_name);
      if (st_file_exists(path))
        success = 1;
      else
      {
        /* else in bin/ARCH */
        (void)sprintf(path, "%s%cbin%c%s%c%s", env, slash, slash,
                      ARCH, slash, exe_name);
        if (st_file_exists(path))
          success = 1;
        else
        {
          /* elsewhere in bin (will run merge under UNIX,
             old version or wrapper under windows) */
          (void)sprintf(path, "%s%cbin%c%s", env, slash, slash, exe_name);
          if (st_file_exists(path))
            success = 1;
        }
      }
    }

    /* Exe not found in GTX_HOME/bin search the PATH */
    if (!success)
    {
      if (search_in_path(exe_name, slash, path))
      {
        _gtx_error("GTXserver executable has not been found");
        goto label_end;
      }
    }
  }

#ifdef winnt
  /* Get exe version */
  if (_gtx_trace_level)
    st_get_exe_version(path);
#endif

  error = GTX_ERROR_NONE;
label_end:
  if (!success)
    (void)strcpy(path, "");
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
****************************************************************************
\brief Set the GTXserver path to be used by \ref GTXClientRunGTXserver()

\param path: full path of the GTXserver executable, should be less than 1024
characters or empty string to remove any previous specification of it.
***************************************************************************/
void GTXClientSetGTXserverPath(const char *path)
{

  GTX_TRACE_FUNC_START("GTXClientSetGTXserverPath",1);
  GTX_TRACE(1, ("(%s)",path));
  (void)strcpy(gtx_server_variable, path);
  GTX_TRACE_FUNC_END("",NULL);
}

/*!
****************************************************************************
\brief Locate GTXserver executable and run it.

This function tries to locate the GTXserver executable using
\ref GTXClientLocateGTXserver() and run it.

\return error code:
  \arg 0 if OK
  \arg 1 if the server was not found or failed to RUN

\param port if you have a preferred port to RUN GTXserver on. If 0 a free
port is taken and its value is returned.
\retval port the new port is return here

\par Remarks:
Typical usage:
\code usigned short port = 0;
if (GTXClientRunGTXserver(&port)) ... \endcode
GTXserver is ran in the background listening on the given port. If the port
is 0, GTXserver is ran on the first free port of the machine. This avoids
trying to run a server on an already used port. The new port is returned
in the port variable and should be used to connect to the server.
***************************************************************************/
int GTXClientRunGTXserver(unsigned short *port)
{
  GTXErrorCode error;
  int err, server_alive, pid;
  char full_path[1024];
  int server_port = 0;
  int narg = 0;
  char *args[10], port_alpha[10], rport_alpha[10];
  unsigned short rport = 0;
  unsigned short control_port = 0;

  _GTX_SOCKET listen_socket = _GTX_SOCKET_ERROR;
  _GTX_SOCKET comm_socket = _GTX_SOCKET_ERROR;
  full_path[0] = '\0';

  error = GTX_ERROR_PARAMETER;
  GTX_TRACE_FUNC_START("GTXClientRunGTXserver",1);
  GTX_TRACE(1, ("(%p=%d)",port,(port==NULL)?0:*port));
  CHECK_INIT(goto label_end;);

  /* First try to use GTX_SERVER environment variable (GTXSERVER deprecated) */
  error = GTX_ERROR_NETWORK;
  if (_gtx_winsock_init()) goto label_end;

  error = GTX_ERROR_PARAMETER;
  if (GTXClientLocateGTXserver(full_path))
    goto label_end;

  /* Run the server */
  args[narg++] = "GTXserver";     /* argv[0] */
  if (_gtx_client_dbg_mode)
    args[narg++] = "-debug";
  if (*port == 0)
  {
    error = GTX_ERROR_NETWORK;
    GTX_TRACE(2, ("Port not given, create control connection"));
    if (_gtx_listen_port(&control_port, 1, &listen_socket))
      goto label_end;
    args[narg++] = "-control";
    (void)sprintf(port_alpha, "%d", (int)control_port);
    args[narg++] = port_alpha;
    /* special arg for old server: add random port */
    srand((unsigned int)time(NULL));
    rport = (unsigned short) (rand() % 1024) + 1024;
    args[narg++] = "-port";
    (void)sprintf(rport_alpha, "%d", (int)rport);
    args[narg++] = rport_alpha;
  }
  else
  {
    args[narg++] = "-port";
    (void)sprintf(port_alpha, "%u", (unsigned int)*port);
    args[narg++] = port_alpha;
  }
  args[narg] = NULL;

  pid = _gtx_exec_server((full_path[0] == '\0')? NULL: full_path, args);
  if (pid < 0)
    goto label_end;

  /* New port -> Wait for server to give the new address */
  if (*port == 0)
  {
    /* Wait for connection */
    GTX_TRACE(2, ("Waiting for server to connect control port"));
    err = _gtx_wait_connect(listen_socket, &comm_socket, 0, 3);
    /* Error during wait */
    if (err < 0)
      goto label_end;

    /* No connection -> return random port */
    if (err == 1)
    {
      /* No error -> rport is returned. */
      /* Client will try to connect to it... */
      GTX_TRACE(2, ("No connection from server during interval. Returning random new port=%d to caller", rport));
      *port = rport;
      error = GTX_ERROR_NONE;
      goto label_end;
    }
    else
    {
      GTX_TRACE(2, ("Reading server port from the socket"));
      if (!_gtx_read_int(comm_socket, &server_port))
        goto label_end;
      GTX_TRACE(2, ("Server in running on port %d", server_port));
    }
    *port = (unsigned short)server_port;
    _gtx_close_socket(&comm_socket);
    _gtx_close_socket(&listen_socket);

  }
  /* When port is given, let time to the server to init */
  else
  {
    GTX_TRACE(2, ("Sleeping 1s to let the server initialize..."));
#ifdef winnt
    Sleep(1000);
#else
    sleep(1);
#endif
    /* Check if server is still alive ! */
#ifdef winnt
    server_alive = 0;
    {
      HANDLE hProcess;
      hProcess = OpenProcess(PROCESS_QUERY_INFORMATION,FALSE,(DWORD)pid);
      server_alive = (hProcess != NULL);
      if (hProcess != NULL)
        CloseHandle(hProcess);
    }
#else
    {
      int status;
      if (waitpid((pid_t)pid, &status, WNOHANG))
        _gtx_perror("Cannot waitpid");
      server_alive = !WIFEXITED(status);
    }
    GTX_TRACE(2, ("GTXserver %s...", server_alive? "is already dead": "is still alive !!"));
#endif
  }
  error = GTX_ERROR_NONE;
label_end:
  _gtx_close_socket(&comm_socket);
  _gtx_close_socket(&listen_socket);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}
