/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_units.c 11482 2008-01-04 20:37:41Z foucher $";

/*!
******************************************************************************
\brief Set the current distance unit

Set the current distance unit.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param unit Distance unit to set

\par Remarks:
This default Unit will be used when creating new variables. It will also be
used to scale length variables in this unit when reading and from this unit
when writing.
Authorized units are:
 \arg micr for \e Micron
 \arg mm for \e Millimeter
 \arg cm for \e Centimeter
 \arg dm for \e Decimeter
 \arg m for \e Meter
 \arg dam for \e Decameter
 \arg hm for \e Hectometer
 \arg km for \e Kilometer
 \arg in for \e Inch
 \arg ft for \e Foot
 \arg yd for \e Yard
 \arg mile for \e Mile
 \arg nmil for \e Nautical \e Mile
 \arg dakm for \e 10 \e Kilometers
 \arg hkm for \e 100 \e Kilometers
 \arg kkm for \e 1000 \e Kilometers
*****************************************************************************/
int GTXClientSetUnit(const char *unit)
{
  GTXErrorCode error;
  char answer[10];
  const char *s[1];

  error = GTX_ERROR_PARAMETER;
  GTX_TRACE_FUNC_START("GTXClientSetUnit",1);
  GTX_TRACE(1, ("(%s)",unit));
  if (strlen(unit) >= 11)
  {
    _gtx_error("GTXClientSetUnit(): Unit \"%s\" cannot be that long", unit);
    goto label_end;
  }

  error = GTX_ERROR_NETWORK;
  if (_gtx_start_packet("CDISTUSET")) goto label_end;

  s[0] = unit;
  if (!_gtx_write_string(_gtx_server_socket, s, 11)) goto label_end;
  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

static char *UNIT_SYMBOL[] = {
  "micr","mm","cm","dm","m","dam","hm","km",
  "in","ft","yd","mile","nmil","dakm","hkm","kkm" };
static double UNIT_FACTOR[] = {
  .0001,.1, 1., 10., 100., 1000., 10000., 100000.,
  2.54, 30.48, 91.44, 160900., 185200.,1000000.,10000000.,100000000.};
static int nunits = sizeof(UNIT_FACTOR)/sizeof(double);

/*!
******************************************************************************
\brief Get the unit factor corresponding to a given unit symbol

This function returns the unit factor corresponding to a given unit symbol or
0. if not a known length symbol.
\param symbol unit symbol to be searched for
\return unit factor
*****************************************************************************/
double GTXClientUnitFactorFromSymbol(const char *symbol)
{
  int i;
  GTX_TRACE_FUNC_START("GTXClientUnitFactorFromSymbol",1);
  GTX_TRACE(1, ("(%s)",symbol));
  for (i = 0; i < nunits; i++)
    if (!strcmp(UNIT_SYMBOL[i], symbol))
    {
      GTX_TRACE_FUNC_END("%g",UNIT_FACTOR[i]);
      return(UNIT_FACTOR[i]);
    }

  /* item is not found */
  GTX_TRACE_FUNC_END("0.",NULL);
  return(0.);
}

/*!
******************************************************************************
\brief Get the unit factor corresponding to a given unit symbol

This function returns the unit symbol corresponding to a given unit symbol or
an empty string if not a known length factor.
\param factor unit factor to be searched for
\return unit symbol
\par Remarks: the returned string is a static string. It doesn't have to be
fred.
*****************************************************************************/
const char *GTXClientUnitSymbolFromFactor(double factor)
{
  int i;
  GTX_TRACE_FUNC_START("GTXClientUnitSymbolFromFactor",1);
  GTX_TRACE(1, ("(%g)",factor));
  for (i = 0; i < nunits; i++)
    if (UNIT_FACTOR[i] == factor)
    {
      GTX_TRACE_FUNC_END("%s",UNIT_SYMBOL[i]);
      return(UNIT_SYMBOL[i]);
    }

  /* item is not found */
  GTX_TRACE_FUNC_END("",NULL);
  return("");
}

/*!
******************************************************************************
\brief Change the unit factor and symbol of a variable

This functions allows to change the unit attached to a variable. If the factor
unit is part of the known units list, it changes the variable to a length one.
Then we can specify a different symbol to display. If the factor, is not known
as a length one, the variable will be float (e.g, not length) and the symbol
will not be used.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param factor new unit factor
\param symbol new unit symbol (used for length or NULL)

\sa GTXClientSetUnit for the list of know units.
*****************************************************************************/
int GTXClientSetVariableUnit(const char *factor,
                             const char *symbol)
{
  GTXErrorCode error;
  int nargs;
  const char *s[1];
  char answer[10];
  char unit_str[11];

  error = GTX_ERROR_PARAMETER;
  GTX_TRACE_FUNC_START("GTXClientSetVariableUnit",1);
  GTX_TRACE(1, ("(%s,%s)",factor,
              (symbol==NULL)? "(nil)": symbol));
  if (factor == NULL)
  {
    _gtx_error
      ("GTXClientSetVariableUnit(): factor cannot be NULL.");
    goto label_end;
  }
  error = GTX_ERROR_NETWORK;
  if (_gtx_start_packet("CVARUNIT")) goto label_end;

  nargs = 1;
  if (symbol != NULL) nargs++;
  if (!_gtx_write_int(_gtx_server_socket, &nargs)) goto label_end;

  s[0] = unit_str;
  unit_str[10] = '\0';

  (void)strncpy(unit_str, factor, 10);
  if (!_gtx_write_string(_gtx_server_socket, s, 11)) goto label_end;

  if (symbol != NULL)
  {
    (void)strncpy(unit_str, symbol, 10);
    if (!_gtx_write_string(_gtx_server_socket, s, 11)) goto label_end;
  }
  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Change the way length units are used

This functions allows to change the way units are used at the creation of new
length variables and reading of existing ones.
When mode is 0, all length variables are read/written using the current unit
(GTXCLientSetUnit).
When mode is 1, each length variable is read/written using its own unit. This
also affect File Information and File Creation that are also based on
coordinates variables which are the main length variables.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param mode new unit mode

\par Remarks
Default mode if 0 which means that you read/write all information in the
current unit set with GTXClientSetUnit.

Warning: setting mode to 1 and back to 0 resets creation units
(set with \sa GTXClientSetCreationUnits) to the current unit.
*****************************************************************************/
int GTXClientSetUnitMode(int mode)
{
  GTXErrorCode error;
  char answer[10];

  GTX_TRACE_FUNC_START("GTXClientSetUnitMode",1);
  GTX_TRACE(1, ("(%d)",mode));
  error = GTX_ERROR_NETWORK;
  if (_gtx_start_packet("CUNITMODE")) goto label_end;
  if (!_gtx_write_int(_gtx_server_socket, &mode)) goto label_end;
  if (_gtx_send_and_wait(answer)) goto label_end;
  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;

  error = GTX_ERROR_NONE;
label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Change the default units for length variables creation

This functions allows to change the default units used for coordinate
variables creation when unit_mode (\ref GTXClientSetUnitMode) is set to 1.
After that, all coordinate variables creation take those units into
account. This has no effect when unit_mode is 0.

\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param x_unit new X coordinate unit
\param y_unit new Y coordinate unit
\param z_unit new Z coordinate unit

\par Remarks
Warning: setting mode to 1 and back to 0 resets creation units.
*****************************************************************************/
int GTXClientSetCreationUnits(const char *x_unit,
                              const char *y_unit,
                              const char *z_unit)
{
  int i;
  const char *s[3];
  char answer[10];
  GTXErrorCode error;

  error = GTX_ERROR_NONE;
  GTX_TRACE_FUNC_START("GTXClientSetCreationUnits",1);
  GTX_TRACE(1, ("(%s,%s,%s)",x_unit,y_unit,z_unit));

  if (!SERVER_4_1)
    goto label_end;

  error = GTX_ERROR_NETWORK;
  if (_gtx_start_packet("CUNITCR")) goto label_end;
  s[0] = x_unit;
  s[1] = y_unit;
  s[2] = z_unit;
  for (i = 0; i < 3; i++)
    if (!_gtx_write_string(_gtx_server_socket, &s[i], 11)) goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;
  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;

  error = GTX_ERROR_NONE;
label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}
