/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_vendor_data.c 22738 2013-08-13 10:21:58Z foucher $";

static GTXVendorDataAttrRec *st_vendor_data_free_attr(GTXVendorDataAttrRec *attr);
static GTXVendorDataAttrRec *st_start_attribute(GTXVendorData vendor_data,
                                                const char *name,
                                                _GTX_VENDOR_DATA_TYPE type);
static GTXVendorDataAttrRec *st_end_attribute(GTXVendorData vendor_data,
                                              GTXVendorDataAttrRec *new);

/*!
******************************************************************************
\brief Create a new GTXVendorData
 
This function creates a new GTXVendorData.
\return a new GTXVendorData or NULL on allocation error
\param identificator Vendor data identificator
\param version Vendor data version
 
\par Remarks:
The memory used by each GTXVendorData must be freed using
\ref GTXClientVendorDataFree.
*****************************************************************************/
GTXVendorData GTXClientVendorDataNew(const char *identificator, int version)
{
  GTXVendorData vendor_data = NULL;

  GTX_TRACE_FUNC_START("GTXClientVendorDataNew",1);
  GTX_TRACE(1, ("(%s,%d)",identificator,version));
  vendor_data = (GTXVendorData)calloc(1, sizeof(GTXVendorDataRec));
  if (vendor_data == NULL) goto label_end;

  (void)strcpy(vendor_data->identificator, identificator);
  vendor_data->version = version;
  vendor_data->attributes = NULL;
label_end:
  GTX_TRACE_FUNC_END("%p",vendor_data);
  return vendor_data;
}

/*!
******************************************************************************
\brief Add an integer attribute to a Vendor Data
 
This function adds an integer attribute to a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor data to be modified
\param name Attribute name
\param value Value of the attribute
*****************************************************************************/
int GTXClientVendorDataAddAttributeInt(GTXVendorData vendor_data,
                                       const char *name,
                                       int value)
{
  GTXErrorCode error;
  GTXVendorDataAttrRec *new;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataAddAttributeInt",1);
  GTX_TRACE(1, ("(%p,%s,%d)", vendor_data,name,value));
  new = st_start_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_INT);
  if( new == NULL ) goto label_end;

  new->values.val_int = value;
  new = st_end_attribute(vendor_data, new);

  error = GTX_ERROR_NONE;

label_end:
  if( error && new ) new = st_vendor_data_free_attr(new);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Add a double attribute to a Vendor Data
 
This function adds an double attribute to a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor data to be modified
\param name Attribute name
\param value Value of the attribute
*****************************************************************************/
int GTXClientVendorDataAddAttributeDouble(GTXVendorData vendor_data,
                                          const char *name, double value)
{
  GTXErrorCode error;
  GTXVendorDataAttrRec *new;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataAddAttributeDouble",1);
  GTX_TRACE(1, ("(%p,%s,%g)", vendor_data,name,value));
  new = st_start_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_DOUBLE);
  if( new == NULL ) goto label_end;

  new->values.val_double = value;
  new = st_end_attribute(vendor_data, new);

  error = GTX_ERROR_NONE;

label_end:
  if( error && new ) new = st_vendor_data_free_attr(new);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Add a string attribute to a Vendor Data
 
This function adds a string attribute to a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor data to be modified
\param name Attribute name
\param value Value of the attribute
*****************************************************************************/
int GTXClientVendorDataAddAttributeString(GTXVendorData vendor_data,
                                          const char *name, const char *value)
{
  GTXErrorCode error;
  GTXVendorDataAttrRec *new;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataAddAttributeString",1);
  GTX_TRACE(1, ("(%p,%s,%s)", vendor_data,name,value));
  new = st_start_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_STRING);
  if( new == NULL ) goto label_end;

  new->values.val_string = strdup(value);
  if( new->values.val_string == NULL )
  {
    _gtx_error("GTXClientVendorDataAddAttributeString(): "
               "Memory allocation error for attribute %s", name);
    goto label_end;
  }
  new = st_end_attribute(vendor_data, new);

  error = GTX_ERROR_NONE;

label_end:
  if (error && new) new = st_vendor_data_free_attr(new);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Add an integer array attribute to a Vendor Data
 
This function adds an integer array attribute to a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor data to be modified
\param name Attribute name
\param nvalues Number of values in the array
\param values Array of integer
*****************************************************************************/
int GTXClientVendorDataAddAttributeIntArray(GTXVendorData vendor_data,
                                            const char *name, int nvalues,
                                            const int *values)
{
  int i;
  GTXErrorCode error;
  GTXVendorDataAttrRec *new;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataAddAttributeIntArray",1);
  GTX_TRACE(1, ("(%p,%s,%d,%p)", vendor_data,name,nvalues,values));
  new = st_start_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_INT_ARRAY);
  if( new == NULL ) goto label_end;

  new->nvalues = nvalues;
  new->values.int_array = (int*)calloc(nvalues, sizeof(int));
  if( new->values.int_array == NULL )
  {
    _gtx_error("GTXClientVendorDataAddAttributeIntArray() "
               "Memory allocation error for attribute %s", name);
    goto label_end;
  }

  for(i=0; i<nvalues; i++) new->values.int_array[i] = values[i];
  new = st_end_attribute(vendor_data, new);

  error = GTX_ERROR_NONE;

label_end:
  if (error && new) new = st_vendor_data_free_attr(new);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Add a double array attribute to a Vendor Data
 
This function adds a double array attribute to a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor data to be modified
\param name Attribute name
\param nvalues Number of values in the array
\param values Array of double
*****************************************************************************/
int GTXClientVendorDataAddAttributeDoubleArray(GTXVendorData vendor_data,
                                               const char *name, int nvalues,
                                               const double *values)
{
  int i;
  GTXErrorCode error;
  GTXVendorDataAttrRec *new;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataAddAttributeDoubleArray",1);
  GTX_TRACE(1, ("(%p,%s,%d,%p)", vendor_data,name,nvalues,values));
  new = st_start_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_DOUBLE_ARRAY);
  if( new == NULL ) goto label_end;

  new->nvalues = nvalues;
  new->values.double_array = (double*)calloc(nvalues, sizeof(double));
  if( new->values.double_array == NULL )
  {
    _gtx_error("GTXClientVendorDataAddAttributeDoubleArray() "
               "Memory allocation error for attribute %s", name);
    goto label_end;
  }
  for(i=0; i<nvalues; i++) new->values.double_array[i] = values[i];
  new = st_end_attribute(vendor_data, new);

  error = GTX_ERROR_NONE;

label_end:
  if (error && new) new = st_vendor_data_free_attr(new);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Add a string array attribute to a Vendor Data
 
This function adds a double array attribute to a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor data to be modified
\param name Attribute name
\param nvalues Number of values in the array
\param values Array of strings
*****************************************************************************/
int GTXClientVendorDataAddAttributeStringArray(GTXVendorData vendor_data,
                                               const char *name, int nvalues,
                                               const char * const *values)
{
  int i;
  GTXErrorCode error;
  GTXVendorDataAttrRec *new;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataAddAttributeStringArray",1);
  GTX_TRACE(1, ("(%p,%s,%d,%p)", vendor_data,name,nvalues,values));
  new = st_start_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_STRING_ARRAY);
  if( new == NULL ) goto label_end;

  new->nvalues = nvalues;
  new->values.string_array = (char**)calloc(nvalues, sizeof(char*));
  if( new->values.string_array == NULL )
  {
    _gtx_error("GTXClientVendorDataAddAttributeStringArray() "
               "Memory allocation error for attribute %s", name);
    goto label_end;
  }

  for(i=0; i<nvalues; i++) new->values.string_array[i] = NULL;

  for(i=0; i<nvalues; i++)
  {
    new->values.string_array[i] = strdup(values[i]);
    if( new->values.string_array[i] == NULL )
    {
      _gtx_error("GTXClientVendorDataAddAttributeStringArray() "
                 "Memory allocation error for attribute %s", name);
      goto label_end;
    }
  }
  new = st_end_attribute(vendor_data,new);

  error = GTX_ERROR_NONE;

label_end:
  if (error && new) new = st_vendor_data_free_attr(new);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Write a previously created Vendor Data
 
This function writes a previously created Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param level Where to write the Vendor Data
  \arg 0 for study
  \arg 1 for directory
  \arg 2 for file
  \arg 3 for variable
\param vendor_data The Vendor Data to be written
*****************************************************************************/
int GTXClientVendorDataWrite(int level, GTXVendorData vendor_data)
{
  GTXVendorDataAttrRec *attr;
  GTXErrorCode error;
  int i, len, count;
  char answer[10];
  const char *s[1];

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataWrite",1);
  GTX_TRACE(1, ("(%d,%p)",level,vendor_data));
  if( _gtx_start_packet("CVDATWRIT") ) goto label_end;

  s[0] = vendor_data->identificator;
  if( !_gtx_write_string(_gtx_server_socket, s, 50) ||
      !_gtx_write_int(_gtx_server_socket, &vendor_data->version) ||
      !_gtx_write_int(_gtx_server_socket, &level) )
    goto label_end;

  /* number of attributes */
  count = 0;
  attr = vendor_data->attributes;
  while( attr != NULL )
  {
    count++;
    attr = attr->next;
  }
  if( !_gtx_write_int(_gtx_server_socket, &count) ) goto label_end;

  attr = vendor_data->attributes;
  while( attr != NULL )
  {
    s[0] = attr->name;
    if( !_gtx_write_string(_gtx_server_socket, s, 50) ||
        !_gtx_write_int(_gtx_server_socket, (int *)&attr->type) ) goto label_end;
    switch( attr->type )
    {
      case _GTX_VENDOR_DATA_TYPE_NOTHING:
        break;

      case _GTX_VENDOR_DATA_TYPE_INT:
        if( !_gtx_write_int(_gtx_server_socket, &attr->values.val_int) )
          goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_DOUBLE:
        if( !_gtx_write_double(_gtx_server_socket, &attr->values.val_double) )
          goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_STRING:
        len = (int)strlen(attr->values.val_string) + 1;
        if( !_gtx_write_int(_gtx_server_socket, &len) ) goto label_end;
        if( !_gtx_write_string(_gtx_server_socket, (const char **)&attr->values.val_string, len) )
          goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_INT_ARRAY:
        if( !_gtx_write_int(_gtx_server_socket, &attr->nvalues) ) goto label_end;
        for(i=0; i<attr->nvalues; i++)
          if( !_gtx_write_int(_gtx_server_socket, &attr->values.int_array[i]) )
            goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_DOUBLE_ARRAY:
        if( !_gtx_write_int(_gtx_server_socket, &attr->nvalues) ) goto label_end;
        for(i=0; i<attr->nvalues; i++)
          if( !_gtx_write_double(_gtx_server_socket, &attr->values.double_array[i]) )
            goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_STRING_ARRAY:
        if( !_gtx_write_int(_gtx_server_socket, &attr->nvalues) ) goto label_end;
        for(i=0; i<attr->nvalues; i++)
        {
          len = (int)strlen(attr->values.string_array[i]) + 1;
          if( !_gtx_write_int(_gtx_server_socket, &len) ) goto label_end;
          if( !_gtx_write_string(_gtx_server_socket, (const char **)&attr->values.string_array[i], len) )
            goto label_end;
        }
        break;

      default:
        /* programmer error */
        _gtx_error("GTXClientVendorDataWrite() "
                   "Wrong type for attribute %d", attr->type);
        exit(1);
    } /* end of switch type */

    attr = attr->next;
  } /* loop on attributes */

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SDONE") ) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get version string from a Vendor Data
 
This function reads the version string from a stored Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param identificator Identificator of the Vendor Data
\param level Where to read the Vendor Data
  \arg 0 for study
  \arg 1 for directory
  \arg 2 for file
  \arg 3 for variable
\retval version Returned Vendor Data version
*****************************************************************************/
int GTXClientVendorDataGetVersion(const char *identificator, int level,
                                  int *version)
{
  GTXErrorCode error;
  char answer[10];

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataGetVersion",1);
  GTX_TRACE(1, ("(%s,%d,%p)", identificator,level,version));
  if( _gtx_start_packet("CVDATVERS") ) goto label_end;

  if( !_gtx_write_string(_gtx_server_socket, &identificator, 50) ||
      !_gtx_write_int(_gtx_server_socket, &level) )
    goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SVDATVERS") ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if( !_gtx_read_int(_gtx_server_socket, version)) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Checks if a Vendor Data exists
 
This function verifies the existence of a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param identificator Identificator of the Vendor Data
\param level Where to read the Vendor Data
  \arg 0 for study
  \arg 1 for directory
  \arg 2 for file
  \arg 3 for variable
\retval exists 0 if it does not exists, 1 if it exists
*****************************************************************************/
int GTXClientVendorDataExists(const char *identificator, int level,
                              int *exists)
{
  GTXErrorCode error;
  char answer[10];

  GTX_TRACE_FUNC_START("GTXClientVendorDataExists",1);
  GTX_TRACE(1, ("(%s,%d,%p)", identificator,level,exists));
  error = GTX_ERROR_SERVER;
  if (!_gtx_is_supported("CVDATEXIS",1))
    goto label_end;

  error = GTX_ERROR_NETWORK;
  if (_gtx_start_packet("CVDATEXIS")) goto label_end;

  if (!_gtx_write_string(_gtx_server_socket, &identificator, 50) ||
      !_gtx_write_int(_gtx_server_socket, &level))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SVDATEXIS")) goto label_end;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_read_int(_gtx_server_socket, exists)) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Read a Vendor Data from Isatis file system
 
This function reads a Vendor Data from the Isatis file system
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param identificator Identificator of the Vendor Data
\param level Where to read the Vendor Data
  \arg 0 for study
  \arg 1 for directory
  \arg 2 for file
  \arg 3 for variable
\retval vendor_data Returned Vendor Data 
\retval version Returned Vendor Data version
\par Remarks:
You can pass NULL as version if you do not want to retrieve version number.
After usage, You should free the Vendor Data using \ref GTXClientVendorDataFree.
*****************************************************************************/
int GTXClientVendorDataRead(const char *identificator, int level,
                            GTXVendorData *vendor_data, int *version)
{
  GTXVendorData local_data;
  GTXVendorDataAttrRec *attr;
  char local_name[50];
  GTXErrorCode error;
  int i, iattr, len, count;
  char answer[10];
  char *s[1];

  attr = NULL; 
  GTX_TRACE_FUNC_START("GTXClientVendorDataRead",1);
  GTX_TRACE(1, ("(%s,%d,%p,%p)", identificator,level,vendor_data,version));
  error = GTX_ERROR_MEMORY;
  local_data = GTXClientVendorDataNew(identificator, 0);
  if( local_data == NULL ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if( _gtx_start_packet("CVDATREAD") ) goto label_end;

  if( !_gtx_write_string(_gtx_server_socket, &identificator, 50) ||
      !_gtx_write_int(_gtx_server_socket, &level) ) goto label_end;
  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SVDAT") ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if( !_gtx_read_int(_gtx_server_socket, &local_data->version) ) goto label_end;
  if( version != NULL ) *version = local_data->version;

  if( !_gtx_read_int(_gtx_server_socket, &count) ) goto label_end;

  for(iattr=0; iattr<count; iattr++)
  {
    s[0] = local_name;
    if( !_gtx_read_string(_gtx_server_socket, s, 50) ) goto label_end;

    attr = st_start_attribute(local_data, local_name,
                              _GTX_VENDOR_DATA_TYPE_NOTHING);
    if( attr == NULL ) goto label_end;

    if( !_gtx_read_int(_gtx_server_socket, (int*)&(attr->type)) ) goto label_end;

    switch(attr->type)
    {
      case _GTX_VENDOR_DATA_TYPE_NOTHING:
        break;

      case _GTX_VENDOR_DATA_TYPE_INT:
        if( !_gtx_read_int(_gtx_server_socket, &(attr->values.val_int)) ) goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_DOUBLE:
        if( !_gtx_read_double(_gtx_server_socket, &(attr->values.val_double)) )
          goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_STRING:
        attr->values.val_string = NULL;
        if( !_gtx_read_int(_gtx_server_socket, &len) ) goto label_end;

        attr->values.val_string = (char*)calloc(len, sizeof(char));
        if( attr->values.val_string == NULL )
        {
          _gtx_error("GTXClientVendorDataRead(): "
                     "Memory allocation error for attribute %s", attr->name);
          error = GTX_ERROR_MEMORY;
          goto label_end;
        }

        if( !_gtx_read_string(_gtx_server_socket, &(attr->values.val_string), len) ) goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_INT_ARRAY:
        attr->values.int_array = NULL;
        if( !_gtx_read_int(_gtx_server_socket, &(attr->nvalues)) ) goto label_end;

        attr->values.int_array = (int*)calloc(attr->nvalues, sizeof(int));
        if( attr->values.int_array == NULL )
        {
          _gtx_error("GTXClientVendorDataRead(): "
                     "Memory allocation error for attribute %s", attr->name);
          error = GTX_ERROR_MEMORY;
          goto label_end;
        }

        for(i=0; i<attr->nvalues; i++)
          if( !_gtx_read_int(_gtx_server_socket, &(attr->values.int_array[i])) )
            goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_DOUBLE_ARRAY:
        attr->values.double_array = NULL;
        if( !_gtx_read_int(_gtx_server_socket, &(attr->nvalues)) ) goto label_end;

        attr->values.double_array = (double*)calloc(attr->nvalues, sizeof(double));
        if( attr->values.double_array == NULL )
        {
          _gtx_error("GTXClientVendorDataRead(): "
                     "Memory allocation error for attribute %s", attr->name);
          error = GTX_ERROR_MEMORY;
          goto label_end;
        }

        for(i=0; i<attr->nvalues; i++)
          if( !_gtx_read_double(_gtx_server_socket, &(attr->values.double_array[i])) )
            goto label_end;
        break;

      case _GTX_VENDOR_DATA_TYPE_STRING_ARRAY:
        attr->values.string_array = NULL;
        if( !_gtx_read_int(_gtx_server_socket, &(attr->nvalues)) ) goto label_end;

        attr->values.string_array = (char**)calloc(attr->nvalues, sizeof(char*));
        if( attr->values.string_array == NULL )
        {
          _gtx_error("GTXClientVendorDataRead(): "
                     "Memory allocation error for attribute %s", attr->name);
          error = GTX_ERROR_MEMORY;
          goto label_end;
        }

        for(i=0; i<attr->nvalues; i++) attr->values.string_array[i] = NULL;
        for(i=0; i<attr->nvalues; i++)
        {
          if( !_gtx_read_int(_gtx_server_socket, &len) ) goto label_end;

          attr->values.string_array[i] = (char*)calloc(len, sizeof(char));
          if( attr->values.string_array[i] == NULL )
          {
            _gtx_error("GTXClientVendorDataRead(): "
                       "Memory allocation error for attribute %s", attr->name);
            error = GTX_ERROR_MEMORY;
            goto label_end;
          }

          if( !_gtx_read_string(_gtx_server_socket, &(attr->values.string_array[i]), len) )
            goto label_end;
        }
        break;

      default:
        _gtx_error("GTXClientVendorDataRead(): "
                   "Wrong Vendor Data Type (%d)", attr->type);

    } /* end of switch type */
    attr = st_end_attribute(local_data,attr);
  } /* end of loop on attributes */

  error = GTX_ERROR_NONE;

label_end:
  if( error ) _gtx_cleanup_socket();
  attr = st_vendor_data_free_attr(attr);
  if( error ) local_data = GTXClientVendorDataFree(local_data);
  *vendor_data = local_data;
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get value from an integer attribute of a Vendor Data
 
This function reads the value from an integer attribute of a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor Data
\param name Name of the attribute to read
\retval value Returned integer value
*****************************************************************************/
int GTXClientVendorDataGetAttributeInt(GTXVendorData vendor_data,
                                       const char *name, int *value)
{
  GTXErrorCode error;
  GTXVendorDataAttrRec *attr;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataGetAttributeInt",1);
  GTX_TRACE(1, ("(%p,%s,%p)", vendor_data,name,value));
  attr = _gtx_vendor_data_search_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_INT);
  if (attr == NULL) goto label_end;
  *value = attr->values.val_int;
  error = GTX_ERROR_NONE;
label_end:
  GTX_TRACE_FUNC_END("%d", error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get value from a double attribute of a Vendor Data
 
This function reads the value from a double attribute of a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor Data
\param name Name of the attribute to read
\retval value Returned double value
*****************************************************************************/
int GTXClientVendorDataGetAttributeDouble(GTXVendorData vendor_data,
                                          const char *name, double *value)
{
  GTXErrorCode error;
  GTXVendorDataAttrRec *attr;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataGetAttributeDouble",1);
  GTX_TRACE(1, ("(%p,%s,%p)", vendor_data,name,value));
  attr = _gtx_vendor_data_search_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_DOUBLE);
  if (attr == NULL) goto label_end;

  *value = attr->values.val_double;
  error = GTX_ERROR_NONE;
label_end:
  GTX_TRACE_FUNC_END("%d", error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get value from a string attribute of a Vendor Data
 
This function reads the value from a string attribute of a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor Data
\param name Name of the attribute to read
\retval value Returned string value
\par Remarks:
The returned string must be freed using:
\code
value = GTXClientFreePointer(value);
\endcode
*****************************************************************************/
int GTXClientVendorDataGetAttributeString(GTXVendorData vendor_data,
                                          const char *name, char **value)
{
  GTXErrorCode error;
  GTXVendorDataAttrRec *attr;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataGetAttributeString",1);
  GTX_TRACE(1, ("(%p,%s,%p)", vendor_data,name,value));
  attr = _gtx_vendor_data_search_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_STRING);
  if (attr == NULL) goto label_end;

  *value = strdup(attr->values.val_string);
  if( *value == NULL )
  {
    _gtx_error("GTXClientVendorDataGetAttributeString(): "
               "Memory allocation error for attribute %s", name);
    goto label_end;
  }
  error = GTX_ERROR_NONE;
label_end:
  GTX_TRACE_FUNC_END("%d", error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get value from an integer array attribute of a Vendor Data
 
This function reads the value from an integer array attribute of a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor Data
\param name        Name of the attribute to read
\param nvalues     Returned number of values in the array
\param values      Returned array of values

\retval value Returned integer array value
\par Remarks:
The returned integer array must be freed using:
\code
values = GTXClientFreePointer(values);
\endcode
*****************************************************************************/
int GTXClientVendorDataGetAttributeIntArray(GTXVendorData vendor_data,
                                            const char *name,
                                            int *nvalues,
                                            int **values)
{
  GTXErrorCode error;
  int i, *array;
  GTXVendorDataAttrRec *attr;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataGetAttributeIntArray",1);
  GTX_TRACE(1, ("(%p,%s,%p,%p)", vendor_data,name,nvalues,values));

  *values = NULL;
  *nvalues = 0;

  array = NULL;
  attr = _gtx_vendor_data_search_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_INT_ARRAY);
  if (attr == NULL) goto label_end;

  array = (int*)calloc(attr->nvalues, sizeof(int));
  if( array == NULL )
  {
    _gtx_error("GTXClientVendorDataGetAttributeIntArray() "
               "Memory allocation error for attribute %s", name);
    goto label_end;
  }
  for(i=0; i<attr->nvalues; i++) array[i] = attr->values.int_array[i];

  *values = array;
  *nvalues = attr->nvalues;
  error = GTX_ERROR_NONE;
label_end:
  GTX_TRACE_FUNC_END("%d", error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get value from a double array attribute of a Vendor Data
 
This function reads the value from a double array attribute of a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor Data
\param name        Name of the attribute to read
\param nvalues     Returned number of values in the array
\param values      Returned array of values

\retval value Returned double array value
\par Remarks:
The returned double array must be freed using:
\code
values = GTXClientFreePointer(values);
\endcode
*****************************************************************************/
int GTXClientVendorDataGetAttributeDoubleArray(GTXVendorData vendor_data,
                                               const char *name,
                                               int *nvalues,
                                               double **values)
{
  int i;
  GTXErrorCode error;
  double *array;
  GTXVendorDataAttrRec *attr;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataGetAttributeDoubleArray",1);
  GTX_TRACE(1, ("(%p,%s,%p,%p)", vendor_data,name,nvalues,values));

  *values = NULL;
  *nvalues = 0;

  attr = _gtx_vendor_data_search_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_DOUBLE_ARRAY);
  if (attr == NULL) goto label_end;

  array = (double*)calloc(attr->nvalues, sizeof(double));
  if( array == NULL )
  {
    _gtx_error("GTXClientVendorDataGetAttributeDoubleArray() "
               "Memory allocation error for attribute %s", name);
    goto label_end;
  }
  for (i=0; i<attr->nvalues; i++) array[i] = attr->values.double_array[i];

  *values = array;
  *nvalues = attr->nvalues;
  error = GTX_ERROR_NONE;
label_end:
  GTX_TRACE_FUNC_END("%d", error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Get value from a string array attribute of a Vendor Data
 
This function reads the value from a string array attribute of a Vendor Data
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param vendor_data Vendor Data
\param name        Name of the attribute to read
\param nvalues     Returned number of strings in the array
\param values      Returned array of strings

\retval value Returned string array value
\par Remarks:
The returned string array must be freed using:
\code
values = GTXClientFreeStringArray(nvalues, values);
\endcode
*****************************************************************************/
int GTXClientVendorDataGetAttributeStringArray(GTXVendorData vendor_data,
                                               const char *name,
                                               int *nvalues,
                                               char ***values)
{
  int i;
  GTXErrorCode error;
  char **array;
  GTXVendorDataAttrRec *attr;

  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientVendorDataGetAttributeStringArray",1);
  GTX_TRACE(1, ("(%p,%s,%p,%p)", vendor_data,name,nvalues,values));

  attr = NULL;
  array = NULL;
  *values = NULL;
  *nvalues = 0;

  attr = _gtx_vendor_data_search_attribute(vendor_data, name, _GTX_VENDOR_DATA_TYPE_STRING_ARRAY);
  if (attr == NULL) goto label_end;

  array = (char**)calloc(attr->nvalues, sizeof(char*));
  if( array == NULL )
  {
    _gtx_error("GTXClientVendorDataGetAttributeStringArray(): "
               "Memory allocation error for attribute %s", name);
    goto label_end;
  }
  for(i=0; i<attr->nvalues; i++) array[i] = NULL;
  for(i=0; i<attr->nvalues; i++)
  {
    array[i] = strdup(attr->values.string_array[i]);
    if( array[i] == NULL )
    {
      _gtx_error("GTXClientVendorDataGetAttributeStringArray() "
                 "Memory allocation error for attribute %s", name);
      goto label_end;
    }
  }

  *values = array;
  *nvalues = attr->nvalues;
  error = GTX_ERROR_NONE;
label_end:
  if (error && attr != NULL)
    array = GTXClientFreeStringArray(attr->nvalues, array);
  GTX_TRACE_FUNC_END("%d", error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Free a given Vendor Data
 
This function frees a Vendor Data 
\return freed Vendor Data (NULL)
\param vendor_data The Vendor Data to be freed
*****************************************************************************/
GTXVendorData GTXClientVendorDataFree(GTXVendorData vendor_data)
{
  GTXVendorDataAttrRec *attr, *attr_next;

  GTX_TRACE_FUNC_START("GTXClientVendorDataFree",1);
  GTX_TRACE(1, ("(%p)", vendor_data));
  if (vendor_data == NULL) goto label_end;

  attr = vendor_data->attributes;
  while(attr != NULL)
  {
    attr_next = attr->next;
    (void)st_vendor_data_free_attr(attr);
    attr = attr_next;
  }
  free(vendor_data);
label_end:
  GTX_TRACE_FUNC_END("(nil)",NULL);
  return NULL;
}

/****************************************************************************
**
** FUNCTION:    st_vendor_data_free_attr
**
** DESCRIPTION: frees a given vendor data attribute
**
** RETURNS:     NULL
**
** IN_ARGS:     attr: vendor data attribute
**
** REMARKS:     This function only frees the attribute but not the other in
** REMARKS:     the linked list
**
*****************************************************************************/
static GTXVendorDataAttrRec *
st_vendor_data_free_attr(GTXVendorDataAttrRec *attr)
{
  if (attr == NULL)
    return(NULL);

  switch (attr->type)
  {
    case _GTX_VENDOR_DATA_TYPE_NOTHING:
    case _GTX_VENDOR_DATA_TYPE_INT:
    case _GTX_VENDOR_DATA_TYPE_DOUBLE:
      break;
    case _GTX_VENDOR_DATA_TYPE_STRING:
      attr->values.val_string = GTXClientFreePointer(attr->values.val_string);
      break;
    case _GTX_VENDOR_DATA_TYPE_INT_ARRAY:
      attr->values.int_array = GTXClientFreePointer(attr->values.int_array);
      break;
    case _GTX_VENDOR_DATA_TYPE_DOUBLE_ARRAY:
      attr->values.double_array =
        GTXClientFreePointer(attr->values.double_array);
      break;
    case _GTX_VENDOR_DATA_TYPE_STRING_ARRAY:
      attr->values.string_array =
        GTXClientFreeStringArray(attr->nvalues, attr->values.string_array);
      break;
    default:
      break;
  }
  free(attr);
  return(NULL);
}

/****************************************************************************
**
** FUNCTION:    st_start_attribute
**
** DESCRIPTION: starts a new attribute. Check name non-existence
**
** RETURNS:     new attribute or NULL on error
**
** IN_ARGS:     vendor_data: vendor data
** IN_ARGS:     name:        attribute name (will be copied)
** IN_ARGS:     type:        attribute type
**
*****************************************************************************/
static GTXVendorDataAttrRec *
st_start_attribute(GTXVendorData vendor_data,
                   const char *name,
                   _GTX_VENDOR_DATA_TYPE type)
{
  GTXVendorDataAttrRec *new,*next;

  next = vendor_data->attributes;
  while (next != NULL)
  {
    if (!strcmp(next->name,name))
    {
      _gtx_error("Attribute %s already exists",name);
      return(NULL);
    }
    next = next->next;
  }

  new = (GTXVendorDataAttrRec *)calloc(1, sizeof(GTXVendorDataAttrRec));
  if (new == NULL)
  {
    _gtx_error("Memory allocation error for attribute %s", name);
    return(NULL);
  }
  new->nvalues = 0;
  new->next = NULL;

  (void)strcpy(new->name,name);
  
  new->type = type;
  return(new);
}

/****************************************************************************
**
** FUNCTION:    st_end_attribute
**
** DESCRIPTION: adds the new attribute to vendor data
**
** RETURNS:     NULL because it must not be freed in calling function
**
** IN_ARGS:     vendor_data: vendor data
** IN_ARGS:     new:         new attribute
**
*****************************************************************************/
static GTXVendorDataAttrRec *
st_end_attribute(GTXVendorData vendor_data,
                 GTXVendorDataAttrRec *new)
{
  GTXVendorDataAttrRec *next;

  next = vendor_data->attributes;
  if (next != NULL)
  {
    while (next->next != NULL) next = next->next;
    next->next = new;
  }
  else
    vendor_data->attributes = new;
  return(NULL);
}

/****************************************************************************
**
** FUNCTION:    _gtx_vendor_data_search_attribute
**
** DESCRIPTION: Searches for an attribute in a Vendor Data
**
** RETURNS:     found attribute or NULL on error (message sent)
**
** IN_ARGS:     vendor_data: vendor data
** IN_ARGS:     name:        attribute name
** IN_ARGS:     exp_type:    expected attribute type
**
*****************************************************************************/
GTXVendorDataAttrRec *
_gtx_vendor_data_search_attribute(GTXVendorData_C vendor_data, const char *name, _GTX_VENDOR_DATA_TYPE exp_type)
{
  GTXVendorDataAttrRec *attr;
 
  if( vendor_data == NULL ) goto label_not_found;
 
  attr = vendor_data->attributes;
  while( attr != NULL )
  {
    if( !strcmp(attr->name, name) ) break;
    attr = attr->next;
  }
 
  if( attr == NULL ) goto label_not_found;
 
  if( attr->type != exp_type )
  {
    _gtx_error("Attribute named %s found with wrong type", name);
    return NULL;
  }
 
  return attr;
 
label_not_found:
  _gtx_error("Attribute named %s not found", name);
  return NULL;
}

/*!
******************************************************************************
\fn GTXVendorData GTXClientVendorDataDuplicate(GTXVendorData vendor_data)
\brief Copies a given Vendor Data
 
This function copies a Vendor Data 
\return new copy of the Vendor Data
\param vendor_data original Vendor Data
*****************************************************************************/
GTXVendorData GTXClientVendorDataDuplicate(GTXVendorData vendor_data)
{
  GTXVendorData tmp = NULL;
  GTXVendorDataAttrRec * move = vendor_data->attributes;


  tmp = GTXClientVendorDataNew(vendor_data->identificator, vendor_data->version);
  while (move != NULL)
  {
    switch (move->type)
    {
      case _GTX_VENDOR_DATA_TYPE_NOTHING:
      case _GTX_VENDOR_DATA_TYPE_INT:
        GTXClientVendorDataAddAttributeInt(tmp, move->name,move->values.val_int);
        break;
      case _GTX_VENDOR_DATA_TYPE_DOUBLE:
        GTXClientVendorDataAddAttributeDouble(tmp, move->name,move->values.val_double);
        break;
      case _GTX_VENDOR_DATA_TYPE_STRING:
        GTXClientVendorDataAddAttributeString(tmp, move->name, move->values.val_string);
        break;
      case _GTX_VENDOR_DATA_TYPE_INT_ARRAY:
        GTXClientVendorDataAddAttributeIntArray(tmp, move->name,move->nvalues, move->values.int_array);
        break;
      case _GTX_VENDOR_DATA_TYPE_DOUBLE_ARRAY:
        GTXClientVendorDataAddAttributeDoubleArray(tmp, move->name,move->nvalues, move->values.double_array);
        break;
      case _GTX_VENDOR_DATA_TYPE_STRING_ARRAY:
        GTXClientVendorDataAddAttributeStringArray(tmp, move->name,move->nvalues, (const char * const *)move->values.string_array);
        break;
      default:
        break;
    }
    move = move->next;
  }
  return tmp;  
}


