/*****************************************************************************
 
            Copyright (c)2009 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_get_faults.c 11546 2008-01-11 18:08:19Z foucher $";

#include <string.h> /* strcpy() on solaris */

/*!
******************************************************************************
\brief Get wireframe system associated to the current file

This function retrieve, from the GTXserver, the wireframe system
associated to the current file. Wireframes in the wireframe system can be masked
with a selection.
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\retval wireframe_system Wireframe system associated to the current file

\par Remarks:
All coordinates are returned in the current unit (\ref GTXClientSetUnit) or
using the corresponding coordinate variable unit if unit_mode is set to 1
(\ref GTXClientSetUnitMode).

Some memory is allocated inside passed wireframe_system. Memory must be freed
using GTXClientFreeWireframes.
*****************************************************************************/
int GTXClientReadWireframes(GTXWireframeSystem *wireframe_system)
{
  char *s[1];
  int i, j;
  GTXErrorCode error;
  char answer[10];
  GTXWireframe *wireframe;
  GTXWireframeObject *object;

  GTX_TRACE_FUNC_START("GTXClientReadWireframes",1);
  GTX_TRACE(1, ("(%d,%p)", wireframe_system));

  wireframe_system->wireframes = NULL;
  wireframe_system->n_wireframes = 0;

  error = GTX_ERROR_SERVER;
  if (!_gtx_is_supported("CWIRENEW", 1))
    goto label_end;

  error = GTX_ERROR_NETWORK;
  if( _gtx_start_packet("CWIREREAD") ) goto label_end;

  if( _gtx_send_and_wait(answer) ) goto label_end;

  error = GTX_ERROR_SERVER;
  if( _gtx_check_answer(answer, "SWIREF") ) goto label_end;

  error = GTX_ERROR_NETWORK;
  if (!_gtx_read_int(_gtx_server_socket, &wireframe_system->n_wireframes))
    goto label_end;

  error = GTX_ERROR_MEMORY;
  wireframe_system->wireframes = (GTXWireframe*)malloc(sizeof(GTXWireframe) * wireframe_system->n_wireframes);
  if(wireframe_system->wireframes == NULL) goto label_end;
  for (i = 0; i < wireframe_system->n_wireframes; i++)
    wireframe_system->wireframes[i].objects = NULL;

  error = GTX_ERROR_NETWORK;
  wireframe = wireframe_system->wireframes;
  for(i=0; i<wireframe_system->n_wireframes; i++, wireframe++)
  {
    s[0] = wireframe->name;
    if (!_gtx_read_string(_gtx_server_socket, s, 50) ||
        !_gtx_read_int(_gtx_server_socket, &wireframe->color[0]) ||
        !_gtx_read_int(_gtx_server_socket, &wireframe->color[1]) ||
        !_gtx_read_int(_gtx_server_socket, &wireframe->color[2]) ||
        !_gtx_read_int(_gtx_server_socket, &wireframe->n_objects))
      goto label_end;

    error = GTX_ERROR_MEMORY;
    wireframe->objects = (GTXWireframeObject*)malloc(sizeof(GTXWireframeObject) * wireframe->n_objects);
    if (wireframe->objects == NULL) goto label_end;

    /* init objects */
    object = wireframe->objects;
    for (j = 0; j < wireframe->n_objects; j++, object++)
    {
      object->vertices = NULL;
      object->triangles = NULL;
    }

    error = GTX_ERROR_NETWORK;
    object = wireframe->objects;
    for (j = 0; j < wireframe->n_objects; j++, object++)
    {
      if (!_gtx_read_int(_gtx_server_socket, &object->n_vertices) ||
          !_gtx_read_int(_gtx_server_socket, &object->n_triangles))
        goto label_end;
      
      error = GTX_ERROR_MEMORY;
      object->vertices = (double*)malloc(3*sizeof(double)*object->n_vertices);
      object->triangles = (int*)malloc(3*sizeof(int)*object->n_triangles);
      if (object->vertices == NULL || object->triangles == NULL)
        goto label_end;

      error = GTX_ERROR_NETWORK;
      if (!_gtx_rw_vector(_gtx_server_socket, (char*)object->vertices,
                          object->n_vertices*3,
                          sizeof(double), (_gtx_proc_rw_t)_gtx_read_double))
        goto label_end;
      if (!_gtx_rw_vector(_gtx_server_socket, (char*)object->triangles,
                          object->n_triangles*3,
                          sizeof(int), (_gtx_proc_rw_t)_gtx_read_int))
        goto label_end;
    } /* loop on active objects */
  } /* end of loop on wireframes */

  error = GTX_ERROR_NONE;

label_end:
  if (error)
  {
    GTXClientFreeWireframes(wireframe_system);
    _gtx_cleanup_socket();
  }
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Frees memory allocated in the wireframe system

This function frees the memory allocated by a previous call to
GTXClientReadWireframes
\param wireframe_system Wireframe system to be freed

\par Remarks:
This function is meant to only free memory allocated by GTXClientReadWireframes.
Only call this function AFTER a call to GTXClientReadWireframes.
Elsewhere memory in the wireframe system is not initialized and an attempt to
free it could cause problems.
Indeed this function frees wireframe_system->wireframes so if you may call this
function before read, first initialize wireframes to NULL.
*****************************************************************************/
void GTXClientFreeWireframes(GTXWireframeSystem *wireframe_system)
{
  int i;
  GTX_TRACE_FUNC_START("GTXClientFreeWireframes",1);
  GTX_TRACE(1, ("(%p)", wireframe_system));
  if (wireframe_system->wireframes)
  {
    for (i = 0; i < wireframe_system->n_wireframes; i++)
    {
      if (wireframe_system->wireframes[i].objects)
        free(wireframe_system->wireframes[i].objects);
    }
    free(wireframe_system->wireframes);
    wireframe_system->wireframes = NULL;
  }
  wireframe_system->n_wireframes = 0;
  GTX_TRACE_FUNC_END(NULL,NULL);
}

/*!
******************************************************************************
\brief Copies a given Wireframe Object
 
This function copies a Wireframe Object
\param object_in original Wireframe Object
\retval object_out pointer on destination Wireframe Object
*****************************************************************************/
void GTXClientWireframeObjectDuplicate(const GTXWireframeObject *object_in,
                                       GTXWireframeObject *object_out)
{
  int i;

  object_out->n_vertices = object_in->n_vertices;
  object_out->n_triangles = object_in->n_triangles;
  object_out->vertices = (double*)malloc(3*sizeof(double)*object_out->n_vertices);
  object_out->triangles = (int*)malloc(3*sizeof(int)*object_out->n_triangles);
  if (object_out->vertices == NULL || object_out->triangles == NULL)
  {
    object_out->n_vertices = 0;
    object_out->n_triangles = 0;
    return;
  }

  for (i = 0; i < object_in->n_vertices*3; i++)
    object_out->vertices[i] = object_in->vertices[i];
  for (i = 0; i < object_in->n_triangles*3; i++)
    object_out->triangles[i] = object_in->triangles[i];
}

/*!
******************************************************************************
\brief Copies a given Wireframe
 
This function copies a Wireframe
\param wireframe_in original Wireframe
\retval wireframe_out pointer on destination Wireframe
*****************************************************************************/
void GTXClientWireframeDuplicate(const GTXWireframe *wireframe_in,
                                 GTXWireframe *wireframe_out)
{
  int i;

  (void)strcpy(wireframe_out->name,wireframe_in->name);
  wireframe_out->color[0] = wireframe_in->color[0];
  wireframe_out->color[1] = wireframe_in->color[1];
  wireframe_out->color[2] = wireframe_in->color[2];
  wireframe_out->n_objects = wireframe_in->n_objects;
  wireframe_out->objects = malloc(wireframe_in->n_objects * sizeof(GTXWireframeObject));
  if (wireframe_out->objects == NULL)
  {
    wireframe_out->n_objects = 0;
    return;
  }
  
  for (i = 0; i < wireframe_out->n_objects; i++)
    GTXClientWireframeObjectDuplicate(&wireframe_in->objects[i], &wireframe_out->objects[i]);
}

/*!
******************************************************************************
\brief Copies a given Wireframe System
 
This function copies a Wireframe System
\return new copy of the Wireframe System
\param wsys_in original Wireframe System
\retval wsys_out copied Wireframe System
*****************************************************************************/
void GTXClientWireframeSystemDuplicate(const GTXWireframeSystem *wsys_in,
                                       GTXWireframeSystem *wsys_out)
{
  int i;

  wsys_out->n_wireframes = wsys_in->n_wireframes;
  wsys_out->wireframes = malloc(wsys_in->n_wireframes * sizeof(GTXWireframe));
  if (wsys_out->wireframes == NULL)
    wsys_out->n_wireframes = 0;
  else
    for (i = 0; i < wsys_in->n_wireframes; i++)
      GTXClientWireframeDuplicate(&wsys_in->wireframes[i], &wsys_out->wireframes[i]);
}


