/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: api_write_float_var.c 25951 2016-04-07 05:52:23Z foucher $";

#include <string.h> /* strcpy() on solaris */

static GTXErrorCode st_write_var_nocheck(gtx_long first, gtx_long last,
                                         int compress, double test_val,
                                         gtx_long nb_val, const double *data)
{
  char answer[10];
  GTXErrorCode error;

  error = GTX_ERROR_NETWORK;

  if (_gtx_start_packet("CWRITVARF")) goto label_end;

  if (SERVER_REQ_4_1_1 &&
      (!_gtx_write_long(_gtx_server_socket, &first) ||
       !_gtx_write_long(_gtx_server_socket, &last)))
    goto label_end;

  if (!_gtx_write_int(_gtx_server_socket, &compress) ||
      !_gtx_write_double(_gtx_server_socket, &test_val)) goto label_end;
  
  if (!_gtx_write_long(_gtx_server_socket, &nb_val))
    goto label_end;

  if (!_gtx_rw_vector(_gtx_server_socket, (char*)data, nb_val,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double))
    goto label_end;

  if (_gtx_send_and_wait(answer)) goto label_end;

  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;

  error = GTX_ERROR_NONE;

label_end:
  if (error) _gtx_cleanup_socket();
  return error;
}

static GTXErrorCode st_write_var(gtx_long first, gtx_long last,
                                 int compress, double test_val,
                                 gtx_long nb_val, const double *data)
{
  GTXErrorCode error;
  int i, j, k, iline, nlines, *scount;
  int lfirst, lnfirst;
  double *data2;

  data2 = NULL;
  scount = NULL;

  /* We need to take care of old servers for the line case
  ** because creating core lines adds false sample for each line */
  if (_gtx_client_server_version < 900000 && !compress)
  {
    GTXFileInfo finfo;
    error = (GTXErrorCode)GTXClientGetFileInfo(&finfo);
    if (error != GTX_ERROR_NONE) goto label_end;
    if (finfo.type == GTX_FILE_TYPE_GRAVITY_LINES &&
        _gtx_is_supported("CLFILECNT", 0))
    {
      error = _gtx_get_line_samples(&nlines, &scount);
      if (error != GTX_ERROR_NONE) goto label_end;

      /* writing the complete file with missing blanks */
      if (((first == -1 && last == -1) ||
           (first == 1 && last == finfo.s_number - finfo.item_number)) &&
          (nb_val == finfo.s_number - finfo.item_number))
      {
        data2 = (double*)malloc(sizeof(double) * finfo.s_number);
        if (data2 == NULL) goto label_end;
        for (iline = j = k = 0; iline < finfo.item_number; iline++)
        {
          data2[k++] = test_val;       /* insert empty value */
          for (i = 0; i < scount[iline]-1; i++)
            data2[k++] = data[j++];
        }
        error = st_write_var_nocheck(1, finfo.s_number, compress, test_val, finfo.s_number, data2);
        goto label_end;
      }
      /* writing an only line */
      else if (first != -1)
      {
        error = _gtx_get_line_samples(&nlines, &scount);
        if (error != GTX_ERROR_NONE) goto label_end;

        lfirst = lnfirst = 1;
        for (iline = 0; iline < finfo.item_number; iline++)
        {
          // If we found Sample Numbers corresponding to a complete line with shifted indices
          if (first == lfirst && last == lfirst + scount[iline]-2)
            break;
          lfirst += scount[iline]-1;
          lnfirst += scount[iline];
        }
        if (iline != finfo.item_number)
        {
          // write the defined samples
          error = st_write_var_nocheck(lnfirst + 1, lnfirst + scount[iline] - 1, compress, test_val,
                                       scount[iline]-1, data);
          goto label_end;
        }
      }
      /* else we cannot do really more, let the error occur... */
    }
  } /* end of version < 9 management */

  error = st_write_var_nocheck(first, last, compress, test_val, nb_val, data);

label_end:
  if (data2 != NULL) free(data2);
  if (scount != NULL) free(scount);
  if (error) _gtx_cleanup_socket();
  return error;
}

/*!
******************************************************************************
\brief Write data to the current numerical variable
 
This function writes data to the current numerical variable
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param compress Samples masked by the selection are in the array but ignored
                if 0, or do not appear in the array if this flag is set to 1
\param test_val Value for undefined sample
\param nb_val Number of samples in the array (must be set to the number
              of selected samples in the current file in compressed mode or
              the number of samples in the file in non-compressed mode)
\param data Array of numerical variable samples
 
\par Remarks:
All the numerical variables are stored in double precision arrays

If the variable is a length one, data must be in the current unit
(\ref GTXClientSetUnit) if unit_mode is 0 (\ref GTXClientSetUnitMode).
If the variable is a length one and unit_mode is 1, it must be expressed in
the variable unit.
*****************************************************************************/
int GTXClientWriteDoubleVariable(int compress, double test_val,
                                 gtx_long nb_val, const double *data)
{
  GTXErrorCode error;
  GTX_TRACE_FUNC_START("GTXClientWriteDoubleVariable",1);
  GTX_TRACE(1, ("(%d,%g,%"GTXLONG_FORMAT",%p)", compress,test_val,nb_val,data));
  error = st_write_var(-1,-1,compress,test_val,nb_val,data);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (int)(error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Write continuous subpart of data to the current numerical variable
 
This function writes a subpart of data to the current numerical variable
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param first Sample Number of the first item to write (from 1 to nsample)
\param last  Sample Number of the last item to write (from 1 to nsample)
\param compress Samples masked by the selection are in the array but ignored
                if 0, or do not appear in the array if this flag is set to 1
\param test_val Value for undefined sample
\param nb_val Number of values in the data array (must be set to the number
              of selected samples in the specified range in compressed mode or
              the number of samples in the range in non-compressed mode)
\param data Array of numerical variable samples
 
\par Remarks:
All the numerical variables are stored in double precision arrays
Warning: This functions needs a server version >= 4.0.7 for 4.0.x series or
>= 4.1.1 for 4.1.x series.
*****************************************************************************/
int GTXClientWriteDoubleVariableSub(gtx_long first, gtx_long last,
                                    int compress, double test_val,
                                    gtx_long nb_val, const double *data)
{
  GTXErrorCode error = GTX_ERROR_NETWORK;

  GTX_TRACE_FUNC_START("GTXClientWriteDoubleVariableSub",1);
  GTX_TRACE(1, ("(%"GTXLONG_FORMAT",%"GTXLONG_FORMAT",%d,%g,%"GTXLONG_FORMAT",%p)",
                first,last,compress,test_val,nb_val,data));
  if (!SERVER_REQ_4_1_1)
  {
    _gtx_error("GTXClientWriteDoubleVariableSub() "
               "does not work with a server version lower than "
               "4.0.7 for 4.0.x series and 4.1.1 for 4.1.x series");
    _gtx_error("Please upgrade your server "
               "(http://www.geovariances.fr)");
  }
  else
    error = st_write_var(first, last, compress, test_val, nb_val, data);
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return (int)(error != GTX_ERROR_NONE);
}

/*!
******************************************************************************
\brief Write line data to the current numerical variable
 
This function writes line data to the current numerical variable
\return error code:
  \arg 0 if Ok
  \arg 1 on error (see \ref GTXClientGetLastError)
\param line_index The line index [1..nline]
\param test_val   Value for undefined sample
\param nsample    Number of samples in the array (must be set to the number
                  of samples of this line even with a active selection)
\param data       Array of numerical variable samples
 
\par Remarks:
All the numerical variables are stored in double precision arrays

Warning: This functions needs a server version >= 8.06beta.

You might consider using \sa GTXClientWriteDoubleVariableSub if you
potentially have a lot of lines to write for efficiency.

*****************************************************************************/
GLOBAL int GTXClientWriteLineDoubleVariable(int line_index, double test_val, int nsample, const double *data)
{
  char answer[10];
  GTXErrorCode error;
  gtx_long tmp;
  
  error = GTX_ERROR_NETWORK;
  GTX_TRACE_FUNC_START("GTXClientWriteLineDoubleVariable",1);
  GTX_TRACE(1, ("(%d,%g,%d,%p)", line_index, test_val, nsample, data));

  if (!_gtx_is_supported("CLFILECNT", 1)) /* CWRITLINF was buggy in first version */
    goto label_end;
  if (_gtx_start_packet("CWRITLINF")) 
    goto label_end;
  
  /* server expects line_index and nsample to be long */
  tmp = line_index;
  if (!_gtx_write_long(_gtx_server_socket, &tmp))
    goto label_end;
  
  tmp = nsample;
  if (!_gtx_write_long(_gtx_server_socket, &tmp))
    goto label_end;

  if (!_gtx_write_double(_gtx_server_socket, &test_val))
    goto label_end;
  
  if (!_gtx_rw_vector(_gtx_server_socket, (char*)data, nsample,
                      sizeof(double), (_gtx_proc_rw_t)_gtx_write_double))
    goto label_end;
  
  if (_gtx_send_and_wait(answer)) goto label_end;
  error = GTX_ERROR_SERVER;
  if (_gtx_check_answer(answer, "SDONE")) goto label_end;
  
  error = GTX_ERROR_NONE;
  
label_end:
  if (error) _gtx_cleanup_socket();
  GTX_TRACE_FUNC_END("%d",error);
  _gtx_client_last_error = error;
  return error;
}
