/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/
 
#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: network.c 16194 2010-03-04 16:31:34Z foucher $";

#include <signal.h> /* signal() */
#include <string.h> /* memcpy() */
#ifdef winnt
#include <winsock.h>
#else
#include <unistd.h>  /* close() */
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/select.h>
#include <sys/resource.h>
#endif

#if defined(winnt)
static int WSAinitialized = 0;
static WSADATA WSAData;
#endif

/***************************************************************************
**
** FUNCTION:    _gtx_listen_port
**
** DESCRIPTION: Open a listening port
**
** RETURNS:     error: error_code : 0 if Ok, !0 elsewhere
**
** IN_ARGS:     port : port to listen to (0 for free)
** IN_ARGS:     wait_queue_length: 1 if will accept only one connection
**
** OUT_ARGS:    port : new dynamicaly chosen port
** OUT_ARGS:    _GTX_SOCKET *listen_socket: opened listening socket
**
***************************************************************************/
int _gtx_listen_port(unsigned short *port,
                     int wait_queue_length,
                     _GTX_SOCKET *listen_socket)
{
  struct sockaddr_in server_address;
  int err,one = 1;
#ifdef winnt
  int namelen;
#else
  socklen_t namelen;
#endif

  _gtx_winsock_init();
  GTX_TRACE(4,("_gtx_listen_port(%p=%d,%d,%p)", port, port == NULL? 0: *port,
               wait_queue_length, listen_socket));
#if defined(winnt)
  err = ((*listen_socket = socket(AF_INET, SOCK_STREAM, 0)) == SOCKET_ERROR);
#else
  err = ((*listen_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0);
#endif
  if (err)
  {
    _gtx_error("Error during socket creation.");
    return(1);
  }
  GTX_TRACE(4,("Created Listening socket: %d", (int)*listen_socket));

#if !defined(winnt) /* SO_REUSEADDR not supported in winsock */
  GTX_TRACE(4,("Setting socket option: REUSEADDR"));
  if ((setsockopt(*listen_socket, SOL_SOCKET, SO_REUSEADDR, (char *)&one,
                  sizeof(one))) == -1)
  {
    _gtx_serror("Could not set socket option SO_REUSEADDR");
    exit(1);
  }
#endif

  GTX_TRACE(4,("Binding socket to 0.0.0.0:%d", *port));
  /*get_myaddress(&server_address);*/
  server_address.sin_family = AF_INET;
  server_address.sin_addr.s_addr = 0;
  server_address.sin_port = htons(*port);
#if defined(winnt)
  err = (bind(*listen_socket, (struct sockaddr *)&server_address,
              sizeof(struct sockaddr_in)) == SOCKET_ERROR);
#else
  err = (bind(*listen_socket, (struct sockaddr *)&server_address,
              sizeof(struct sockaddr_in)));
#endif
  if (err)
  {
    _gtx_serror("Cannot bind socket");
    _gtx_close_socket(listen_socket);
    _gtx_winsock_free();
    return(2);
  }     

  GTX_TRACE(4,("Set socket to listen for %d connections", wait_queue_length));
  if (listen(*listen_socket, wait_queue_length))
  {
    _gtx_serror("Cannot listen to socket");
    _gtx_close_socket(listen_socket);
    _gtx_winsock_free();
    return(3);
  }
  GTX_TRACE(4,("Socket listening."));

  /* Return connected port */
  if (*port == 0)
  {
    namelen = sizeof(server_address);
    if (getsockname(*listen_socket, (struct sockaddr *)&server_address,
                    &namelen) < 0)
    {
      _gtx_error("Cannot get new port.");
      return(4);
    }
    *port = ntohs(server_address.sin_port);
    _gtx_debug("New Socket Port: %d", *port);
  }
  return(0);
}

/***************************************************************************
**
** FUNCTION:    _gtx_wait_connect
**
** DESCRIPTION: Wait for connection on our port
**
** RETURNS:     error_code : 0 if Ok, -1 with a real_error, 1 if no
** RETURNS:                  connection happened on non-blocking socket
**
** IN_ARGS:     listen_socket: opened listening socket
** IN_ARGS:     blocking:      0 for a non-blocing socket, 1 if blocking
** IN_ARGS:     time_s:        timeout for non-blocking sockets (in seconds)
**
** OUT_ARGS:    comm_socket:   opened communication socket
**
***************************************************************************/
int _gtx_wait_connect(_GTX_SOCKET listen_socket,
                      _GTX_SOCKET *comm_socket,
                      int blocking,
                      int time_s)
{
  int err,addr;
  fd_set fdset;
  struct timeval timeout;       /* Time select (sec, usec) */
  struct sockaddr_in client_address;
#ifdef winnt
  int socket_size;
#else
  socklen_t socket_size;
#endif

#ifndef winnt
  struct rlimit rlp;
#endif

  _gtx_debug("Wait for connection...");
  GTX_TRACE(4,("_gtx_wait_connect(%d,%p,%d,%d)",
              (int)listen_socket, comm_socket, blocking, time_s));

  /* Case of non-blocking connection -> use select */
  if (! blocking)
  {
    FD_ZERO(&fdset);
    FD_SET(listen_socket, &fdset);
    timeout.tv_sec = time_s; timeout.tv_usec = 0;

#if defined (winnt)
    err = (select(0 /*unused in winsock */, &fdset,
                  NULL, NULL, &timeout) < 0);
#elif defined (aix)
    err = (select(howmany(FD_SETSIZE, NFDBITS), &fdset,
                  NULL, NULL, &timeout) < 0);
#else
    (void)getrlimit(RLIMIT_NOFILE, &rlp);
#if defined (hpux)
    err = (select((size_t)rlp.rlim_cur, (int*)&fdset,
                  NULL, NULL, &timeout) < 0);
#else  /* hpux */
    err = (select((int)rlp.rlim_cur, &fdset,
                  NULL, NULL, &timeout) < 0);
#endif
#endif
    if (err)
    {
      _gtx_perror("_gtx_wait_connect: select() failed.");
      return(-1);
    }

    /* no connection -> return(1) */
    if (FD_ISSET(listen_socket, &fdset) == 0)
      return(1);
  }

  GTX_TRACE(4,("Calling accept on socket"));
  /* Blocking mode or connection in non-blocking*/
  socket_size = sizeof(struct sockaddr_in);
  *comm_socket = accept(listen_socket,
                        (struct sockaddr *)&client_address,
                        &socket_size);
#if defined(winnt)
  err = (*comm_socket == SOCKET_ERROR);
#else
  err = (*comm_socket < 0);
#endif

  if (err)
  {
    _gtx_serror("Call to accept() failed.\n");
#ifdef DEBUG
    perror("accept()");
#endif
    _gtx_close_socket(&listen_socket);
#if defined(winnt)
    _gtx_winsock_free();
#endif
    return(-1);
  }
  addr = ntohl(client_address.sin_addr.s_addr);
  _gtx_debug("Get a client connection from %d.%d.%d.%d:%d.",
             (addr>>24)&0xff,(addr>>16)&0xff,(addr>>8)&0xff,addr&0xff,
             ntohs(client_address.sin_port));
  GTX_TRACE(4,("New socket for communication: %d", (int)*comm_socket));
  return(0);
}

/***************************************************************************
**
** FUNCTION:    _gtx_connect
**
** DESCRIPTION: connect to the given host/port
**
** RETURNS:     int error   : error_code : 0 if Ok, !0 elsewhere
**
** IN_ARGS:     char *host : server name or ip adress
** IN_ARGS:     unsigned short port     : server port
**
** OUT_ARGS:    _GTX_SOCKET *comm_socket: opened communication socket
**
** REMARKS:     write buffer is emptied. As it is unique for all sockets
** REMARKS:     all writes must have been flushed before.
**
***************************************************************************/
int _gtx_connect(const char *host,
                 unsigned short port,
                 _GTX_SOCKET *comm_socket)
{
  int err,addr;
  struct hostent *server_machine; /* server's machine */
  struct sockaddr_in server_adress; /* server IP address */

  GTX_TRACE(4, ("_gtx_connect(%s,%d,%p)",host,port,comm_socket));
  _gtx_empty_write_buffer();

#if defined(winnt)
  /* this code is to workaround a problem with 'localhost' */
  if (!strcmp(host, "localhost") || !strcmp(host, "127.0.0.1"))
  {
    char hostname[256];
    DWORD size = 256;
    GTX_TRACE(4,("Calling GetComputerName instead of using localhost"));
    if (!GetComputerName((LPTSTR)hostname, &size))
    {
      _gtx_perror("_gtx_connect(%s:%d), GetComputerName",
                              host, port);
      return(1);
    }
    GTX_TRACE(4,("GetComputerName returned %s", hostname));
    server_machine = gethostbyname(hostname);
  }
  else
    server_machine = gethostbyname(host);
#else
  (void)sethostent(0);
  server_machine = gethostbyname(host);
  endhostent();
#endif

  if (server_machine == NULL)
  {
    _gtx_herror("_gtx_connect(%s:%d), gethostbyname", host, port);
    return(1);
  }
  addr = ntohl(*(int*)server_machine->h_addr);
  GTX_TRACE(4,("gethostbyname returned name=%s, addr=%d.%d.%d.%d",
               server_machine->h_name,
               (addr>>24)&0xff,(addr>>16)&0xff,(addr>>8)&0xff,addr&0xff));

  /* Copy of the server coordinates */
  (void)memcpy((char *)&server_adress.sin_addr,
               server_machine->h_addr,
               server_machine->h_length);

  server_adress.sin_family = server_machine->h_addrtype; 
  server_adress.sin_port   = htons(port);

  /* socket creation */
  GTX_TRACE(4,("Creating socket for connection"));
  *comm_socket = socket(AF_INET, SOCK_STREAM, 0);
#if defined(winnt)
  err = (*comm_socket == _GTX_SOCKET_ERROR);
#else
  err = (*comm_socket < 0);
#endif
  if (err)
  {
    _gtx_serror("_gtx_connect(%s:%d), socket creation failed",
                            host, port);
    *comm_socket = _GTX_SOCKET_ERROR;
    return 2;
  }
    
  /* connection try */
  GTX_TRACE(4,("Socket %d created, connecting to %s:%d",
               (int)*comm_socket, host, port));
  if (connect(*comm_socket, (struct sockaddr *)&server_adress,
              sizeof(struct sockaddr_in)))
  {
    _gtx_serror("_gtx_connect(%s:%d), connect",
                            host, port);
    _gtx_close_socket(comm_socket);
    return(3);
  }

  GTX_TRACE(4, ("... connected"));
  return(0);
}

/***************************************************************************
**
** FUNCTION:    _gtx_close_socket
**
** DESCRIPTION: close a given socket
**
** IN_ARGS:     _GTX_SOCKET *sockptr: opened communication socket
**
** OUT_ARGS:    _GTX_SOCKET *sockptr: close communication socket
**
***************************************************************************/
void _gtx_close_socket(_GTX_SOCKET *sockptr)
{
  GTX_TRACE(4, ("_gtx_close_socket(%p=%d)",sockptr,
                (sockptr==NULL)?0:(int)*sockptr));
  if (*sockptr == _GTX_SOCKET_ERROR)
    return;
#if defined(winnt)
  closesocket(*sockptr);
#else
  (void)close(*sockptr);
#endif
  *sockptr = _GTX_SOCKET_ERROR;
}

/****************************************************************************
**
** FUNCTION: _gtx_winsock_init
**
** PURPOSE:  Initialiaze the Windows Socket API
**
** RETURNS:  0 if OK, 1 if cannot be initialized
**
** REMARKS:  Does it only once in our code
**
*****************************************************************************/
int _gtx_winsock_init(void)
{
#if defined(winnt)
  if (!WSAinitialized)
  {
    GTX_TRACE(1,("Initializing Windows Sockets"));
    if (WSAStartup(MAKEWORD(1,1), &WSAData))
    {
      _gtx_herror("Unable to initialize Winsock.");
      GTX_TRACE_FUNC_END("1",NULL);
      return(1);
    }
    WSAinitialized = 1;
  }
#endif
  return (0);
}

/****************************************************************************
**
** FUNCTION: _gtx_winsock_free
**
** PURPOSE:  Free the windows socket API
**
** REMARKS:  Will only do it if already initialized by our code
**
*****************************************************************************/
void _gtx_winsock_free(void)
{
#if defined(winnt)
  if (WSAinitialized)
  {
    WSACleanup();
    WSAinitialized = 0;
  }
#endif
}
