/*****************************************************************************
 
            Copyright (c)2007 Geovariances, Avon, France.
 
    In consideration  of payment of  the license fee,  which is a part of
    the price you  paid for this  product, Geovariances (GV) as licensor,
    grants you, the licensee, a non-exclusive right to use this copy of a
    GV software product.
    GV reserves all rights not  expressly granted to licensee. GV retains
    titleship and ownership  of software.  This license is not  a sale of
    the original  software or any  copy. GV also  retains  titleship  and
    ownership of any modifications or  derivations of this software.  Any
    modifications of this software  must be clearly marked as such.  This
    copyright message must  appear in its entirety  in this software,  or
    any modifications or derivations thereof.
 
    Geovariances welcomes any comments, suggestions, bug reports, etc. At
    the discretion  of Geovariances,  any customer  supplied  bug  fixes,
    enhancements, or utility codes will be distributed in future software
    releases (the contributor will of course be credited).
 
            Geovariances
            49bis, Avenue Franklin Roosevelt
            77210 Avon, FRANCE
 
             Phone: +33-(0)-160.749.100
               Fax: +33-(0)-164.228.728
            e-mail: support@geovariances.fr
 
                        All Rights Reserved
 
*****************************************************************************/

#include <GTXClientP.h>

static char svnid[] _GTX_UNUSED = "$Id: read_write.c 11818 2008-02-22 16:21:23Z foucher $";

#include <malloc.h>
#include <limits.h>             /* UINT_MAX */
#if !defined(winnt)
#include <netinet/in.h>
#endif

enum {_BUFFER_MAXSIZE=1024};
static char _buffer[_BUFFER_MAXSIZE];
static int _buffer_nbytes = 0;
// We use the same buffer for reading and writing because we never both at once.
static int _buffer_rnbytes = 0;
static int _buffer_rpos = 0;
int _gtx_disable_rw_trace = 0;

/****************************************************************************
**
** FUNCTION:    _gtx_flush_read
**
** DESCRIPTION: flush the internal read buffer
**
** RETURNS:     TRUE if OK, FALSE on error
**
** IN_ARGS:     s      : socket
** IN_ARGS:     nneeded: number of bytes needed for next reading
**
** REMARKS:     All reading operations must be preceded by _gtx_flush_read()
** REMARKS:     if not enough data is available
**
*****************************************************************************/
bool_t _gtx_flush_read(_GTX_SOCKET s,
                       int nneeded)
{
  int i, len, non_consumed;
  bool_t code = TRUE;

  GTX_TRACE(5,("Flushing Read buffer (%d bytes)", _buffer_rnbytes));
  non_consumed = (_buffer_rnbytes - _buffer_rpos);
  if (non_consumed)
  {
    for (i = 0; i < non_consumed; i++)
      _buffer[i] = _buffer[_buffer_rpos + i];
  }
  _buffer_rnbytes = non_consumed;
  _buffer_rpos = 0;

  while (_buffer_rnbytes - _buffer_rpos < nneeded)
  {
    len = recv(s, _buffer + _buffer_rnbytes,
         _BUFFER_MAXSIZE-_buffer_rnbytes, 0);
    if (len <= 0)
    {
      code = FALSE;
      break;
    }
    _buffer_rnbytes += len;
  }

  return code;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_bytes
**
** DESCRIPTION: reads bytes from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     len:  number of bytes
**
** OUT_ARGS:    addr: bytes array
**
*****************************************************************************/
bool_t _gtx_read_bytes(_GTX_SOCKET s, void *addr, gtx_ulong len)
{
  gtx_ulong i;
  char *char_ptr = (char*)addr;

  for (i = 0; i < len; i++)
  {
    if ((unsigned int)(_buffer_rnbytes-_buffer_rpos) < 1 &&
        !_gtx_flush_read(s,1))
      return FALSE;
    *(char_ptr++) = _buffer[_buffer_rpos++];
  }

  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_int
**
** DESCRIPTION: reads an int from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    ip: returned int
**
*****************************************************************************/
bool_t _gtx_read_int(_GTX_SOCKET s, int *ip)
{
  int i;
  char *char_ptr = (char*)ip;

  if (_buffer_rnbytes-_buffer_rpos < sizeof(int32_t) &&
      !_gtx_flush_read(s, sizeof(int32_t)))
    return FALSE;

  for (i = 0; i < sizeof(int32_t); i++)
    *(char_ptr++) = _buffer[_buffer_rpos++];

  *ip = (int)ntohl((int32_t)*ip);
  if (!_gtx_disable_rw_trace)
  { GTX_TRACE(3,("Read Int(%d)", *ip)); }
  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_long
**
** DESCRIPTION: reads a gtx_long (UNIX long 32/64) from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    ip: returned int
**
*****************************************************************************/
bool_t _gtx_read_long(_GTX_SOCKET s, gtx_long *lp)
{
  int i;
  char *char_ptr = (char*)lp;

#ifndef GTX_EXE_64
  if (_buffer_rnbytes-_buffer_rpos < sizeof(int32_t) &&
      !_gtx_flush_read(s,sizeof(int32_t)))
    return FALSE;

  for (i = 0; i < sizeof(int32_t); i++)
    *(char_ptr++) = _buffer[_buffer_rpos++];

  *lp = (int)ntohl((int32_t)*lp);
  if (!_gtx_disable_rw_trace)
  { GTX_TRACE(3,("Read Long(32bits)(%"GTXLONG_FORMAT")", *lp)); }
#else
  if (_gtx_client64 && _gtx_server64)
  {
    if (_buffer_rnbytes-_buffer_rpos < sizeof(gtx_long) &&
        !_gtx_flush_read(s, sizeof(gtx_long)))
      return FALSE;

    for (i = 0; i < sizeof(gtx_long); i++)
      *(char_ptr++) = _buffer[_buffer_rpos++];

# ifdef GTX_LITTLE_ENDIAN
    {
      int32_t tmp,*ip = (int32_t*)lp;
      tmp = ntohl(*ip);
      *ip = ntohl(*(ip+1));
      *(ip+1) = tmp;
    }
# endif
    if (!_gtx_disable_rw_trace)
    { GTX_TRACE(3,("Read Long(64bits)(%"GTXLONG_FORMAT")", *lp)); }
  }
  else
  {
    int32_t ival;
    char_ptr = (char*)&ival;
    if (_buffer_rnbytes-_buffer_rpos < sizeof(int32_t) &&
        !_gtx_flush_read(s, sizeof(int32_t)))
      return FALSE;

    for (i = 0; i < sizeof(int32_t); i++)
      *(char_ptr++) = _buffer[_buffer_rpos++];

    *lp = (gtx_long)(int32_t)ntohl(ival);
    if (!_gtx_disable_rw_trace)
    { GTX_TRACE(3,("Read Long(32->64bits)(%"GTXLONG_FORMAT")", *lp)); }
  }
#endif
  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_uint
**
** DESCRIPTION: reads an unsigned int from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    uip: returned unsigned int
**
*****************************************************************************/
bool_t _gtx_read_uint(_GTX_SOCKET s, unsigned int *uip)
{
  if (!_gtx_read_int(s, (int*)uip)) return FALSE;
  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_short
**
** DESCRIPTION: reads a short from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    sp: returned short
**
*****************************************************************************/
bool_t _gtx_read_short(_GTX_SOCKET s, short *sp)
{
  int i;
  if (!_gtx_read_int(s, &i)) return FALSE;
  *sp = (short)i;
  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_ushort
**
** DESCRIPTION: reads an unigned short from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    usp: returned unsigned short
**
*****************************************************************************/
bool_t _gtx_read_ushort(_GTX_SOCKET s, unsigned short *usp)
{
  unsigned int ui;
  if (!_gtx_read_int(s, (int*)&ui)) return FALSE;
  *usp = (unsigned short)ui;
  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_char
**
** DESCRIPTION: reads a char from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    cp: returned char
**
*****************************************************************************/
bool_t _gtx_read_char(_GTX_SOCKET s, char *cp)
{
  int i;
  if (!_gtx_read_int(s, &i)) return FALSE;
  *cp = (char)i;
  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_uchar
**
** DESCRIPTION: reads an unsigned char from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    ucp: returned unsigned char
**
*****************************************************************************/
bool_t _gtx_read_uchar(_GTX_SOCKET s, unsigned char *ucp)
{
  unsigned int ui;
  if (!_gtx_read_int(s, (int*)&ui)) return(FALSE);
  *ucp = (unsigned char)ui;
  return(TRUE);
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_double
**
** DESCRIPTION: reads a double from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    dp: returned double
**
*****************************************************************************/
bool_t _gtx_read_double(_GTX_SOCKET s, double *dp)
{
  int32_t *i32p;
  int i,old;
  bool_t code = FALSE;

  i32p = (int32_t*)dp;
  old = _gtx_disable_rw_trace;
  _gtx_disable_rw_trace = 1;

#if defined(GTX_BIG_ENDIAN)
  if (!_gtx_read_int(s, &i)) goto label_end;
  *i32p++ = (int32_t)i;
  if (!_gtx_read_int(s, &i)) goto label_end;
  *i32p = (int32_t)i;
#endif

#if defined(GTX_LITTLE_ENDIAN)
  if (!_gtx_read_int(s, &i)) goto label_end;
  *(i32p+1) = (int32_t)i;
  if (!_gtx_read_int(s, &i)) goto label_end;
  *i32p = (int32_t)i;
#endif

  _gtx_disable_rw_trace = old;
  if (!_gtx_disable_rw_trace)
  { GTX_TRACE(3,("Read Dbl(%g)", *dp)); }
  code = TRUE;
label_end:
  _gtx_disable_rw_trace = old;
  return code;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_bool
**
** DESCRIPTION: reads a boolean from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    bp: returned boolean
**
*****************************************************************************/
bool_t _gtx_read_bool(_GTX_SOCKET s, bool_t *bp)
{
  int ib;
  if (!_gtx_read_int(s, &ib)) return FALSE;
  *bp = (ib == 0) ? FALSE : TRUE;
  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_read_string
**
** DESCRIPTION: reads a string from a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** OUT_ARGS:    cpp:    returned string
** IN_ARGS:     maxsize: maximum size including \0 terminator
**
** REMARKS:     The parameter cpp references a pointer to storage;
** REMARKS:     If the pointer is null, then the necessary storage is
** REMARKS:     allocated.
**
*****************************************************************************/
bool_t _gtx_read_string(_GTX_SOCKET s, char **cpp, unsigned int maxsize)
{
  unsigned int old, size, realsize, nrest;
  register char *sp = *cpp;
  static char rest_tab[3] = {0, 0, 0};

  old = _gtx_disable_rw_trace;
  _gtx_disable_rw_trace = 1;
  if (!_gtx_read_uint(s, &size)) { _gtx_disable_rw_trace = old; return FALSE;}
  _gtx_disable_rw_trace = old;

  realsize = size + 1;
  if (realsize > maxsize) return FALSE;
  if (realsize == 0) return TRUE;
  if (sp == NULL) *cpp = sp = (char*)malloc(realsize);
  if (sp == NULL)
  {
    _gtx_error("_gtx_read_string(): Cannot allocate memory");
    return FALSE;
  }
  sp[size] = 0;

  nrest = size % 4;
  if (nrest > 0) nrest = 4 - nrest;
  if (!_gtx_read_bytes(s, sp, size)) return FALSE;
  if (!_gtx_read_bytes(s, rest_tab, nrest)) return FALSE;
  if (!_gtx_disable_rw_trace)
  { GTX_TRACE(3,("Read Str(%s)", sp)); }
  return TRUE;
}

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/*****                                                                  *****/
/*****                       Writing Functions                          *****/
/*****                                                                  *****/
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/

/****************************************************************************
**
** FUNCTION:    _gtx_empty_write_buffer
**
** DESCRIPTION: empty the internal write buffer
**
*****************************************************************************/
void _gtx_empty_write_buffer(void)
{
  GTX_TRACE(5,("Emptying Write buffer"));
  _buffer_nbytes = 0;
#if defined(DEBUG)
  memset(_buffer, -1, _BUFFER_MAXSIZE);
#endif
}

/****************************************************************************
**
** FUNCTION:    _gtx_flush_write
**
** DESCRIPTION: flush the internal buffer
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
**
** REMARKS:     All writing operations must be followed by _gtx_flush_write()
**
*****************************************************************************/
bool_t _gtx_flush_write(_GTX_SOCKET s)
{
  bool_t code = TRUE;
  GTX_TRACE(5,("Flushing Write buffer (%d bytes)", _buffer_nbytes));
  if (_buffer_nbytes != 0 &&
      send(s, _buffer, _buffer_nbytes, 0) == _GTX_SOCKET_ERROR)
    code = FALSE;
  _gtx_empty_write_buffer();
  return code;
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_bytes
**
** DESCRIPTION: writes bytes to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     addr: bytes array
** IN_ARGS:     len:  number of bytes
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_bytes(_GTX_SOCKET s, const void *addr, gtx_ulong len)
{
  register gtx_ulong i;
  bool_t err;
  register const char *char_ptr = (char*)addr;

  if (len == 0) return TRUE;
  if (s == _GTX_SOCKET_ERROR) return FALSE;

  if (_buffer_nbytes+len > _BUFFER_MAXSIZE)
  {
    err = _gtx_flush_write(s);
    if (err == FALSE) return FALSE;
  }

  for(i=0; i<len; i++)
  {
    _buffer[_buffer_nbytes++] = *(char_ptr++);
    if (_buffer_nbytes == _BUFFER_MAXSIZE)
    {
      err = _gtx_flush_write(s);
      if (err == FALSE) return FALSE;
    }
  }

  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_int
**
** DESCRIPTION: writes an int to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     ip:   int value
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_int(_GTX_SOCKET s, const int *ip)
{
  int myint = (int)htonl(*ip);
  int i;
  bool_t err;
  char *char_ptr = (char*)&myint;

  if (!_gtx_disable_rw_trace)
  { GTX_TRACE(3,("Write Int(%d)", *ip)); }
  if (s == _GTX_SOCKET_ERROR) return FALSE;
  if (_buffer_nbytes+sizeof(int32_t) > _BUFFER_MAXSIZE)
  {
    err = _gtx_flush_write(s);
    if (err == FALSE) return FALSE;
  }

  for(i = 0; i < sizeof(int32_t); i++)
    _buffer[_buffer_nbytes++] = *(char_ptr++);

  if (_buffer_nbytes == _BUFFER_MAXSIZE)
  {
    err = _gtx_flush_write(s);
    if (err == FALSE) return FALSE;
  }

  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_long
**
** DESCRIPTION: writes a gtx_long (UNIX long 32/64) to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s:  socket
** IN_ARGS:     lp: gtx_long value
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_long(_GTX_SOCKET s, const gtx_long *lp)
{
  int myint;
  int i,nbytes;
  bool_t err;
  char *char_ptr;

#ifndef GTX_EXE_64
  nbytes = 4;
  if (!_gtx_disable_rw_trace)
  { GTX_TRACE(3,("Write Long(32bits)(%"GTXLONG_FORMAT")", *lp)); }
  myint = (int)htonl(*lp);
  char_ptr = (char*)&myint;
#else
  {
    gtx_long mylong;

    if (_gtx_client64 && _gtx_server64)
    {
      nbytes = 8;
      if (!_gtx_disable_rw_trace)
      { GTX_TRACE(3,("Write Long(64bits)(%"GTXLONG_FORMAT")", *lp)); }
# ifdef GTX_LITTLE_ENDIAN
      {
        int32_t *ips = (int32_t*)lp;
        int32_t *ipd = (int32_t*)&mylong;
        *ipd = ntohl(*(ips+1));
        *(ipd+1) = ntohl(*ips);
      }
# else
      mylong = *lp;
# endif
      char_ptr = (char*)&mylong;
    }
    else
    {
      nbytes = 4;
      if (!_gtx_disable_rw_trace)
      { GTX_TRACE(3,("Write Long(64->32bits)(%"GTXLONG_FORMAT")", *lp)); }
      if (*lp > INT_MAX || *lp < INT_MIN)
      {
        _gtx_error("_gtx_write_long(): Server & Client are not both 64 bits");
        return(FALSE);
      }
      myint = (int)htonl((unsigned int)*lp);
      char_ptr = (char*)&myint;
    }
  }
#endif
  if (s == _GTX_SOCKET_ERROR) return FALSE;
  /* First flush buffer if needed */
  if (_buffer_nbytes+nbytes > _BUFFER_MAXSIZE)
  {
    err = _gtx_flush_write(s);
    if (err == FALSE) return FALSE;
  }

  for(i = 0; i < nbytes; i++)
    _buffer[_buffer_nbytes++] = *(char_ptr++);

  if (_buffer_nbytes == _BUFFER_MAXSIZE)
  {
    err = _gtx_flush_write(s);
    if (err == FALSE) return FALSE;
  }

  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_uint
**
** DESCRIPTION: writes an unsigned int to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     uip: unsigned int value
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_uint(_GTX_SOCKET s, const unsigned int *uip)
{
  return _gtx_write_int(s, (int*)uip);
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_short
**
** DESCRIPTION: writes a short to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     sp: returned short
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_short(_GTX_SOCKET s, const short *sp)
{
  int i = (int)*sp;
  return _gtx_write_int(s, &i);
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_ushort
**
** DESCRIPTION: writes an unigned short to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     usp: unsigned short
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_ushort(_GTX_SOCKET s, const unsigned short *usp)
{
  unsigned int ui = (unsigned int)*usp;
  return _gtx_write_int(s, (int*)&ui);
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_char
**
** DESCRIPTION: writes a char to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     cp: char
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_char(_GTX_SOCKET s, const char *cp)
{
  int i = (int)*cp;
  return _gtx_write_int(s, &i);
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_uchar
**
** DESCRIPTION: writes an unsigned char to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     ucp: unsigned char
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_uchar(_GTX_SOCKET s, const unsigned char *ucp)
{
  unsigned int ui = (unsigned int)*ucp;
  return _gtx_write_int(s, (int*)&ui);
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_double
**
** DESCRIPTION: writes a double to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     dp: double
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_double(_GTX_SOCKET s, const double *dp)
{
  int32_t *i32p;
  int i,old;
  bool_t code = FALSE;

  i32p = (int32_t*)dp;

  old = _gtx_disable_rw_trace;
  if (!_gtx_disable_rw_trace)
  { GTX_TRACE(3,("Write Dbl(%g)", *dp)); }
  _gtx_disable_rw_trace = 1;
#if defined(GTX_BIG_ENDIAN)
  i = (int)*i32p++;
  if (!_gtx_write_int(s, &i)) goto label_end;
  i = (int)*i32p;
  if (!_gtx_write_int(s, &i)) goto label_end;
#elif defined(GTX_LITTLE_ENDIAN)
  i = (int)*(i32p+1);
  if (!_gtx_write_int(s, &i)) goto label_end;
  i = (int)*i32p;
  if (!_gtx_write_int(s, &i)) goto label_end;
#else
#error "No endian defined"
#endif

  code = TRUE;
label_end:
  _gtx_disable_rw_trace = old;
  return code;
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_bool
**
** DESCRIPTION: writes a boolean to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     bp: boolean
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_bool(_GTX_SOCKET s, const bool_t *bp)
{
  int ib = (*bp == FALSE) ? 0 : 1;
  return _gtx_write_int(s, &ib);
}

/****************************************************************************
**
** FUNCTION:    _gtx_write_string
**
** DESCRIPTION: writes a string to a fd
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s: socket
** IN_ARGS:     cpp:    string
** IN_ARGS:     maxsize: maximum size including NULL terminator
**
** REMARKS:     The function only writes in the internal buffer. You must
** REMARKS:     call _gtx_flush_write to flush it at the end of writing ops.
**
*****************************************************************************/
bool_t _gtx_write_string(_GTX_SOCKET s, const char * const *cpp, unsigned int maxsize)
{
  unsigned int old, size, realsize, nrest;
  register const char *sp = *cpp;
  static char rest_tab[3] = {0, 0, 0};

  if (!_gtx_disable_rw_trace)
  { GTX_TRACE(3,("Write Str(%s)", sp)); }
  size = (unsigned int)strlen(sp);
  old = _gtx_disable_rw_trace;
  _gtx_disable_rw_trace = 1;
  if (!_gtx_write_uint(s, &size)) {_gtx_disable_rw_trace = old; return FALSE;}
  _gtx_disable_rw_trace = old;

  realsize = size + 1;
  if (realsize > maxsize) return FALSE;

  nrest = size % 4;
  if (nrest > 0) nrest = 4 - nrest;
  if (!_gtx_write_bytes(s, sp, size)) return FALSE;
  if (!_gtx_write_bytes(s, rest_tab, nrest)) return FALSE;
  return TRUE;
}

/****************************************************************************
**
** FUNCTION:    _gtx_rw_vector
**
** DESCRIPTION: reads/writes a vector
**
** RETURNS:     TRUE or FALSE
**
** IN_ARGS:     s       : socket
** IN_ARGS:     basep   : base pointer
** IN_ARGS:     nelem   : number of elements in the vector
** IN_ARGS:     elemsize: size of one element
** IN_ARGS:     rw_proc : read or write procedure
**
** REMARKS:     When writing, the function only writes in the internal buffer.
** REMARKS:     You must call _gtx_flush_write to flush it at the end of
** REMARKS:     writing ops.
**
*****************************************************************************/
bool_t _gtx_rw_vector(_GTX_SOCKET s,
                      char *basep,
                      gtx_ulong nelem,
                      unsigned int elemsize,
                      _gtx_proc_rw_t rw_proc)
{
  register gtx_ulong i;
  register char *elptr;
  bool_t code = FALSE;
  int old;

  i = 0;
  old = _gtx_disable_rw_trace;
  elptr = basep;
  if (nelem > 10)
  {
    for (i = 0; i < 10; i++) {
      if (! (*rw_proc)(s, elptr, UINT_MAX)) {
        goto label_end;
      }
      elptr += elemsize;
    }
    GTX_TRACE(3,("... + some other %"GTXULONG_FORMAT" values", nelem-10));
    _gtx_disable_rw_trace = 1;
  }
  for ( ; i < nelem; i++) {
    if (! (*rw_proc)(s, elptr, UINT_MAX)) {
      goto label_end;
    }
    elptr += elemsize;
  }
  code = TRUE;
label_end:
  _gtx_disable_rw_trace = old;
  return(code);
}
